/***************************************************************************
                          ebqtsocket.cpp  -  description
                             -------------------
    begin                : Thu Jul 25 2002
    copyright            : (C) 2002-3 by Chris Boyle
    email                : cmb@everybuddy.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   In addition, when you distribute EbQt binaries under section 3 of     *
 *   the GPL, I waive the requirement for the source code of Qt to be      *
 *   available. Source for all other parts is still required.              *
 *                                                                         *
 ***************************************************************************/

#include "ebqtsocket.h"

#include <qapplication.h>
#include <qcstring.h>		// QByteArray
#include <qprocess.h>
#include <qtimer.h>		// delayed kill of core (singleShot())

#include <sys/types.h>		// socketpair
#include <sys/socket.h>

EbQtSocket::EbQtSocket(QObject * parent, const char * name) :
	QObject(parent, name),
	sock(new QSocket(this)),
	launchedCore(NULL),
	sigSent(FALSE)
{
	// handle this one ourself (the point of this class)...
	connect(sock, SIGNAL(readyRead()), SLOT(getNewBytes()));

	// just pass the buck on most others
	// dunno why we didn't just inherit QSocket actually...
	// oh yeah, accessibility of dangerous parts of it...
	// or something... hmm.
	connect(sock, SIGNAL(hostFound()), SIGNAL(hostFound()));
	connect(sock, SIGNAL(connected()), SLOT(gotConnected()));
	connect(sock, SIGNAL(connectionClosed()), SIGNAL(connectionClosed()));
	connect(sock, SIGNAL(delayedCloseFinished()), SIGNAL(delayedCloseFinished()));
	connect(sock, SIGNAL(error(int)), SIGNAL(error(int)));
}

EbQtSocket::~EbQtSocket()
{
	sock->flush();  // no delayed close, please
	sock->close();
}

void EbQtSocket::connectToHost(const QString & host, const QByteArray & cookie,
	Q_UINT16 port)
{
	if (launchedCore != NULL)
		launchedCore->deleteLater();
	launchedCore = NULL;
	
	autoResp = cookie;
	sock->connectToHost(host, port);
	// it seems updateSockStatus() gets called here by itself...
	// I guess QSocket must be firing connectionClosed() or some such
}

bool EbQtSocket::launchAndConnect(const QString & cmd)
{
	int sockets[2];
	if (socketpair(PF_UNIX, SOCK_STREAM, 0, sockets) != 0)
		return FALSE;
	killLaunchedCore();
	launchedCore = new QProcess(QStringList::split(' ',
		cmd.arg(sockets[1]).arg(sockets[1])), this);
	connect(launchedCore, SIGNAL(destroyed()), launchedCore, SLOT(kill()));
	launchedCore->setCommunication(0);
	if (! launchedCore->start()) {
		launchedCore->deleteLater();
		launchedCore = NULL;
		return FALSE;
	}
	sock->blockSignals(TRUE);
	sock->setSocket(sockets[0]);
	sock->blockSignals(FALSE);
	return TRUE;
}

void EbQtSocket::killLaunchedCore()
{
	if (launchedCore == NULL)
		return;
	launchedCore->tryTerminate();
	QTimer::singleShot(3000, launchedCore, SLOT(deleteLater()));
	launchedCore = NULL;
	close();
}

void EbQtSocket::getNewBytes()
{
	// these keep track of a partially-read command
	static Q_UINT8 elementsToGo = 0;
	static Q_UINT16 nextSz = 0;
	static EbQtCommand command;

	int avail;
	
	while ((avail = sock->bytesAvailable()) > 0) {
		if (elementsToGo > 0) {  // zero-element commands _are_ handled below
			if (nextSz > 0) {  // and now so are zero-byte elements (26/12/02)... D'OH!
				// expecting nextsz bytes
				if (avail < nextSz)
					break; // element is incomplete

				// got an element
				char * p = new char[nextSz+1];
				sock->readBlock(p, nextSz);
				p[nextSz]='\0';
				command.append(QString::fromUtf8(p));
				nextSz = 0;

				if (--elementsToGo <= 0) {
					// got a complete command
					gotOne(command);
					command.clear();
				}
			} else {
				// expecting start of element
				if (avail < 2)
					break; // element size is incomplete
				nextSz = (sock->getch() << 8) + sock->getch();
				if (nextSz == 0) {  // zero-byte element
					command << QString::null;
					if (--elementsToGo <= 0) {
						// got a complete command
						gotOne(command);
						command.clear();
					}
					// leave nextSz at 0, expecting something other than elem body
				}
				// now expecting an element of this many bytes
			}
		} else {
			// expecting start of a command
			// we know avail >= 1
			elementsToGo = sock->getch();
			// now expecting this many elements
			if (elementsToGo == 0) {
				// zero-element (null) command
				command.clear(); // just making sure
				gotOne(command);
				// leave elementsToGo at 0, expect start of command again
			}
		}
	}
	// there are now no complete parts available
}

void EbQtSocket::gotConnected()
{
	// the autoResp is not sent in the usual way, but just dumped
	sock->writeBlock(autoResp.data(), autoResp.size());
	emit(connected());
}

void EbQtSocket::gotOne(const EbQtCommand & command)
{
	queue.append(command);
	if (sigSent)
		return;
	// next 2 lines MUST NOT be reversed!
	// (if events are processed immediately, sigSent will be reset inside
	// emit, which is what we want in that case)
	sigSent = TRUE;
	emit(gotNewCommand());
}

EbQtSocket &EbQtSocket::operator>>(EbQtCommand &command)
{
	sigSent = FALSE;
	command = queue.first();
	if(! queue.isEmpty())
		queue.remove(queue.first());
	return *this;
}

EbQtSocket &EbQtSocket::operator<<(const EbQtCommand &command)
{
	EbQtCommand::ConstIterator it;
	uint size = (2*command.count() + 1), pos = 1;
	
#ifdef EBQT_TRANSCRIPT
	qDebug(">>ebqt>>: %s", command.join(" ").ascii());
#endif
	
	for (it = command.begin(); it != command.end(); it++)
		size += (*it).utf8().length();
	
	QByteArray buf(size);  // variable size w/o compiler complaints
	buf[0]=command.count();
	for (it = command.begin(); it != command.end(); it++) {
		int length = (*it).utf8().length();
		buf[pos++] = length/256;
		buf[pos++] = length%256;
		memcpy(buf.data() + pos, (*it).utf8(), length);
		pos += length;
	}
	
	sock->writeBlock(buf.data(), size);
	return *this;
}

void EbQtSocket::close()
{
	sock->flush();
	sock->close();
#if (QT_VERSION >= 0x030100)
	sock->clearPendingData();
#endif
	queue.clear();
}

