/***************************************************************************
                          ebqtprefpage.cpp  -  description
                             -------------------
    begin                : Mon Jan 6 2003
    copyright            : (C) 2002-2004 by Chris Boyle
    email                : cmb@everybuddy.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   In addition, when you distribute EbQt binaries under section 3 of     *
 *   the GPL, I waive the requirement for the source code of Qt to be      *
 *   available. Source for all other parts is still required.              *
 *                                                                         *
 ***************************************************************************/

#include <qlayout.h>
#include <qtabwidget.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qscrollview.h>
#include <qregexp.h>

#include "ebqtprefpage.h"
#include "ebqtservice.h"

EbQtPrefPage::EbQtPrefPage(QWidget *parent, const char *name,
	QDict<EbQtService> * allServices) :
	QWidget(parent, name),
	layout(new QVBoxLayout(this, 3)),
	componentView(new QScrollView(this)),
	componentBox(new QWidget(componentView->viewport())),
	componentLayout(new QVBoxLayout(componentBox, 3)),
	emptyLabel(new QLabel(tr("(no preferences available)"), componentBox)),
	row(1),  // after empty label (even when invisible...)
	tabs(new QTabWidget(this, "tabs")),
	allServices(allServices),
	applyBtn(0),
	revertBtn(0),
	prevSubPage(0)
{
	// master list of components on this page
	components.setAutoDelete(TRUE);

	// stretch the components when necessary (both dimensions)
	componentView->setResizePolicy(QScrollView::AutoOneFit);
	componentView->setFrameShape(QFrame::NoFrame);
	componentView->setFrameShadow(QFrame::Plain);
	// but don't reveal its edges (shouldn't happen anyway with the above, but make sure)
	componentView->viewport()->setPaletteBackgroundColor(colorGroup().background());
	componentView->addChild(componentBox);
	// the placeholder...
	componentLayout->addWidget(emptyLabel);
	// now add stretch in box's layout
	componentLayout->addStretch();
	layout->addWidget(componentView);

	connect(tabs, SIGNAL(currentChanged(QWidget *)),
		SLOT(setSubPage(QWidget *)));
	layout->addWidget(tabs);

	if (parent && parent->inherits("EbQtPrefPage")) {
		((EbQtPrefPage *)parent)->addSubPage(this);
	} else {
		QHBoxLayout * bottomRow = new QHBoxLayout(layout, 3);
		bottomRow->addStretch();

		applyBtn = new QPushButton("&Apply", this);
		connect(applyBtn, SIGNAL(clicked()),
			// apply recursively, then refresh (which is always recursive)
			SLOT(applyThenRefresh()));
		applyBtn->setDefault(TRUE);

		revertBtn = new QPushButton("&Revert", this);
		connect(revertBtn, SIGNAL(clicked()),
			SLOT(refresh()));

		bottomRow->addWidget(revertBtn);
		bottomRow->addWidget(applyBtn);
	}

	// we need this for some reason... :-/
	setEnabled(TRUE);

	// nothing's visible until it's non-empty
	updateEmpty();
}

EbQtPrefPage::~EbQtPrefPage()
{
}

void EbQtPrefPage::showEvent(QShowEvent *)
{
	layout->activate();
}

void EbQtPrefPage::setCaption(QString caption)
{
	if (caption == this->caption()) return;
	
	// do magic
	QRegExp parenSvc(" \\(([^\\(\\)]+)\\)$");
	EbQtService * s;
	if (allServices && (parenSvc.search(caption) > -1)
		&& (s = (*allServices)[parenSvc.cap(1)])) {
		svcPixmap = s->pixmap(16,16);  // FIXME: need tab height!
		caption.truncate(parenSvc.pos(0));
	} else {
		svcPixmap.resize(0,0);  // make null
	}
	
	// emit first so old value is available
	emit(captionChanged(caption, this));
	
	// actually set the caption
	QWidget::setCaption(caption);
}

void EbQtPrefPage::addComponent(const QString & type, const QString & name,
	const QString & title, const QString & value)
{
	QWidget * component;
	if (type == "label") {  // neither a separate label nor a normal layout insertion
		component = new QLabel(title, componentBox);
		((QLabel *)component)->setAlignment(
			(((QLabel *)component)->alignment()
			& ~Qt::SingleLine) | Qt::WordBreak);
		((QLabel *)component)->setTextFormat(Qt::PlainText);
		((QLabel *)component)->setSizePolicy(QSizePolicy(
			QSizePolicy::Preferred, QSizePolicy::Preferred,
			TRUE));
		componentLayout->insertWidget(row++, component);
//		componentLayout->addWidget(component);
	} else {  // normal layout insertion
		if (type == "button") {  // no label
			component = new QPushButton(title, componentBox, name);
			// this one we _do_ connect somewhere whatever we do
			connect((QPushButton *)component, SIGNAL(clicked()),
				// grabs name, apply()s, and emits more sensibly
				SLOT(pressedButton()));
		} else if (type == "toggle") {
			component = new QCheckBox(title, componentBox, name);
			((QCheckBox *)component)->setChecked(value.toUInt() > 0);
			// connect toggled somewhere, if we were going to do that
		} else {  // needs separate label with component as buddy
			if (type == "string") {
				component = new QLineEdit(componentBox, name);
				((QLineEdit *)component)->setText(value);
				// connect textChanged somewhere, if we were going to do that
			} else if (type == "password") {
				component = new QLineEdit(componentBox, name);
				((QLineEdit *)component)->setEchoMode(QLineEdit::Password);
				((QLineEdit *)component)->setText(value);
				// connect textChanged somewhere, if we were going to do that
			} else if (type == "option") {
				component = new QComboBox(componentBox, name);
				// store selection in it
				((QComboBox *)component)->insertItem(value);
				((QComboBox *)component)->setCurrentItem(0);
				// expect option_data
				((QComboBox *)component)->setEnabled(FALSE);
				// connect activated somewhere, if we were going to do that
			} else {  // already established it's not a label or button
				qWarning("prefs: martian prefs component type!");
				// don't allow changes, we don't know what it is
				component = new QLabel(this, name);
				((QLabel *)component)->setFrameStyle(
					  QFrame::LineEditPanel
					| QFrame::Sunken);
				((QLabel *)component)->setAlignment(
					(((QLabel *)component)->alignment()
					& ~Qt::SingleLine)
					| Qt::WordBreak);
				((QLabel *)component)->setText(value);
			}
			// so here we have something that needs a label
			QLabel * label = new QLabel(component, title, componentBox);
			label->setTextFormat(Qt::PlainText);
			componentLayout->insertWidget(row++, label);
//			componentLayout->addWidget(label);
			connect(component, SIGNAL(destroyed(QObject *)),
				label, SLOT(deleteLater()));
			label->show();
		}  // label
		componentLayout->insertWidget(row++, component);
//		componentLayout->addWidget(component);
	}  // layout
	componentView->setMinimumSize(componentBox->sizeHint().width()
		+ componentView->verticalScrollBar()->sizeHint().width(),
		componentView->minimumSize().height());
	// _all_ components get this
	components.insert(name, component);
	// this took _ages_ to find: you need this if you add a
	// component to a widget that's already visible
	component->show();
	// update whether buttons are shown etc etc
	updateEmpty();
	componentView->show();
}

void EbQtPrefPage::addOptionData(const QString & name, const QStringList & options)
{
	QWidget * component;
	if (! (component = components[name])) {
		qWarning("prefs: option_data for martian component!");
		return;
	}
	if (! component->inherits("QComboBox")) {
		qWarning("prefs: option_data for non-QComboBox component!");
		return;
	}
	
	QComboBox * combo = (QComboBox *)component;
	combo->blockSignals(TRUE);  // not connected atm but future-proofing
	QString prevSelection = combo->currentText();
	combo->clear();
	combo->insertStringList(options);
	combo->setEnabled(TRUE);
	// go to most recently added equal item, if possible, without editing items
	for (int i=combo->count()-1; i>=0; i--) {
		if (combo->text(i) == prevSelection) {
			combo->setCurrentItem(i);
			break;
		}
	}
	combo->blockSignals(FALSE);
}

void EbQtPrefPage::addSubPage(EbQtPrefPage * page)
{
	tabs->blockSignals(TRUE);  // don't want prevSubPage setting here
	tabs->addTab(page, page->pixmap(), page->caption());
	connect(page, SIGNAL(captionChanged(const QString &, EbQtPrefPage *)),
		SLOT(changeSubPageCaption(const QString &, EbQtPrefPage *)));
	if (page->name() == prevSubPage) tabs->showPage(page);
	tabs->show();
	tabs->blockSignals(FALSE);
	updateEmpty();
}

void EbQtPrefPage::changeSubPageCaption(const QString & caption, EbQtPrefPage * page)
{
	tabs->changeTab(page, page->pixmap(), caption);
}

void EbQtPrefPage::clearComponents()
{
	components.clear();  // autoDelete is on (see constructor)
	// component destroyed -> label deleteLater (addComponent())

	row = 1;
	updateEmpty();
}

void EbQtPrefPage::clearSubPages()
{
	QWidget * w;
	tabs->blockSignals(TRUE);  // "selections" will happen in here, ignore them
	while ((w = tabs->page(0))) {
		tabs->removePage(w);
		delete w;
	}
	tabs->blockSignals(FALSE);
	
	updateEmpty();
}

void EbQtPrefPage::apply()
{
	// components
	QDictIterator<QWidget> ci(components);
	for (; ci.current(); ++ci) {
		if ((*ci)->inherits("QLineEdit")) {
			// string or password
			emit(wantsValueSet((*ci)->name(),
				((QLineEdit *)(*ci))->text(),
				this));
		} else if ((*ci)->inherits("QCheckBox")) {
			// toggle
			emit(wantsValueSet((*ci)->name(),
				((QCheckBox *)(*ci))->isChecked() ? "1" : "0",
				this));
		} else if ((*ci)->inherits("QComboBox")) {
			// option
			emit(wantsValueSet((*ci)->name(),
				((QComboBox *)(*ci))->currentText(),
				this));
		} else if (! (*ci)->inherits("QLabel")
			&& ! (*ci)->inherits("QButton")) {
			qWarning("not touching martian pref \"%s\" with a 10' pole!",
				(*ci)->name());
		}
	}
	
	// subpages
	for (int i=0; i<tabs->count(); i++)
		((EbQtPrefPage *)(tabs->page(i)))->apply();
}

void EbQtPrefPage::pressedButton()
{
	EbQtPrefPage * tlp = this;
	while (tlp->parent()
		&& tlp->parent()->inherits("EbQtPrefPage"))
		tlp = (EbQtPrefPage *)(tlp->parent());
	tlp->apply();
	emit(wantsValueSet((QObject::sender())->name(), "1", this));
	emit(wantsSave());
	tlp->refresh();
}

void EbQtPrefPage::selectAnySubpage(const QString & name)
{
	for (int i=0; i<tabs->count(); i++) {
		if (tabs->page(i)->name() == name) {
			tabs->setCurrentPage(i);
			return;
		}
	}
}

void EbQtPrefPage::updateEmpty()
{
	bool componentsEmpty = components.isEmpty();
	bool tabsEmpty = ! tabs->page(0);
	if (tabsEmpty)
		tabs->hide();
	else
		tabs->show();
	if (componentsEmpty && tabsEmpty) {
		emptyLabel->show();
		componentView->show();
		if (revertBtn)	revertBtn->hide();
		if (applyBtn)	applyBtn ->hide();
	} else {
		emptyLabel->hide();
		if (componentsEmpty) { qDebug("empty"); componentView->hide(); }
		if (revertBtn)	revertBtn->show();
		if (applyBtn)	applyBtn ->show();
	}
}

