/***************************************************************************
                               ebqtmainwin.cpp
                             -------------------
    begin                : Wed Jul 24 10:35:42 BST 2002
    copyright            : (C) 2002-3 by Chris Boyle
    email                : cmb@everybuddy.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   In addition, when you distribute EbQt binaries under section 3 of     *
 *   the GPL, I waive the requirement for the source code of Qt to be      *
 *   available. Source for all other parts is still required.              *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qmenubar.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>	// add button
#include <qpopupmenu.h>		// add menu, rmb menus
#include <qstatusbar.h>
#include <qwhatsthis.h>
#include <qcombobox.h>		// showWho, showHow
#include <qaction.h>
#include <qtabwidget.h>		// mainwin and tabbed convs
#include <qmessagebox.h>	// warnings
#include <qspinbox.h>		// cookie
#include <qtable.h>		// localAcctsView
#include <qlayout.h>
#include <qsocket.h>
#include <qfile.h>		// cookie file
#include <qdir.h>		//
#include <qlineedit.h>		// join dialog
#include <qpushbutton.h>	// dialogs
#include <qlistbox.h>		// list dialog
#include <qtextedit.h>		// text dialog
#include <qtooltip.h>		// tooltips
#include <qregexp.h>		// parsing saved geometries
#include <qtimer.h>		// efficient geometry saving
#include <qapplication.h>	// ::desktop() for screen geometry
#include <qgroupbox.h>		// need to {en,dis}able these
#include <qsplitter.h>		// topSplitter

#include "ebqtmainwin.h"

#include "ebqtsocket.h"
#include "ebqtservice.h"
#include "ebqtlocalaccount.h"
#include "ebqtaccount.h"
#include "ebqtcontact.h"
#include "ebqtgroup.h"

#include "ebqtdlgerror.h"
#include "ebqtdlgyesno.h"
#include "ebqtdlglist.h"
#include "ebqtdlgtext.h"
#include "ebqtawaydlg.h"
#include "ebqtignoredlg.h"
#include "ebqtjoindlg.h"

#include "ebqtadd.h"
#include "ebqtbuddyview.h"
#include "ebqtconv.h"
#include "ebqtprefpage.h"

EbQtMainWin::EbQtMainWin(QWidget * parent, const char * name) :
	EbQtMainWinUI(parent, name, 0),  // don't force top-level
	sock(new EbQtSocket(this)),
	buddyView(new EbQtBuddyView(buddyListFrame, "buddyView")),
#ifdef Q_WS_X11
	trayIcon(NULL),
#endif
	toggleInvisibility(new QAction(tr("Toggle hiding all windows"),
		tr("H&ide all windows"), QKeySequence(), this,
		"toggleInvisibility", TRUE)),
	addDlg(NULL),
	lastAway(FALSE),
	convsTabWidget(NULL),
	geoChangeTimer(new QTimer(this, "geoChangeTimer")),
	topSplitter(0)
{
	if (parent && parent->inherits("QSplitter")) {
		topSplitter = (QSplitter *)parent;
		topSplitter->setResizeMode(this, QSplitter::KeepSize);
		topSplitter->setCaption(caption());  // won't need changing
		const QPixmap * i = icon();
		if (i) topSplitter->setIcon(*i);
		topSplitter->resize(topSplitter->minimumSizeHint());
	}
	
	// these are the master lists
	allGroups.setAutoDelete(TRUE);
	allServices.setAutoDelete(TRUE);
	accountItems.setAutoDelete(TRUE);
	contactItems.setAutoDelete(TRUE);
	groupItems.setAutoDelete(TRUE);
	// NOT localAcctItems, we need to delete the whole row involved
	
	// buddies tab
	listLayout = new QVBoxLayout(buddyListFrame, 3, 0);
	listLayout->addWidget(buddyView);
	listLayout->activate();
	connect(buddyView, SIGNAL(selectionChanged()),
		SLOT(setSelectionActions()));
	connect(buddyView, SIGNAL(expanded(QListViewItem *)),
		SLOT(customTreeExpansion()));
	connect(buddyView, SIGNAL(collapsed(QListViewItem *)),
		SLOT(customTreeExpansion()));
	connect(buddyView,
		SIGNAL(contextMenuRequested(QListViewItem *, const QPoint &, int)),
		SLOT(buddyRMB(QListViewItem *, const QPoint &)));
	connect(buddyView, SIGNAL(doubleClicked(QListViewItem *)),
		SLOT(tryOpenConv(QListViewItem *)));
	connect(buddyView,
		SIGNAL(itemRenamed(QListViewItem *, int, const QString &)),
		SLOT(tryRename(QListViewItem *, int, const QString &)));
	buddyView->setColumnWidth(0, (buddyView->width()*4)/5);
	buddyView->viewport()->installEventFilter(this);
	changeWhoQuietly(showWho->currentItem());
	connect(this, SIGNAL(buddyListChanged()), SLOT(updateAll()));
	
	// put last item (help) onto right side of bar
	// (ignored by some styles)
	menuBar->insertSeparator(menuBar->count() - 1);

	statusBar()->setSizeGripEnabled(FALSE);
	statusBar()->clear();
	
	// local accts pref page
	EbQtPrefPage * p = new EbQtPrefPage(acctPrefsFrame, "accounts",
		&allServices);
	connect(p, SIGNAL(subPageSelected(EbQtPrefPage *, EbQtPrefPage *)),
		SLOT(selectLocalAcct(EbQtPrefPage *)));
	connect(p, SIGNAL(wantsRefresh(EbQtPrefPage *)),
		SLOT(refreshPrefPage(EbQtPrefPage *)));
	connect(p,
		SIGNAL(wantsValueSet(const QString &, const QString &,
			EbQtPrefPage *)),
		SLOT(trySetPref(const QString &, const QString &,
			EbQtPrefPage *)));
	connect(p, SIGNAL(wantsSave()), SLOT(trySaveConfig()));
	// don't worry about caption
	// don't worry about this disappearing, it won't until the whole app does
	prefPages.insert("accounts", p);

	// pipes command
	resetLaunchCmd();

	// cookie boxes
	(new QGridLayout(cookieFrame, (EBQT_COOKIE_LENGTH / 4), 4,
		/* margin */ 1))->setAutoAdd(true);
	for (int i=0; i < EBQT_COOKIE_LENGTH; i++) {
		cookieBoxes.append(new QSpinBox(0, 255, 1, cookieFrame,
			QString("cookieChar") + QString::number(i)));
	}
	
	connect(sock, SIGNAL(hostFound()), SLOT(hostFound()));
	connect(sock, SIGNAL(connected()), SLOT(hostConnected()));
	connect(sock, SIGNAL(connectionClosed()), SLOT(connectionClosed()));
	connect(sock, SIGNAL(error(int)), SLOT(processSockError(int)));
	connect(sock, SIGNAL(gotNewCommand()), SLOT(processCommands()));
	
	connect(toggleInvisibility, SIGNAL(toggled(bool)),
		SLOT(setInvisibility(bool)));
#ifdef Q_WS_X11
	trayIcon = new EbQtTrayIcon(this, "trayIcon");
	trayIcon->setPixmap(*(icon()));
	connect(trayIcon, SIGNAL(clicked(const QPoint &)),
		toggleInvisibility, SLOT(toggle()));
	connect(trayIcon, SIGNAL(rightClicked(const QPoint &)),
		SLOT(doTrayMenu(const QPoint &)));
	trayIcon->requestDock();
#endif
	updateSummary();
	
	// default to a local cookie
	loadCookie(TRUE);
	
	setConnected();
	
	connect(geoChangeTimer, SIGNAL(timeout()),
		SLOT(putGeometries()));
}

EbQtMainWin::~EbQtMainWin()
{
}

bool EbQtMainWin::loadCookie(bool quietly)
{
	QFile cookieFile(QDir::homeDirPath() + "/.everybuddy/authcookie");
	if (cookieFile.open(IO_ReadOnly)) {
		unsigned int port = (cookieFile.getch()<<8) + cookieFile.getch();
		addrCombo->setCurrentText(QString("localhost")
			+ (port != EBQT_DEFAULT_PORT
				? QString(":%1").arg(port)
				: QString::null));
		for (int i=0; i < EBQT_COOKIE_LENGTH; i++) {
			int c = cookieFile.getch();
			if (c > -1) cookieBoxes.at(i)->setValue(c);
		}
		return true;
	} else {
		if (! quietly)
			QMessageBox::warning(this, tr("No cookie file"),
				tr("There is no cookie file, maybe eb-lite\n"
				"isn't running on this machine."));
		return false;
	}
}

void EbQtMainWin::tryConnect()
{
	sock->close();
	setConnected();
	
	uint port = EBQT_DEFAULT_PORT;
	QStringList l = QStringList::split(':', addrCombo->currentText());
	if (l.isEmpty() || l.count() > 2 || l[0].isEmpty()
		|| (l.count() == 2 && (port = l[1].toUInt()) <= 0)) {
		QMessageBox::warning(this, tr("Error"),
			tr("You must enter a valid address (hostname or IP,\n"
			"optionally followed by a colon and port number)"),
			QMessageBox::Cancel, QMessageBox::NoButton);
		return;
	}
	autoLaunchGroup->setEnabled(FALSE);
	connectBtn->setEnabled(FALSE);
	disconnectBtn->setEnabled(TRUE);
	addrCombo->setEnabled(FALSE);
	addrLabel->setEnabled(FALSE);
	cookieFrame->setEnabled(FALSE);
	cookieBtn->setEnabled(FALSE);
	cookieLabel->setEnabled(FALSE);
	
	QByteArray a(EBQT_COOKIE_LENGTH);
	for (int i=0; i < EBQT_COOKIE_LENGTH; i++)
		a[i] = cookieBoxes.at(i)->value();
	sock->connectToHost(l[0], a, port);
	hostLookup();
}

void EbQtMainWin::launchAndConnect()
{
	manualLaunchGroup->setEnabled(FALSE);
	launchBtn->setEnabled(FALSE);
	launchCmdEdit->setEnabled(FALSE);
	launchCmdReset->setEnabled(FALSE);
	const QString & cmd = launchCmdEdit->text();
	if (! (cmd.contains("%1") && cmd.contains("%2"))) {
		killLaunchedCore();  // re-enable stuff
		QMessageBox::warning(this, tr("Invalid command"),
			tr("The command must pass the input and output pipes\n"
			"(%1 and %2) to the core. To reset to the default\n"
			"command, use the button to the right of the command."),
			QMessageBox::Cancel, QMessageBox::NoButton);
		return;
	}
	if (! sock->launchAndConnect(cmd)) {
		killLaunchedCore();  // re-enable stuff
		QMessageBox::warning(this, tr("Failed to start"),
			tr("The command failed to start. Please check you have\n"
			"installed eb-lite properly, and that the command is\n"
			"correct. To reset to the default command, use the\n"
			"button to the right of the command."),
			QMessageBox::Cancel, QMessageBox::NoButton);
		return;
	}
	if (sock->state() != QSocket::Connected) {
		killLaunchedCore();  // re-enable stuff
		QMessageBox::warning(this, tr("Failed to connect"),
			tr("The command started, but failed to connect. This is\n"
			"very strange, perhaps eb-lite has changed the way\n"
			"auto-launch works. If the default command (use the\n"
			"reset button to the right of the command) doesn't\n"
			"work, and you have eb-lite properly installed, then\n"
			"please report this as a bug."),
			QMessageBox::Cancel, QMessageBox::NoButton);
		return;
	}
	// kludge alert...
	processCommand(EbQtCommand("cookie_accepted"));
	killBtn->setEnabled(TRUE);
}

void EbQtMainWin::resetLaunchCmd()
{
	launchCmdEdit->setText(EBQT_DEFAULT_LAUNCH_CMD);
}

void EbQtMainWin::killLaunchedCore()
{
	disconnectCore();
	sock->killLaunchedCore();
}

void EbQtMainWin::hostLookup()
{
	setConnected();
	statusBar()->message(tr("Looking up host..."));
}

void EbQtMainWin::hostFound()
{
	setConnected();
	statusBar()->message(tr("Connecting to host..."));
}

void EbQtMainWin::hostConnected()
{
	statusBar()->message(tr("Sent cookie..."));
}

void EbQtMainWin::connectionClosed(bool showMsg)
{
	sock->blockSignals(TRUE);
	sock->close(); // make sure
	sock->blockSignals(FALSE);
	// mop up all server-specific data
	// order is important for buddyView: account items are deepest
	allServices.clear();	// takes care of local accts -> accts too
	allGroups.clear();	// takes care of contacts
	for (QDictIterator<QDialog> di(dialogs); di.current();) {
		delete dialogs.take(di.currentKey());  // ++di automatically
	}
	autoLaunchGroup->setEnabled(TRUE);
	killBtn->setEnabled(FALSE);
	launchCmdEdit->setEnabled(TRUE);
	launchCmdReset->setEnabled(TRUE);
	launchBtn->setEnabled(TRUE);
	manualLaunchGroup->setEnabled(TRUE);
	disconnectBtn->setEnabled(FALSE);
	addrCombo->setEnabled(TRUE);
	addrLabel->setEnabled(TRUE);
	cookieBtn->setEnabled(TRUE);
	cookieFrame->setEnabled(TRUE);
	cookieLabel->setEnabled(TRUE);
	connectBtn->setEnabled(TRUE);
	setConnected();
	statusBar()->message(tr("Disconnected"), EBQT_STATUSBAR_DELAY);
	if (showMsg)
		QMessageBox::warning(this, tr("Disconnected"),
			tr("The eb-lite server disconnected unexpectedly.\n"
			"It may have crashed."));
}

void EbQtMainWin::processSockError(int error)
{
	QString errMsg;
	connectionClosed(FALSE);  // about to show a message here, don't show another
	switch(error) {
		case QSocket::ErrConnectionRefused:
		errMsg = tr("Connection refused, maybe eb-lite\n"
			"isn't running on that host?");
		break;
		case QSocket::ErrHostNotFound:
		errMsg = tr("Unknown host.\n"
			"Check the address, and check\n"
			"that you can reach that host.");
		break;
		case QSocket::ErrSocketRead:
		errMsg = tr("There was an error reading from the server.");
		break;
		default:
		errMsg = tr("There was an unknown network error.");
		break;
	}
	QMessageBox::warning(this, tr("Network error"), errMsg);
	addrCombo->setFocus();
}

void EbQtMainWin::setConnected()
{
	connected = (sock->state() == QSocket::Connected);
	updateSignedOnOff();
	tabs->setTabEnabled(listTab, connected);
	tabs->setTabEnabled(localAcctsTab, connected);
	tabs->showPage(
		connected
			? signOff->isOn()
				? localAcctsTab
				: listTab
			: connectTab);
	
	if (! connected) {
		for (QPtrDictIterator<EbQtConv> sci(singleConvs); sci.current(); ++sci)
			removeConversation(sci.current());  // and delete
		for (QDictIterator<EbQtConv> gci(groupConvs); gci.current(); ++gci)
			removeGroupConv(gci.current());  // and delete
		away->blockSignals(TRUE);
		away->setOn(FALSE);
		away->blockSignals(FALSE);
	}
	away->setEnabled(connected);
	
	add->setEnabled(connected);
	remove->setEnabled(connected);
	setSelectionActions();
	joinGroup->setEnabled(connected);
	showPrefs->setEnabled(connected);
	saveConfig->setEnabled(connected);
	optionsMenu->setEnabled(connected);
	
	if (sock->state() == QSocket::Idle)
		launchCmdEdit->setFocus();
}

void EbQtMainWin::updateSignedOnOff()
{
	bool seenOn  = FALSE;
	bool seenOff = FALSE;
	
	QDictIterator<EbQtService> si(allServices);
	for(; si.current(); ++si) {
		QDictIterator<EbQtLocalAccount> li = (*si)->localAccounts();
		for (; li.current(); ++li) {
			if ((*li)->ready())
				seenOn = TRUE;
			else
				seenOff = TRUE;
		}
	}
	
	signOn ->blockSignals(TRUE);
	signOff->blockSignals(TRUE);
	
	if (! seenOff) {
		if (! seenOn)
			// no accts
			tabs->setCurrentPage(1);
		else if (! signOn->isOn())
			// final acct just signed on
			tabs->setCurrentPage(0);
	}
	signOn ->setOn(seenOn && ! seenOff);
	signOff->setOn(! seenOn);
	signOn ->setEnabled(seenOff);
	signOff->setEnabled(seenOn);
	
	signOn ->blockSignals(FALSE);
	signOff->blockSignals(FALSE);
}

void EbQtMainWin::tabChanged(QWidget * tab)
{
	if (tab == listTab) {
		add   ->setEnabled(TRUE);
		setSelectionActions();
		buddyView->setFocus();
	} else {
		openConv->setEnabled(FALSE);
		add     ->setEnabled(tab == localAcctsTab);
		remove  ->setEnabled(tab == localAcctsTab);
		ignore  ->blockSignals(TRUE);
		ignore  ->setOn(FALSE);
		ignore  ->blockSignals(FALSE);
		ignore  ->setEnabled(FALSE);
		rename  ->setEnabled(FALSE);
		makePreferred->setEnabled(FALSE);
#if (QT_VERSION >= 0x030100)
		//ignore->setVisible(FALSE);
		rename->setVisible(FALSE);
		makePreferred->setVisible(FALSE);
#endif
	}
}

void EbQtMainWin::setSelectionActions()
{
	QListViewItem * selected = buddyView->selectedItem();
	QObject * tmp;
	
	if (! selected
		// FIXME: assumption (cast)...
		|| ! (tmp = ((EbQtBuddyViewItem *)selected)->owner())) {
		openConv->setEnabled(FALSE);
		remove->setEnabled(FALSE);
		ignore->blockSignals(TRUE);
		ignore->setOn(FALSE);
		ignore->blockSignals(FALSE);
		ignore->setEnabled(FALSE);
		rename->setEnabled(FALSE);
		makePreferred->setEnabled(FALSE);
#if (QT_VERSION >= 0x030100)
		//ignore->setVisible(FALSE);
		rename->setVisible(FALSE);
		makePreferred->setVisible(FALSE);
#endif
		return;
	}
	
	if (tmp->inherits("EbQtAccount")) {
		openConv->setEnabled(TRUE);
		remove->setEnabled(TRUE);
		ignore->blockSignals(TRUE);
		ignore->setOn(FALSE);
		ignore->blockSignals(FALSE);
		ignore->setEnabled(FALSE);
		rename->setEnabled(FALSE);
		makePreferred->setEnabled(TRUE);
#if (QT_VERSION >= 0x030100)
		//ignore->setVisible(TRUE);
		rename->setVisible(FALSE);
		makePreferred->setVisible(TRUE);
#endif
	} else if (tmp->inherits("EbQtContact")) {
		openConv->setEnabled(TRUE);
		remove->setEnabled(TRUE);
		ignore->blockSignals(TRUE);
		ignore->setOn(((EbQtContact *)tmp)->ignored());
		ignore->blockSignals(FALSE);
		ignore->setEnabled(TRUE);
		rename->setEnabled(TRUE);
		makePreferred->setEnabled(FALSE);
#if (QT_VERSION >= 0x030100)
		//ignore->setVisible(TRUE);
		rename->setVisible(TRUE);
		makePreferred->setVisible(FALSE);
#endif
	} else if (tmp->inherits("EbQtGroup")) {
		openConv->setEnabled(FALSE);
		remove->setEnabled(TRUE);
		ignore->blockSignals(TRUE);
		ignore->setOn(FALSE);
		ignore->blockSignals(FALSE);
		ignore->setEnabled(FALSE);
		rename->setEnabled(TRUE);
		makePreferred->setEnabled(FALSE);
#if (QT_VERSION >= 0x030100)
		//ignore->setVisible(FALSE);
		rename->setVisible(TRUE);
		makePreferred->setVisible(FALSE);
#endif
	} else {
		openConv->setEnabled(FALSE);
		remove->setEnabled(FALSE);
		ignore->blockSignals(TRUE);
		ignore->setOn(FALSE);
		ignore->blockSignals(FALSE);
		ignore->setEnabled(FALSE);
		rename->setEnabled(FALSE);
		makePreferred->setEnabled(FALSE);
#if (QT_VERSION >= 0x030100)
		//ignore->setVisible(FALSE);
		rename->setVisible(FALSE);
		makePreferred->setVisible(FALSE);
#endif
	}
}

void EbQtMainWin::buddyRMB(QListViewItem * item, const QPoint & pos)
{
	QObject * o;
	if (! item || ! (o = ((EbQtBuddyViewItem *)item)->owner())) return;
	buddyView->setSelected(item, TRUE);
	QPopupMenu * p = NULL;
	if (o->inherits("EbQtAccount")) {
		p = accountMenu(this, (EbQtAccount *)o, TRUE, TRUE);
	} else if (o->inherits("EbQtContact")) {
		p = contactMenu(this, (EbQtContact *)o, TRUE);
	} else if (o->inherits("EbQtGroup")) {
		p = groupMenu(this, (EbQtGroup *)o);
	} else return;
	connect(p, SIGNAL(aboutToHide()), p, SLOT(deleteLater()));
	p->exec(pos);
}
void EbQtMainWin::localAcctRMB(int row, int, const QPoint & pos)
{
	localAcctsView->setCurrentCell(row, 0);
	EbQtService * s;
	if (! (s = allServices[localAcctsView->text(row,4)])) return;
	EbQtLocalAccount * l;
	if (! (l = s->localAccount(localAcctsView->text(row,3)))) return;
	QPopupMenu * p = localAccountMenu(this, l, TRUE);
	connect(p, SIGNAL(aboutToHide()), p, SLOT(deleteLater()));
	p->exec(pos);
}

void EbQtMainWin::startRename()
{
	QListViewItem * item = buddyView->selectedItem();
	if (! (item && item->isVisible())) return;
	raiseAndActivate(topSplitter);  // may have been triggered from other windows
	tabs->showPage(listTab);
	item->setRenameEnabled(0, TRUE);
	item->startRename(0);
	item->setRenameEnabled(0, FALSE);
}

void EbQtMainWin::startRename(EbQtContact * c)
{
	buddyView->setSelected(contactItems[c], TRUE);
	startRename();
}
void EbQtMainWin::startRename(EbQtGroup * g)
{
	buddyView->setSelected(groupItems[g], TRUE);
	startRename();
}

void EbQtMainWin::makeAcctPreferred()
{
	QListViewItem * item = buddyView->selectedItem();
	if (! item) return;
	QObject * o = ((EbQtBuddyViewItem *)item)->owner();
	if (! o->isA("EbQtAccount")) return;
	makeAcctPreferred((EbQtAccount *)o);
}
void EbQtMainWin::makeAcctPreferred(EbQtAccount * a)
{
	// move to its own contact, which makes it first and preferred
	a->tryMoveTo(a->contact());
}

void EbQtMainWin::toggleToolBar(bool toggle, bool sendToServer)
{
	if (sendToServer)
		putGeneralPref("show_toolbar", (toggle ? "1" : "0"));
	if (toggle)
		toolBar->show();
	else
		toolBar->hide();
}

void EbQtMainWin::toggleStatusBar(bool toggle, bool sendToServer)
{
	if (sendToServer)
		putGeneralPref("show_statusbar", (toggle ? "1" : "0"));
	if (toggle)
		statusBar()->show();
	else
		statusBar()->hide();
}

void EbQtMainWin::showAboutBox()
{
	QMessageBox::about(this,tr("About EbQt"),
		tr("EbQt version %1 &copy; 2002-2004 %2").arg(VERSION)
			.arg(EBQT_AUTHOR_HTML)
		+ "<br>\n" +
		tr("Most non-Qt icons &copy; the <i>kdeartwork</i> authors "
		"and/or contributors, special thanks to the artists behind "
		"<i>kdeclassic</i>.") + "<br>\n<br>\n" +
		tr("<i>EbQt</i> is a remote UI (front-end) for "
		"<i>eb-lite</i>, (the new <i>Everybuddy</i> core by "
		"Meredydd Luff, with total UI separation) and Qt/X11.") + 
		"<br>\n<br>\n<tt>" +
		tr("This program is free software; you can redistribute it "
		"and/or modify it under the terms of the GNU General Public "
		"License as published by the Free Software Foundation; "
		"either version 2 of the License, or (at your option) any "
		"later version.") +
		"<br>\n<br>\n" +
		tr("In addition, when you distribute EbQt binaries under "
		"section 3 of the GPL, I waive the requirement for the source "
		"code of Qt to be available. Source for all other parts is "
		"still required.")+"</tt>");
}

void EbQtMainWin::aboutQt()
{
	QMessageBox::aboutQt(this);
}

void EbQtMainWin::trySignOn(bool on)
{
	// first, update on/off status silently (without calling fn again)
	updateSignedOnOff();
	if (on) {
		tabs->setCurrentPage(1);
		statusBar()->message(tr("Signing on..."), EBQT_STATUSBAR_DELAY);
		(*sock) << "sign_on_all";
	}
}

void EbQtMainWin::trySignOff(bool off)
{
	// first, update on/off status silently (without calling fn again)
	updateSignedOnOff();
	if (off) {
		tabs->setCurrentPage(1);
		statusBar()->message(tr("Signing off..."), EBQT_STATUSBAR_DELAY);
		(*sock) << "sign_off_all";
	}
}

void EbQtMainWin::processCommands()
{
	EbQtCommand cmd;
	while(! sock->isEmpty()) {
		(*sock) >> cmd;
		processCommand(cmd);
	}
}


#define EBQT_CMDARGSFORBID(x,m) \
 if (command.count() x) { \
 	qWarning("mainwin: " m, command[0].ascii()); \
 	return; \
 }
#define	EBQT_CMDHASARGS(x) \
	EBQT_CMDARGSFORBID(!= x + 1,"ignoring %s with wrong arg count")
#define	EBQT_CMDMINARGS(x) \
	EBQT_CMDARGSFORBID( < x + 1,"ignoring incomplete %s")
#define	EBQT_CMDMAXARGS(x) \
	EBQT_CMDARGSFORBID( > x + 1,"ignoring %s with too many args")

void EbQtMainWin::processCommand(const EbQtCommand & command)
{
	// most commands use these
	EbQtLocalAccount * l = NULL;  // hush, g++, I _do_ check these
	EbQtService      * s = NULL;
	EbQtAccount      * a = NULL;
	EbQtContact      * c = NULL;
	EbQtGroup        * g = NULL;

	if (command.count() == 0) {
#ifdef EBQT_TRANSCRIPT
		qDebug("<<core<<: [ NULL COMMAND ]");
#endif
		return;
	}

#ifdef EBQT_TRANSCRIPT
	qDebug("<<core<<: %s", command.join(" ").ascii());
#endif  // EBQT_TRANSCRIPT

	// and now, ladies and gentlemen, I present:  (drum roll)...
	// the longest if-elseif-elseif-...-else KNOWN TO MAN. :-)
	if (command[0] == "cookie_accepted") {
		EBQT_CMDHASARGS(0)
		
		// tell the core what type of client we are
		(*sock) << (EbQtCommand("msg_capable")
			<< "1");
		(*sock) << (EbQtCommand("html_strip")
			<< "0");
		
		// now we need to ask for the service, local account and contact lists
		statusBar()->message(tr("Getting service list..."));
		(*sock) << "list_services";  // isn't auto-conversion wonderful? =)
		// and we _don't_ do setConnected until we see list_contacts_done
	} else if (command[0] == "cookie_rejected") {
		EBQT_CMDHASARGS(0)
		if (QMessageBox::warning(this, tr("Disconnected"),
			tr("The authentication cookie was rejected.\n"
			"Would you like EbQt to read the local\n"
			"cookie and try again?"),
			QMessageBox::Yes | QMessageBox::Default,
			QMessageBox::No  | QMessageBox::Escape)
			== QMessageBox::Yes) {
			if (loadCookie())
				tryConnect();
		} else {
			// will do a status bar msg but no dialog
			connectionClosed(FALSE);
		}

	} else if (command[0] == "list_services_done") {
		EBQT_CMDMAXARGS(0)
		if (allServices.count() <= 0) {
			QMessageBox::warning(this, tr("No services"),
				tr("This server has no services. This shouldn't happen.\n"
				"Please check you installed eb-lite properly."));
			connectionClosed(FALSE);
			return;
		}
		// preferences
		getInitialPrefs();
		// non-service actions
		(*sock) << "list_actions";
		statusBar()->message(tr("Getting local account list..."));
		(*sock) << "list_local_accounts"; // isn't auto-conversion wonderful? =)
	} else if (command[0] == "list_service_done") {
		EBQT_CMDMAXARGS(0)
		// nothing to do
	} else if (command[0] == "list_service") {
		EBQT_CMDHASARGS(3)
		createBuddyList(QString::null, QString::null, QString::null, command[1]);
		s = allServices[command[1]];

		QColor colour;
		// understands #rrggbb, X11 names, and apparently a few other names too...
		colour.setNamedColor(command[2]);
		if (! colour.isValid())
			qWarning("mainwin: service colour invalid");
		else
			s->setColour(colour);

		s->setCaps(command[3].toUInt());

		// the rest is described in subsequent commands up to list_service_done
		// they all name the service, so no state is needed here
	} else if (command[0] == "list_service_actions") {
		EBQT_CMDMINARGS(3)
		unsigned int actionCount = command[3].toUInt();
		EBQT_CMDHASARGS(actionCount + 3);
		if (! (s = allServices[command[1]])) {
			qWarning("mainwin: list_service_actions for martian service");
			return;
		}
		QStringList * actions;
		if (command[2] == "buddy")
			actions = &(s->buddyActions);
		else if (command[2] == "groupchat")
			actions = &(s->groupChatActions);
		else if (command[2] == "group_user")
			actions = &(s->groupUserActions);
		else {
			qWarning("mainwin: list_service_actions for martian action type!");
			return;
		}

		actions->clear();
		for (unsigned int i=0; i<actionCount; i++) {
			actions->append(command[i+4]);
		}
	} else if (command[0] == "list_service_states") {
		EBQT_CMDMINARGS(2)
		unsigned int stateCount = command[2].toUInt();
		EBQT_CMDHASARGS(stateCount + 2);
		if (! (s = allServices[command[1]])) {
			qWarning("mainwin: list_service_states for martian service");
			return;
		}
		s->states.clear();
		for (unsigned int i=0; i<stateCount; i++) {
			if (! s->states.contains(command[i+3]))
				s->states.append(command[i+3]);
		}

	} else if (command[0] == "list_local_accounts_done") {
		EBQT_CMDMAXARGS(0)
		localAcctsView->setCurrentCell(0,0);
		// acct editor requires services and local accounts...
		// "...and can now logically be got."
		refreshAcctPrefs();
		// done services and local accounts, one last little thing ;-)
		statusBar()->message(tr("Getting contact list..."));
		(*sock) << "list_contacts";
	} else if (command[0] == "list_local_account") {
		EBQT_CMDHASARGS(2)
		createBuddyList(QString::null, QString::null, command[1], command[2]);
		// refreshAcctPrefs() will happen once all accounts loaded
	} else if (command[0] == "add_local_account") {
		EBQT_CMDHASARGS(2)
		createBuddyList(QString::null, QString::null, command[1], command[2]);
		// get all acct prefs again since we shouldn't try to guess
		// name/s of new page(s)
		refreshAcctPrefs();
		// we may now be in a different signed on/off state...
		updateSignedOnOff();
		// let e.g. add dialog know
		emit(buddyListChanged());
		statusBar()->message(tr("Local account added"), EBQT_STATUSBAR_DELAY);
	} else if (command[0] == "del_local_account") {
		EBQT_CMDHASARGS(2)
		if (! (s = allServices[command[2]])) {
			qWarning("mainwin: del_local_account on martian service!");
			return;
		}
		// auto-delete is on
		// the server will have sent us delete_account for all buddies already
		s->removeLocal(command[1]);  // this removes its table row too
		
		// remove the pref page for it
		refreshAcctPrefs();
		// we may now be in a different signed on/off state...
		updateSignedOnOff();
		// let e.g. add dialog know
		emit(buddyListChanged());
		statusBar()->message(tr("Local account removed"), EBQT_STATUSBAR_DELAY);

	} else if (command[0] == "list_actions_done") {
		EBQT_CMDMAXARGS(0)

	} else if (command[0] == "list_actions") {
		EBQT_CMDMINARGS(2)
		unsigned int actCount = command[2].toUInt();
		EBQT_CMDHASARGS(actCount + 2);
		QStringList actions;
		for (unsigned int i=0; i<actCount; i++) {
			if (! actions.contains(command[i+3]))
				actions.append(command[i+3]);
		}
		if (command[1] == "general") {
			if (generalActions.count() > 0) {
				// extra one for separator
				for (unsigned int i=0; i <= generalActions.count(); i++)
					actionsMenu->removeItemAt(actionsMenu->count() - 2);
			}
			generalActions = actions;
			if (generalActions.count() > 0) {
				actionsMenu->insertSeparator(actionsMenu->count() - 2);
				for (unsigned int i=0; i < generalActions.count(); i++)
					actionsMenu->setItemParameter(
						actionsMenu->insertItem(generalActions[i],
						this, SLOT(tryGeneralAction(int)), 0, -1,
						actionsMenu->count() - 2), i);
			}
		} else if (command[1] == "group") {
			EbQtGroup::actions = actions;
		} else if (command[1] == "contact") {
			EbQtContact::actions = actions;
		} else if (command[1] == "buddy_generic") {
			EbQtAccount::actions = actions;
		} else {
			qWarning("mainwin: list_actions of martian type!");
			return;
		}

	} else if (command[0] == "list_contacts_done") {
		EBQT_CMDMAXARGS(0)
		statusBar()->message(tr("Connected"), EBQT_STATUSBAR_DELAY);
		setConnected();
		
		// and now we're open for business
		(*sock) << (EbQtCommand("message_hold")
			<< "0");
		// this happens automatically
		//(*sock) << "get_held_messages";
		
		// any group chats waiting for us? (now we have local accts, contacts)
		(*sock) << "list_group_chats";
		
		changeHow(showHow->currentItem());
		// and _now_ do all the contact blobs etc
		emit(buddyListChanged());
		
		// if we just connected, and there are no signed on accts,
		// then away makes no difference so default to off
		if (away->isOn()) {
			if (signOff->isOn()) {
				// nothing signed on, don't bother user
				(*sock) << "unset_away";
			} else {
				// confirm or reject it by popping up the dialog
				// (since if there were no UIs just now it'll be on)
				
				// FIXME!!! cpu -> 100% ?!?!?
				// (have tried qApp->processEvents(); it doesn't help)
				tryAway();
			}
		}

	} else if (command[0] == "list_group") {
		EBQT_CMDHASARGS(1)
		createBuddyList(command[1]);
	} else if (command[0] == "add_group") {
		EBQT_CMDHASARGS(1)
		createBuddyList(command[1]);
		// let e.g. add dialog know
		emit(buddyListChanged());
		statusBar()->message(tr("Group added"), EBQT_STATUSBAR_DELAY);
	} else if (command[0] == "del_group") {
		EBQT_CMDHASARGS(1)
		if (! allGroups[command[1]]) {
			qWarning("mainwin: del_group on martian group!");
			return;
		}
		allGroups.remove(command[1]);
		emit(buddyListChanged());  // let e.g. add dialog know
		statusBar()->message(tr("Group removed"), EBQT_STATUSBAR_DELAY);

	} else if (command[0] == "list_contact") {
		EBQT_CMDMINARGS(2)
		createBuddyList(command[1], command[2]);
		if (command.count() < 5) return;
		allGroups[command[1]]->contact(command[2])->setIgnored(
			(command[3].toUInt() > 0), command[4]);
	} else if (command[0] == "add_contact") {
		EBQT_CMDMINARGS(2)
		createBuddyList(command[1], command[2]);
		emit(buddyListChanged());  // let e.g. add dialog know
		if (command.count() < 5) return;
		allGroups[command[1]]->contact(command[2])->setIgnored(
			(command[3].toUInt() > 0), command[4]);
		statusBar()->message(tr("Contact added"), EBQT_STATUSBAR_DELAY);
	} else if (command[0] == "del_contact") {
		EBQT_CMDHASARGS(2)
		if (! (g = allGroups[command[1]])
			|| ! (c = g->contact(command[2]))) {
			qWarning("mainwin: del_contact on martian contact!");
			return;
		}
		EbQtConv * conv;
		if ((conv = singleConvs[c]))
			conv->tryClose();  // should delete
		g->removeContact(command[2]);  // autoDelete is on
		emit(buddyListChanged());  // let e.g. add dialog know
		statusBar()->message(tr("Contact removed"), EBQT_STATUSBAR_DELAY);

	} else if (command[0] == "rename_contact") {
		EBQT_CMDHASARGS(3)
		if (! (g = allGroups[command[1]])
			|| ! (c = g->contact(command[2]))) {
			qWarning("mainwin: rename_contact on martian contact");
			return;
		}
		c->setName(command[3]);
		emit(buddyListChanged());  // let e.g. add dialog know
		statusBar()->message(tr("Contact renamed"), EBQT_STATUSBAR_DELAY);

	} else if (command[0] == "move_contact") {
		EBQT_CMDHASARGS(3)
		if (! (g = allGroups[command[1]])
			|| ! (c = g->contact(command[2]))) {
			qWarning("mainwin: move_contact on martian contact");
			return;
		}
		if (! (g = allGroups[command[3]])) {
			qWarning("mainwin: move_contact to martian group!");
			return;
		}
		groupItems[c->group()]->takeItem(contactItems[c]);
		c->moveTo(g);
		groupItems[g]->insertItem(contactItems[c]);
		emit(buddyListChanged());  // let e.g. add dialog know
		statusBar()->message(tr("Contact moved"), EBQT_STATUSBAR_DELAY);
	} else if (command[0] == "move_account") {
		EBQT_CMDHASARGS(5)
		if (! (s = allServices[command[2]])
			|| ! (l = s->localAccount(command[1]))
			|| ! (a = l->account(command[3]))) {
			qWarning("mainwin: move_account on martian account!");
			return;
		}
		if (! (g = allGroups[command[4]])
			|| ! (c = g->contact(command[5]))) {
			qWarning("mainwin: move_account to martian contact");
			return;
		}
		EbQtContact * prevContact = a->contact();
		contactItems[a->contact()]->takeItem(accountItems[a]);
		a->moveTo(c);
		contactItems[c]->insertItem(accountItems[a]);
		accountItems[a]->moveItem(0);  // make it the first
		EbQtConv * conv;
		if (c != prevContact &&
			(conv = singleConvs[prevContact]))
			conv->refreshAccounts();
		if ((conv = singleConvs[c]))
			conv->refreshAccounts();
		// let e.g. add dialog know
		emit(buddyListChanged());
		statusBar()->message(c == prevContact
			? tr("Account order changed")
			: tr("Account moved"),
			EBQT_STATUSBAR_DELAY);

	} else if (command[0] == "list_account") {
		EBQT_CMDMINARGS(5)
		// group, contact, local, service, remote
		createBuddyList(command[1], command[2], command[3], command[4], command[5]);
		if (command.count() < 7) return;
		// TODO: is_blocked
		// actually may be removed, ask meredydd...
	} else if (command[0] == "add_account") {
		EBQT_CMDMINARGS(5)
		// group, contact, local, service, remote
		createBuddyList(command[1], command[2], command[3], command[4], command[5]);
		// let e.g. add dialog know
		emit(buddyListChanged());
		statusBar()->message(tr("Account added"), EBQT_STATUSBAR_DELAY);
		if (command.count() < 7) return;
		// TODO: is_blocked
		// actually may be removed, ask meredydd...
	} else if (command[0] == "del_account") {
		EBQT_CMDHASARGS(3)
		if (! (s = allServices[command[2]])
			|| ! (l = s->localAccount(command[1]))
			|| ! (a = l->account(command[3]))) {
			qWarning("mainwin: del_account on martian account!");
			return;
		}
		if (a->contact()) {  // not to worry if not there
			a->contact()->takeAccount(a);
			// any conversations will know from signals
		}
		l->removeAccount(a->name());  // will delete it too
		// let e.g. add dialog know
		emit(buddyListChanged());
		statusBar()->message(tr("Account removed"), EBQT_STATUSBAR_DELAY);

	} else if (command[0] == "buddy_logout") {
		EBQT_CMDHASARGS(3)
		if ((s = allServices[command[2]])
			&& (l = s->localAccount(command[1]))
			&& (a = l->account(command[3]))) {
			a->setStatus(EbQtAccount::Offline, "");
		} else {
			qWarning("mainwin: buddy_logout: service %s or "
				"buddy %s does not exist",
				command[2].ascii(), command[3].ascii());
		}
	} else if (command[0] == "buddy_login") {
		EBQT_CMDHASARGS(3)
		if ((s = allServices[command[2]])
			&& (l = s->localAccount(command[1]))
			&& (a = l->account(command[3]))) {
			a->setStatus(EbQtAccount::Online, "");
		} else {
			qWarning("mainwin: buddy_login: %s / %s does not exist",
				command[2].ascii(), command[3].ascii());
		}
	} else if (command[0] == "buddy_status") {
		EBQT_CMDHASARGS(5)
		if ((s = allServices[command[2]])
			&& (l = s->localAccount(command[1]))
			&& (a = l->account(command[3]))) {
			unsigned int status = command[4].toUInt();
			switch(status) {  // N.B. internally these numbers are DIFFERENT now
				case  0:
				a->setStatus(EbQtAccount::Offline, command[5]); break;
				case  1:
				a->setStatus(EbQtAccount::Online,  command[5]); break;
				case  2:
				a->setStatus(EbQtAccount::Away,    command[5]); break;
				default:
				qWarning("mainwin: invalid buddy_status"); break;
			}
		} else
			qWarning("mainwin: buddy_logoff: %s / %s does not exist",
				command[2].ascii(), command[3].ascii());

	} else if (command[0] == "ignore_contact") {
		EBQT_CMDHASARGS(3)
		if (! (g = allGroups[command[1]])
			|| ! (c = g->contact(command[2]))) {
			qWarning("mainwin: ignore_contact on martian contact!");
			return;
		}
		c->setIgnored(TRUE, command[3]);
		setSelectionActions();
		statusBar()->message(tr("Now ignoring %1").arg(c->name()),
			EBQT_STATUSBAR_DELAY);
	} else if (command[0] == "unignore_contact") {
		EBQT_CMDHASARGS(2)
		if (! (g = allGroups[command[1]])
			|| ! (c = g->contact(command[2]))) {
			qWarning("mainwin: unignore_contact on martian contact!");
			return;
		}
		c->setIgnored(FALSE, c->ignoreMsg());
		setSelectionActions();
		statusBar()->message(tr("Stopped ignoring %1").arg(c->name()),
			EBQT_STATUSBAR_DELAY);

	} else if (command[0] == "local_account_update") {
		EBQT_CMDHASARGS(5)
		if ((s = allServices[command[2]])
			&& (l = s->localAccount(command[1]))) {
			
			l->setConnected(command[3].toUInt() > 0);
			l->setReady(command[4].toUInt() > 0);
			l->setStatusMsg(command[5]);
			updateSignedOnOff();
			// perhaps some magic buttons appear when we're online
			//if (isConnected())
			//	refreshAcctPrefs();
		} else {
			qWarning("mainwin: local_account_update: %s / %s does not exist",
				command[1].ascii(), command[2].ascii());
		}

	} else if (command[0] == "set_away") {
		EBQT_CMDHASARGS(2)
		lastAway = TRUE;
		lastAwayTitle = command[1];
		lastAwayBody = command[2];
		updateAway();
		statusBar()->message(tr("Set away"), EBQT_STATUSBAR_DELAY);
	} else if (command[0] == "unset_away") {
		EBQT_CMDHASARGS(0)
		lastAway = FALSE;
		updateAway();
		statusBar()->message(tr("Unset away"), EBQT_STATUSBAR_DELAY);

	} else if (command[0] == "client_error") {
		EBQT_CMDHASARGS(1)
		EbQtDlgError * d = new EbQtDlgError(this);
		d->setCaption(tr("Error"));
		d->setMessage(command[1]
			+ "\n\n" +
			tr("This was received as a \"client\" error from the server.\n"
			"This type of error may indicate a bug in EbQt.\n"
			"If you think that is the case, please tell\n"
			"%1").arg(EBQT_AUTHOR));
		d->setMsgIcon(QMessageBox::standardIcon(QMessageBox::Critical));
		connect(d->ok, SIGNAL(clicked()), d, SLOT(deleteLater()));
		d->show();

	} else if (command[0] == "error_dialog") {
		EBQT_CMDHASARGS(3)
		EbQtDlgError * d = new EbQtDlgError(this, command[1]);
		d->setCaption(command[2]);
		d->setMessage(command[3]);
		d->setMsgIcon(QMessageBox::standardIcon(QMessageBox::Warning));
		// the tryResolve slots use QObject::sender()
		connect(d->ok, SIGNAL(clicked()), SLOT(tryResolveErrorDlg()));
		dialogs.insert(command[1], d);
		d->installEventFilter(this);  // block close events
		d->show();
	} else if (command[0] == "yesno_dialog") {
		EBQT_CMDHASARGS(4)
		EbQtDlgYesNo * d = new EbQtDlgYesNo(this, command[1]);
		d->setCaption(command[2]);
		d->setMessage(command[3]);
//		d->setMsgIcon(QMessageBox::standardIcon(QMessageBox::Information));
//			// ::Question would have been nice...
		// the tryResolve slots use QObject::sender()
		connect(d->yes, SIGNAL(clicked()), SLOT(tryResolveYesNoDlg()));
		connect(d->no,  SIGNAL(clicked()), SLOT(tryResolveYesNoDlg()));
		dialogs.insert(command[1], d);
		d->installEventFilter(this);  // block close events
		d->show();
	} else if (command[0] == "list_dialog") {
		EBQT_CMDMINARGS(4)
		unsigned int numChoices = command[4].toUInt();
		EBQT_CMDHASARGS(numChoices + 4)
		EbQtDlgList * d = new EbQtDlgList(this, command[1]);
		d->setCaption(command[2]);
		d->setMessage(command[3]);
//		d->setMsgIcon(QMessageBox::standardIcon(QMessageBox::Information));
//			// ::Question would have been nice...
		for (unsigned int i=0; i<numChoices; i++) {
			d->choices->insertItem(command[i+5]);
		}
		// the tryResolve slots use QObject::sender()
		connect(d->ok, SIGNAL(clicked()), SLOT(tryResolveListDlg()));
		dialogs.insert(command[1], d);
		d->installEventFilter(this);  // block close events
		d->show();
	} else if (command[0] == "text_dialog") {
		EBQT_CMDMINARGS(2)
		EbQtDlgText * d = new EbQtDlgText(this, command[1]);
		d->setCaption((command.count() > 3)
			? command[2]
			: tr("Edit text"));
		d->setMessage((command.count() > 3) ? command[3] : QString::null);
//		d->typeIcon->setPixmap(QMessageBox::standardIcon(QMessageBox::Information));
//			// ::Question would have been nice...
		d->setRawMode(command.count() <= 3);
		d->setText((command.count() > 3) ? command[4] : command[2]);
		// the tryResolve slots use QObject::sender()
		connect(d->ok, SIGNAL(clicked()), SLOT(tryResolveTextDlg()));
		dialogs.insert(command[1], d);
		d->installEventFilter(this);  // block close events
		d->show();
	} else if (command[0] == "dialog_resolved") {
		EBQT_CMDHASARGS(1)
		QDialog * d = dialogs.take(command[1]);
		if (d) {
			delete d;
		} else {
			qWarning("mainwin: dialog_resolved on non-existent dialog!");
		}

	} else if (command[0] == "message_send") {
		EBQT_CMDHASARGS(6)
		if (! (g = allGroups[command[1]])
			|| ! (c = g->contact(command[2]))
			|| ! (s = allServices[command[4]])
			|| ! (l = s->localAccount(command[3]))
			|| ! (a = l->account(command[5]))
			|| ! (a->contact() == c)) {
			qWarning("mainwin: invalid message_send");
			return;
		}
		getConv(c)->sentMsg(l, a, command[6]);

	} else if (command[0] == "message_receive") {
		EBQT_CMDHASARGS(6)
		if (! (g = allGroups[command[1]])
			|| ! (c = g->contact(command[2]))
			|| ! (s = allServices[command[4]])
			|| ! (l = s->localAccount(command[3]))
			|| ! (a = l->account(command[5]))
			|| ! (a->contact() == c)) {
			qWarning("mainwin: invalid message_receive");
			return;
		}
		getConv(c)->recvMsg(l, a, command[6]);

	} else if (command[0] == "group_chat_send") {
		EBQT_CMDHASARGS(2)
		EbQtConv * gc;
		if (! (gc = groupConvs[command[1]])) {
			qWarning("mainwin: group_chat_send on martian conv!");
			return;
		}
		gc->sentGroupMsg(command[2]);

	} else if (command[0] == "group_chat_recv") {
		EBQT_CMDHASARGS(3)
		EbQtConv * gc;
		if (! (gc = groupConvs[command[1]])) {
			qWarning("mainwin: group_chat_recv on martian conv!");
			return;
		}
		gc->recvGroupMsg(command[2], command[3]);
	} else if (command[0] == "group_chat_3rdperson") {
		EBQT_CMDHASARGS(2)
		EbQtConv * gc;
		if (! (gc = groupConvs[command[1]])) {
			qWarning("mainwin: group_chat_3rdperson on martian conv!");
			return;
		}
		gc->recvGroup3rd(command[2]);

	} else if (command[0] == "notify_3rdperson") {
		EBQT_CMDHASARGS(3)
		if (! (g = allGroups[command[1]])
			|| ! (c = g->contact(command[2]))) {
			qWarning("mainwin: invalid notify_3rdperson");
			return;
		}
		getConv(c)->recv3rd(0, 0, command[3]);

	} else if (command[0] == "message_hold") {
		EBQT_CMDHASARGS(1)
		// TODO: menu toggle-item somewhere?

	} else if (command[0] == "held_message") {
		EBQT_CMDMINARGS(5)
		if (command.count() > 5) {
			EBQT_CMDHASARGS(7)
			if (! (g = allGroups[command[2]])
				|| ! (c = g->contact(command[3]))
				|| ! (s = allServices[command[5]])
				|| ! (l = s->localAccount(command[4]))
				|| ! (a = l->account(command[6]))
				|| ! (a->contact() == c)) {
				qWarning("mainwin: invalid held_message");
				return;
			}
			getConv(c)->recvMsg(l, a, command[7], command[1].toInt());
		} else {
			// eb-lite core version << 20031206
			if (! (s = allServices[command[3]])
				|| ! (l = s->localAccount(command[2]))
				|| ! (a = l->account(command[4]))
				|| ! (c = a->contact())) {
				qWarning("mainwin: invalid held_message");
				return;
			}
			getConv(c)->recvMsg(l, a, command[5], command[1].toInt());
		}

	} else if (command[0] == "held_sent_message") {
		EBQT_CMDMINARGS(5)
		if (command.count() > 5) {
			EBQT_CMDHASARGS(7)
			if (! (g = allGroups[command[2]])
				|| ! (c = g->contact(command[3]))
				|| ! (s = allServices[command[5]])
				|| ! (l = s->localAccount(command[4]))
				|| ! (a = l->account(command[6]))
				|| ! (a->contact() == c)) {
				qWarning("mainwin: invalid held_sent_message");
				return;
			}
			getConv(c)->sentMsg(l, a, command[7], command[1].toInt());
		} else {
			// eb-lite core version << 20031206
			if (! (s = allServices[command[3]])
				|| ! (l = s->localAccount(command[2]))
				|| ! (a = l->account(command[4]))
				|| ! (c = a->contact())) {
				qWarning("mainwin: invalid held_sent_message");
				return;
			}
			getConv(c)->sentMsg(l, a, command[5], command[1].toInt());
		}

	} else if (command[0] == "held_3rdperson") {
		EBQT_CMDHASARGS(4)
		if (! (g = allGroups[command[2]])
			|| ! (c = g->contact(command[3]))) {
			qWarning("mainwin: invalid held_3rdperson");
			return;
		}
		getConv(c)->recv3rd(0, 0, command[4], command[1].toInt());

	} else if (command[0] == "held_messages_done") {
		EBQT_CMDHASARGS(1)  // message count
		// TODO: do anything here? (check msg count?)

	} else if (command[0] == "message_waiting") {
		EBQT_CMDHASARGS(0)
		// FIXME: right thing to do?
		statusBar()->message(tr("Message(s) held, fetching..."),
			EBQT_STATUSBAR_DELAY);
		if (isConnected())
			(*sock) << "get_held_messages";
		// else we're sending message_hold 0 anyway once we're ready

	} else if (command[0] == "new_group_chat") {
		EBQT_CMDMINARGS(3)
		EBQT_CMDMAXARGS(4)
		if (! (s = allServices[command[2]])
			|| ! (l = s->localAccount(command[1]))) {
			qWarning("mainwin: new_group_chat on martian local account!");
			return;
		}
		EbQtConv * gc;
		if (! (gc = groupConvs[command[3]])) {
//			QSettings settings;
//			startSettings(settings);
			gc = newConv(groupConvsInTabs->isOn(),
				QString("groupchat %1")
				.arg(command[3]));
			groupConvs.insert(command[3], gc);
			gc->setGroupChat(command[3], l);
			connect(gc,
				SIGNAL(convClosing(EbQtConv *)),
				SLOT(removeGroupConv(EbQtConv *)));
			connect(gc,
				SIGNAL(wantsGroupSend(const QString &, const QString &)),
				SLOT(tryGroupSend(const QString &, const QString &)));
			connect(gc,
				SIGNAL(wantsGroupInvite(const QString &, const QString &)),
				SLOT(tryGroupInvite(const QString &, const QString &)));
			connect(gc,
				SIGNAL(wantsGroupLeave(const QString &)),
				SLOT(tryGroupLeave(const QString &)));
			connect(gc,
				SIGNAL(wantsIgnore(EbQtContact *)),
				SLOT(tryIgnore(EbQtContact *)));
			connect(gc,
				SIGNAL(wantsChatAction(
					const QString &,
					const QString &,
					EbQtLocalAccount *)),
				SLOT(tryChatAction(
					const QString &,
					const QString &,
					EbQtLocalAccount *)));
			connect(gc,
				SIGNAL(wantsChatUserAction(
					const QString &,
					const QString &,
					const QString &)),
				SLOT(tryChatUserAction(
					const QString &,
					const QString &,
					const QString &)));
			connect(gc,
				SIGNAL(wantsChatUserMenu(
					const QString &, int,
					const QPoint &,
					EbQtConv *)),
				SLOT(doChatUserMenu(
					const QString &, int,
					const QPoint &,
					EbQtConv *)));
			connect(gc,
				SIGNAL(wantsOpenConv(const QString &,
					EbQtConv *)),
				SLOT(tryOpenConv(const QString &,
					EbQtConv *)));
			connect(gc,
				SIGNAL(wantsAdd(const QString &,
					EbQtLocalAccount *, EbQtConv *)),
				SLOT(tryAdd(const QString &,
					EbQtLocalAccount *, EbQtConv *)));
			gc->show();
			statusBar()->message(tr("Joined group chat"),
				EBQT_STATUSBAR_DELAY);
		}  // else warn?
		if (groupConvsInTabs->isOn())
			convsTabWidget->showPage(gc);
		raiseAndActivate(gc->topLevelWidget());
		if (command.count() > 4)  // fixed title, e.g. irc channel name
			gc->setCaption(command[4]);
	} else if (command[0] == "close_group_chat") {
		EBQT_CMDHASARGS(1)
		EbQtConv * gc;
		if (! (gc = groupConvs[command[1]])) {
			qWarning("mainwin: group_chat_close on martian conv!");
			return;
		}
		removeGroupConv(gc);

	} else if (command[0] == "list_group_chat_done") {
		EBQT_CMDHASARGS(0)
		// not useful to us
	} else if (command[0] == "list_group_user") {
		EBQT_CMDHASARGS(2)
		EbQtConv * gc;
		if (! (gc = groupConvs[command[1]])) {
			qWarning("mainwin: list_group_user on martian conv!");
			return;
		}
		gc->addGroupUser(command[2], TRUE);  // quietly
	} else if (command[0] == "group_chat_joined") {
		EBQT_CMDHASARGS(2)
		EbQtConv * gc;
		if (! (gc = groupConvs[command[1]])) {
			qWarning("mainwin: group_chat_joined on martian conv!");
			return;
		}
		gc->addGroupUser(command[2]);
	} else if (command[0] == "group_chat_left") {
		EBQT_CMDHASARGS(2)
		EbQtConv * gc;
		if (! (gc = groupConvs[command[1]])) {
			qWarning("mainwin: group_chat_left on martian conv!");
			return;
		}
		gc->removeGroupUser(command[2]);

	} else if (command[0] == "list_pref_page") {
		EBQT_CMDHASARGS(2)
		EbQtPrefPage * p;
		if (! (p = prefPages[command[1]])) {
			qWarning("mainwin: unexpected pref page, displaying "
				"as top-level dialog!");
			p = newPrefPageInDialog(command[1]);
		}
		p->setCaption(command[2]);

	} else if (command[0] == "list_subpages") {
		EBQT_CMDHASARGS(1)
		EbQtPrefPage * p;
		if (! (p = prefPages[command[1]])) {
			qWarning("mainwin: list_subpages on martian prefs page!");
			return;
		}
		p->clearSubPages();

	} else if (command[0] == "list_subpage") {
		EBQT_CMDHASARGS(3)
		EbQtPrefPage * p;
		if (! (p = prefPages[command[1]])) {
			qWarning("mainwin: list_subpage on martian page!");
			return;
		}
		if (prefPages[command[2]]) {
			qWarning("mainwin: list_subpage on already existing "
				"subpage %s of %s!",
				command[1].ascii(), command[2].ascii());
			return;
		}
		
		// make p a new child page
		// constructor will notify parent page of its existence
		prefPages.insert(command[2],
			(p = new EbQtPrefPage(p, command[2], &allServices)));
		connect(p, SIGNAL(destroyed(QObject *)),
			SLOT(removePrefPage(QObject *)));
		connect(p, SIGNAL(wantsRefresh(EbQtPrefPage *)),
			SLOT(refreshPrefPage(EbQtPrefPage *)));
		connect(p, SIGNAL(wantsValueSet(const QString &,
			const QString &, EbQtPrefPage *)),
			SLOT(trySetPref(const QString &, const QString &,
				EbQtPrefPage *)));
		connect(p, SIGNAL(wantsSave()), SLOT(trySaveConfig()));
		p->setCaption(command[3]);
		refreshPrefPage(command[2]);

	} else if (command[0] == "list_subpages_done") {
		EBQT_CMDHASARGS(0)
		// TODO:
		// nothing to do atm... now if this had some _context_,
		// it could be _useful_... like not clearing the subpages
		// but removing the ones not mentioned at this stage

	} else if (command[0] == "list_components") {
		EBQT_CMDHASARGS(1)
		EbQtPrefPage * p;
		if (! (p = prefPages[command[1]])) {
			qWarning("mainwin: list_components on martian prefs page!");
			return;
		}
		p->clearComponents();

	} else if (command[0] == "list_component") {
		EBQT_CMDHASARGS(5)
		EbQtPrefPage * p;
		if (! (p = prefPages[command[1]])) {
			qWarning("mainwin: list_component on martian page!");
			return;
		}
		// type, name, title, value
		p->addComponent(command[2], command[3], command[4], command[5]);

	} else if (command[0] == "option_data") {
		EBQT_CMDMINARGS(3)
		unsigned int optionCount = command[3].toUInt();
		EBQT_CMDMAXARGS(optionCount + 3)
		EbQtPrefPage * p;
		if (! (p = prefPages[command[1]])) {
			qWarning("mainwin: option_data on martian page!");
			return;
		}
		QStringList options;
		for (unsigned int i=0; i<optionCount; i++) {
			options.append(command[i+4]);
		}
		p->addOptionData(command[2], options);

	} else if (command[0] == "list_components_done") {
		EBQT_CMDHASARGS(0)
		// TODO: same gripe as list_subpages_done

	} else if (command[0] == "list_pref_page_done") {
		EBQT_CMDHASARGS(0)
		// nothing to do, ever.

	} else if (command[0] == "return_data") {
		EBQT_CMDMINARGS(3)
		QString key(command[1]);
		QString value(command[2]);
		QString section(command[3]);
		if (value.isEmpty())
			return;  // defaults mechanism: set your default before get_data
		if (section == "general") {
			EBQT_CMDHASARGS(3)
			gotGeneralPref(key, value);
		} else if (section == "contact") {
			EBQT_CMDHASARGS(5)
			EbQtGroup * g;
			EbQtContact * c;
			if (! (g = allGroups[command[4]])
				|| ! (c = g->contact(command[5]))) {
				qWarning("mainwin: return_data on martian contact!");
				return;
			}
			gotContactPref(key, value, c);
		} else {
			qWarning("mainwin: return_data of martian type!");
		}

	} else {
		qWarning("mainwin: processCommand: unknown command");
		statusBar()->message(tr("%1 not understood!").arg(command[0]), 5000);
	}
}

void EbQtMainWin::createBuddyList(
	const QString &mkGroup, const QString &mkContact,
	const QString &mkLocal, const QString &mkService,
	const QString &mkAccount)
{
	EbQtGroup * g = NULL;
	EbQtContact * c = NULL;
	EbQtLocalAccount * l = NULL;
	EbQtService * s = NULL;
	EbQtAccount * a = NULL;

	EbQtBuddyViewItem * i;

	if (! mkGroup.isEmpty()) {
		if (! (g = allGroups[mkGroup])) {
			g = new EbQtGroup(this, mkGroup);
			QListViewItem * after = buddyView->firstChild();
			if (! after) {
				i = new EbQtBuddyViewItem(buddyView, mkGroup, g);
			} else {
				while (after->nextSibling())
					after = after->nextSibling();
				i = new EbQtBuddyViewItem(buddyView,
					(EbQtBuddyViewItem *)after, mkGroup, g);
			}
			i->setRenameEnabled(0, FALSE);
			i->setRenameEnabled(1, FALSE);
			i->setDropEnabled(TRUE);
			buddyView->blockSignals(TRUE);
			i->setOpen(TRUE);
			buddyView->blockSignals(FALSE);
			groupItems.insert(g,i);
			connect(g, SIGNAL(itemNeedsUpdate(EbQtGroup *)),
				SLOT(updateGroupItem(EbQtGroup *)));
			connect(g, SIGNAL(wantsAction(int, EbQtGroup *)),
				SLOT(tryGroupAction(int, EbQtGroup *)));
			connect(g, SIGNAL(wantsRenameStart(EbQtGroup *)),
				SLOT(startRename(EbQtGroup *)));
			connect(g, SIGNAL(wantsRemove(EbQtGroup *)),
				SLOT(tryRemove(EbQtGroup *)));
			connect(g, SIGNAL(destroyed(QObject *)),
				SLOT(removeGroupItem(QObject *)));
			allGroups.insert(mkGroup, g);
		}
		if (! mkContact.isEmpty()) {
			if (! (c = g->contact(mkContact))) {
				c = new EbQtContact(g, mkContact);
				EbQtBuddyViewItem * gi = groupItems[g];
				QListViewItem * after = gi->firstChild();
				if (! after) {
					i = new EbQtBuddyViewItem(gi, mkContact, c);
				} else {
					while (after->nextSibling())
						after = after->nextSibling();
					i = new EbQtBuddyViewItem(gi,
						(EbQtBuddyViewItem *)after, mkContact, c);
				}
				i->setRenameEnabled(0, FALSE);
				i->setRenameEnabled(1, FALSE);
				i->setDragEnabled(TRUE);
				i->setDropEnabled(TRUE);
				buddyView->blockSignals(TRUE);
				i->setOpen(showHow->currentItem() == 1);
				buddyView->blockSignals(FALSE);
				contactItems.insert(c,i);
				connect(c, SIGNAL(itemNeedsUpdate(EbQtContact *)),
					SLOT(updateContactItem(EbQtContact *)));
				connect(c, SIGNAL(statusChanged(
					EbQtAccount::AccountStatus, EbQtContact *)),
					SLOT(notifyContactStatus(
					EbQtAccount::AccountStatus, EbQtContact *)));
				connect(c, SIGNAL(wantsAction(int, EbQtContact *)),
					SLOT(tryContactAction(int, EbQtContact *)));
				// renaming is started from RMB (since dblclick -> conv)
				connect(c, SIGNAL(nameChanged(const QString &,
						EbQtContact *)),
					SLOT(renameContact(const QString &,
						EbQtContact *)));
				connect(c, SIGNAL(wantsRenameStart(EbQtContact *)),
					SLOT(startRename(EbQtContact *)));
				connect(c, SIGNAL(wantsRemove(EbQtContact *)),
					SLOT(tryRemove(EbQtContact *)));
				connect(c, SIGNAL(wantsOpenConv(EbQtContact *)),
					SLOT(tryOpenConv(EbQtContact *)));
				connect(c, SIGNAL(wantsToMoveTo(const QString &,
						EbQtContact *)),
					SLOT(tryMoveContact(const QString &,
						EbQtContact *)));
				connect(c, SIGNAL(wantsIgnore(bool, EbQtContact *)),
					SLOT(tryIgnore(bool, EbQtContact *)));
				connect(c, SIGNAL(destroyed(QObject *)),
					SLOT(removeContactItem(QObject *)));
			}
		}
	}
	
	if (! (mkService.isEmpty() || (s = allServices[mkService]))) {
		s = new EbQtService(this, mkService);
		allServices.insert(mkService, s);
	}
	
	if (s && (! mkLocal.isEmpty()) && ! (l = s->localAccount(mkLocal))) {
		l = new EbQtLocalAccount(s, mkLocal);

		// we don't want signals about incomplete things while we build them
		localAcctsView->blockSignals(TRUE);

		// fill it in in the top row
		localAcctsView->insertRows(0);
		localAcctsView->setText(0,3, mkLocal);
		localAcctsView->setText(0,4, mkService);
		localAcctsView->setPixmap(0,4, s->pixmap(
			localAcctsView->rowHeight(0),
			localAcctsView->rowHeight(0)));
		localAcctsView->setItem(0,1,
			new QCheckTableItem(localAcctsView, ""));
		localAcctsView->setItem(0,2,
			new QCheckTableItem(localAcctsView, ""));
		localAcctsView->setItem(0,0,
			new QComboTableItem(localAcctsView, s->states));

		// default Offline if there is one
		((QComboTableItem *)(localAcctsView->item(0,0)))
			->setCurrentItem("Offline");

		// grab it quick
		localAcctItems.insert(l,localAcctsView->item(0,3));
		// because after this we're never sure what row it's on
		localAcctsView->sortColumn(3, TRUE, TRUE);
		localAcctsView->sortColumn(4, TRUE, TRUE);
		// avoid initial mess which would have combo selected and not current
		// => looking awful for some reason (some styles)
		localAcctsView->setCurrentCell(localAcctItems[l]->row(), 0);

		// ok, if stuff happens to these items, we can now cope :-)
		localAcctsView->blockSignals(FALSE);

		connect(l, SIGNAL(itemNeedsUpdate(EbQtLocalAccount *)),
			SLOT(updateLocalAcctItem(EbQtLocalAccount *)));
		connect(l, SIGNAL(wantsStatusChange(const QString &,
			EbQtLocalAccount *)),
			SLOT(tryChangeLocalStatus(const QString &,
			EbQtLocalAccount *)));
		connect(l, SIGNAL(wantsRemove(EbQtLocalAccount *)),
			SLOT(tryRemove(EbQtLocalAccount *)));
		connect(l, SIGNAL(destroyed(QObject *)),
			SLOT(removeLocalAcctItem(QObject *)));
		s->addLocal(l);
		updateLocalAcctItem(l);
	}

	if (c && l && s
		&& (! mkAccount.isEmpty()) && ! (a = l->account(mkAccount))) {
		a = new EbQtAccount(c, l, mkAccount);
		EbQtBuddyViewItem * ci = contactItems[c];
		QListViewItem * after = ci->firstChild();
		if (! after) {
			i = new EbQtBuddyViewItem(ci, mkAccount, a);
		} else {
			while (after->nextSibling())
				after = after->nextSibling();
			i = new EbQtBuddyViewItem(ci,
				(EbQtBuddyViewItem *)after, mkAccount, a);
		}
		i->setRenameEnabled(0, FALSE);
		i->setRenameEnabled(1, FALSE);
		i->setDragEnabled(TRUE);
		accountItems.insert(a,i);
		connect(a, SIGNAL(itemNeedsUpdate(EbQtAccount *)),
			SLOT(updateAccountItem(EbQtAccount *)));
		connect(a, SIGNAL(wantsServiceAction(int, EbQtAccount *)),
			SLOT(tryAcctServiceAction(int, EbQtAccount *)));
		connect(a, SIGNAL(wantsGenericAction(int, EbQtAccount *)),
			SLOT(tryAcctGenericAction(int, EbQtAccount *)));
		connect(a, SIGNAL(wantsToMoveTo(EbQtContact *, EbQtAccount *)),
			SLOT(tryMoveAccount(EbQtContact *, EbQtAccount *)));
		connect(a, SIGNAL(wantsToBePreferred(EbQtAccount *)),
			SLOT(makeAcctPreferred(EbQtAccount *)));
		connect(a, SIGNAL(wantsOpenConv(EbQtAccount *)),
			SLOT(tryOpenConv(EbQtAccount *)));
		connect(a, SIGNAL(wantsRemove(EbQtAccount *)),
			SLOT(tryRemove(EbQtAccount *)));
		connect(a, SIGNAL(destroyed(QObject *)),
			SLOT(removeAccountItem(QObject *)));
		updateAccountItem(a);
		EbQtConv * conv;
		if ((conv = singleConvs[c]))
			conv->refreshAccounts();
	}
}

void EbQtMainWin::tryShowPrefs()
{
	EbQtPrefPage * p;
	if (! (p = prefPages["mainprefs"])) {
		newPrefPageInDialog("mainprefs");  // stop it being labelled unexpected
	} else {
		// dialog is already open, call attention to it
		raiseAndActivate(p);
	}
	// refresh it in any case
	refreshPrefPage("mainprefs");
}

void EbQtMainWin::updateShowContactsExpanded()
{
	buddyView->blockSignals(TRUE); // customExpansion
	for (QDictIterator<EbQtGroup> gi(allGroups); (*gi)!=0; ++gi) {
		groupItems[*gi]->setOpen(TRUE);
		for (QDictIterator<EbQtContact> ci = (*gi)->contacts();
			(*ci)!=0; ++ci) {
			contactItems[*ci]->setOpen(showContactsExpanded);
		}
	}
	buddyView->blockSignals(FALSE);
}
void EbQtMainWin::changeHow(int how)
{
	if (how == 0)  // "(custom)"
		return;
	showContactsExpanded = (how == 1);
	updateShowContactsExpanded();
	putGeneralPref("show_contacts_expanded", (showContactsExpanded ? "1" : "0"));
}
void EbQtMainWin::changeHow()
{
	showHow->blockSignals(TRUE);
	showHow->setCurrentItem(showContactsExpanded ? 1 : 2);
	showHow->blockSignals(FALSE);
	updateShowContactsExpanded();
}

void EbQtMainWin::customTreeExpansion()
{
	showHow->setCurrentItem(0);
}
void EbQtMainWin::changeWho(int who)
{
	changeWhoQuietly(who);
	putGeneralPref("show_offline_contacts", (showOfflineContacts ? "1" : "0"));
}
void EbQtMainWin::changeWhoQuietly(int who)
{
	showOfflineContacts = (who == 0);
	updateAll();
}
void EbQtMainWin::changeWho()
{
	showWho->blockSignals(TRUE);
	showWho->setCurrentItem(showOfflineContacts ? 0 : 1);
	showWho->blockSignals(FALSE);
	updateAll();
}

void EbQtMainWin::updateAll()
{
	for (QDictIterator<EbQtGroup> gi(allGroups); (*gi)!=0; ++gi) {
		updateGroupItem(*gi);
	}
}
void EbQtMainWin::updateAccountItem(EbQtAccount * a)
{
	QListViewItem * i;
	if (! (i = accountItems[(EbQtAccount *)a])) return;
	if ((! showOfflineContacts)
		&& a->status() <= EbQtAccount::Offline) {
		i->setVisible(FALSE);
	} else {
		i->setVisible(TRUE);
		i->setText(1, a->statusMsg().isEmpty()
			? EbQtAccount::statusDesc(a->status())
			: a->statusMsg());
		i->setPixmap(0, a->service()->pixmap(20,20,  // i->height(),i->height(),
			a->status() != EbQtAccount::Online));
	}
}
void EbQtMainWin::updateContactItem(EbQtContact * c)
{
	QListViewItem * i;
	if (! (i = contactItems[(EbQtContact *)c])) return;
	if ((! showOfflineContacts)
		&& ! c->isOnline()) {
		i->setVisible(FALSE);
	} else {
		i->setVisible(TRUE);
		i->setText(1, c->statusMsg());
		EbQtAccount * a = c->pickAccount();
		if (a) {
			i->setPixmap(0, a->service()->pixmap(20,20,
					// i->height(),i->height(),
				a->status() != EbQtAccount::Online));
		} else {
			i->setPixmap(0, EbQtService::drawPixmap(
				QColor() /* invalid */, 20, 20));
		}
		for (QPtrListIterator<EbQtAccount> ai = c->accounts();
			ai.current(); ++ai) {
			updateAccountItem(*ai);
		}
	}
	updateSummary();  // TODO: don't do this so often!
}
void EbQtMainWin::updateGroupItem(EbQtGroup * g)
{
	QListViewItem * i;
	if (! (i = groupItems[(EbQtGroup *)g])) return;
	if ((! showOfflineContacts)
		&& ! g->isOnline()) {
		i->setVisible(FALSE);
	} else {
		i->setVisible(TRUE);
		for (QDictIterator<EbQtContact> ci = g->contacts();
			ci.current(); ++ci) {
			updateContactItem(*ci);
		}
	}
}
void EbQtMainWin::updateLocalAcctItem(EbQtLocalAccount * l)
{
	int row = localAcctItems[(EbQtLocalAccount *)l]->row();

	((QComboTableItem *)(localAcctsView->item(row,0)))->setCurrentItem(l->statusMsg());
	// FIXME: status not previously encountered will not be set

	((QCheckTableItem *)(localAcctsView->item(row,1)))->setChecked(l->connected());
	((QCheckTableItem *)(localAcctsView->item(row,2)))->setChecked(l->ready());
}

void EbQtMainWin::renameContact(const QString & newName, EbQtContact * contact)
{
	QListViewItem * i = contactItems[contact];
	if (i)
		i->setText(0, newName);
}

void EbQtMainWin::tryBuddyAction(const QString & action, EbQtAccount * a)
{
	(*sock) << (EbQtCommand("perform_action")
		<< "buddy"
		<< a->localAccount()->name()
		<< a->service()->name()
		<< a->name()
		<< action);
}

void EbQtMainWin::tryAdd(const QString & acct, EbQtLocalAccount * l,
	EbQtConv *)
{
	tryAdd(l->service()->name(),  // TODO: conv->channel() minus any #
		acct, l->name(), l->service()->name(), acct);
}
void EbQtMainWin::tryAdd(const QString & g, const QString & c,
	const QString & l, const QString & s, const QString & a)
{
	if (addDlg == NULL) {
		// give the new dialog the proverbial keys to the city...
		addDlg = new EbQtAdd(this, &allGroups, &allServices, "addDlg");
		connect(addDlg, SIGNAL(wantsToAdd(const QString &, const QString &,
			const QString &, const QString &, const QString &)),
			SLOT(tryCreateBuddyList(const QString &, const QString &,
			const QString &, const QString &, const QString &)));
		addDlg->installEventFilter(this);
		addDlg->show();
	} else {
		// dialog is already open, call attention to it
		raiseAndActivate(addDlg);
	}
	
	if (g.isEmpty())	addDlg->groupCB  ->clearEdit();
	else			addDlg->groupCB  ->setCurrentText(g);
	if (c.isEmpty())	addDlg->contactCB->clearEdit();
	else			addDlg->contactCB->setCurrentText(c);
	if (s.isEmpty())	addDlg->refreshLocals();
	else			addDlg->serviceCB->setCurrentText(s);
	if (! l.isEmpty())	addDlg->localCB  ->setCurrentText(l);
	if (a.isEmpty())	addDlg->accountCB->clearEdit();
	else			addDlg->accountCB->setCurrentText(a);
	if ((! s.isNull()) && l.isEmpty())
		addDlg->localCB  ->setFocus();
	else if (g.isEmpty())
		addDlg->groupCB  ->setFocus();
	else if (c.isEmpty())
		addDlg->contactCB->setFocus();
	else	addDlg->accountCB->setFocus();
}
void EbQtMainWin::tryAdd()
{
	if (tabs->currentPage() == listTab) {
		if (! buddyView->selectedItem()) {
			tryAdd(QString::null, QString::null,
				QString::null, QString::null, QString::null);
			return;
		}
		QObject * o = ((EbQtBuddyViewItem *)(buddyView->selectedItem()))->owner();
		if (! o) return;  // really shouldn't happen
		if (o->inherits("EbQtAccount")) {
			tryAdd(((EbQtAccount *)o)->contact()->group()->name(),
				((EbQtAccount *)o)->contact()->name(),
				QString::null, QString::null, QString::null);
		} else if (o->inherits("EbQtContact")) {
			tryAdd(((EbQtContact *)o)->group()->name(),
				((EbQtContact *)o)->name(),
				QString::null, QString::null, QString::null);
		} else if (o->inherits("EbQtGroup")) {
			tryAdd(((EbQtGroup *)o)->name(), QString::null,
				QString::null, QString::null, QString::null);
		} // else some funky stuff is afoot
	} else if (tabs->currentPage() == localAcctsTab) {
		int row = localAcctsView->currentRow();
		EbQtService * s;
		if (row >= 0 && row < localAcctsView->numRows()
			&& (s = allServices[localAcctsView->text(row,4)])) {
			tryAdd(QString::null, QString::null,
				QString::null, s->name(), QString::null);
		} else {
			tryAdd(QString::null, QString::null,
				QString::null, "", QString::null);
		}
	} // else connectTab, shouldn't have been enabled anyway
}

void EbQtMainWin::tryCreateBuddyList(
	const QString &mkGroup, const QString &mkContact,
	const QString &mkLocal, const QString &mkService,
	const QString &mkAccount)
{
	EbQtGroup        * g = NULL;
	EbQtContact      * c = NULL;
	EbQtLocalAccount * l = NULL;
	EbQtService      * s = NULL;
	EbQtAccount      * a = NULL;
	bool addedSomething = FALSE;

	if (! mkGroup.isEmpty()) {
		if (! (g = allGroups[mkGroup])) {
			(*sock) << (EbQtCommand("add_group")
				<< mkGroup);
			addedSomething = TRUE;
		}
		if (! mkContact.isEmpty() &&
			(! g || ! (c = g->contact(mkContact)))) {
			(*sock) << (EbQtCommand("add_contact")
				<< mkGroup
				<< mkContact);
			addedSomething = TRUE;
		}
	}

	// services are the odd one out, we can't make them
	// (except maybe by loading plugins... in the not-too-distant future)

	// everything below here needs a valid service
	if (! mkService.isEmpty() && (s = allServices[mkService])) {

		if (! mkLocal.isEmpty() && ! (l = s->localAccount(mkLocal))) {
			(*sock) << (EbQtCommand("add_local_account")
				<< mkLocal
				<< mkService);
			addedSomething = TRUE;
		}

		if (! mkGroup.isEmpty() && ! mkContact.isEmpty()
			&& ! mkLocal.isEmpty() && ! mkAccount.isEmpty()  // svc checked
			// valid account and place to put it, good so far
			&& (! c || ! l || ! (a = l->account(mkAccount))
			|| a->contact() != c)) {
			// doesn't all exist already either
			(*sock) << (EbQtCommand("add_account")
				<< mkGroup
				<< mkContact
				<< mkLocal
				<< mkService
				<< mkAccount);
			addedSomething = TRUE;
		}
	}

	if (addedSomething) {
		statusBar()->message(tr("Adding..."), EBQT_STATUSBAR_DELAY);
		trySaveConfig();
	} else {
		qWarning("mainwin: tryCreateBuddyList didn't add anything!");
	}
}

void EbQtMainWin::tryRemove()
{
	if (tabs->currentPage() == listTab) {
		QObject * tmp = ((EbQtBuddyViewItem *)(buddyView->selectedItem()))->owner();
		if (! tmp) {
			QMessageBox::warning(this, tr("No selection"),
				tr("You must select something to remove first."));
			return;
		}
		QString target;
		if (tmp->inherits("EbQtAccount")) {
			tryRemove((EbQtAccount *)tmp);
		} else if (tmp->inherits("EbQtContact")) {
			tryRemove((EbQtContact *)tmp);
		} else if (tmp->inherits("EbQtGroup")) {
			tryRemove((EbQtGroup *)tmp);
		}
	} else if (tabs->currentPage() == localAcctsTab) {
		int row = localAcctsView->currentRow();
		if (row < 0 || row > localAcctsView->numRows()) return;
		EbQtService * s;
		if (! (s = allServices[localAcctsView->text(row,4)])) return;
		EbQtLocalAccount * l;
		if (! (l = s->localAccount(localAcctsView->text(row,3)))) return;
		tryRemove(l);
	} else {
		QMessageBox::warning(this, tr("Can't do that on this tab"),
			tr("That button shouldn't have been enabled here,\n"
			"please report this as a bug."));
	}
}
void EbQtMainWin::tryRemove(EbQtAccount * a)
{
	if (QMessageBox::warning(this, tr("Remove account?"),
		QString(a->name()) + "\n\n"
		+ tr("Really remove this account from your buddy list?")
		+ "\n\n"
		+ tr("Depending on the service, if you change your mind\n"
		  "and add it again, your contact's permission might\n"
		  "be required (again)."),
		QMessageBox::Yes | QMessageBox::Default,
		QMessageBox::No  | QMessageBox::Escape)
		== QMessageBox::Yes) {
		statusBar()->message(tr("Removing account..."),
			EBQT_STATUSBAR_DELAY);
		(*sock) << (EbQtCommand("del_account")
			<< a->localAccount()->name()
			<< a->service()->name()
			<< a->name());
		// server will send us confirmation or an error
		trySaveConfig();
	}
}
void EbQtMainWin::tryRemove(EbQtContact * c)
{
	if (c->accounts().count() > 0) {
		QMessageBox::warning(this, tr("Contact has accounts"),
			tr("This contact still has accounts, "
				"remove them first."));
		// if we wanted to just delete the accounts
		// too, we'd have to do it here (recursively),
		// the core sends a client error otherwise
		// (not that it's defficult to do, I'm just
		// being consistent)
		return;
	}
	if (QMessageBox::warning(this, tr("Remove contact?"),
		QString(c->name()) + "\n\n"
		+ tr("Really remove this contact from your buddy list?"),
		QMessageBox::Yes | QMessageBox::Default,
		QMessageBox::No  | QMessageBox::Escape)
		== QMessageBox::Yes) {
		statusBar()->message(tr("Removing contact..."),
			EBQT_STATUSBAR_DELAY);
		(*sock) << (EbQtCommand("del_contact")
			<< c->group()->name()
			<< c->name());
		// server will send us confirmation or an error
		trySaveConfig();
	}
}
void EbQtMainWin::tryRemove(EbQtGroup * g)
{
	if (g->contacts().count() > 0) {
		QMessageBox::warning(this, tr("Group not empty"),
			tr("This group still has contacts, "
				"remove them first."));
		// again, we could do it here (recursively)
		// if that was wanted
		return;
	}
	if (QMessageBox::warning(this, tr("Remove group?"),
		QString(g->name()) + "\n\n"
		+ tr("Really remove this group from your buddy list?"),
		QMessageBox::Yes | QMessageBox::Default,
		QMessageBox::No  | QMessageBox::Escape)
		== QMessageBox::Yes) {
		statusBar()->message(tr("Removing group..."),
			EBQT_STATUSBAR_DELAY);
		(*sock) << (EbQtCommand("del_group")
			<< g->name());
		// server will send us confirmation or an error
		trySaveConfig();
	}
}
void EbQtMainWin::tryRemove(EbQtLocalAccount * l)
{
	if (QMessageBox::warning(this, tr("Remove local account?"),
		QString(l->name()) + "\n\n"
		+ tr("Really remove this local account?")
		+ "\n\n"
		+ tr("This will forget the settings of this account\n"
		"(e.g. passwords), and if this service doesn't keep\n"
		  "your buddy list on the server, you'll lose that\n"
		  "too (unless you re-add this account before the\n"
		  "next time the config is saved)."),
		QMessageBox::Yes, QMessageBox::No) == QMessageBox::Yes) {
			statusBar()->message(tr("Removing local account..."),
			EBQT_STATUSBAR_DELAY);
		(*sock) << (EbQtCommand("del_local_account")
			<< l->name()
			<< l->service()->name());
		trySaveConfig();
	}
}

void EbQtMainWin::tryChangeLocalStatus(int row, int col)
{
	if (col != 0) {
		qWarning("mainwin: huh? column %u should be read-only!", col);
		return;
	}
	
	EbQtService * s;
	EbQtLocalAccount * l;
	if (! (s = allServices[localAcctsView->text(row,4)])
		|| ! (l = s->localAccount(localAcctsView->text(row,3)))) {
		qWarning("tryChangeLocalStatus on martian local account!");
		return;
	}
	
	// the status combo
	QComboTableItem * i = (QComboTableItem *)(localAcctsView->item(row,col));
	
	if (i->currentText() == l->statusMsg()) return;
	
	tryChangeLocalStatus(i->currentText(), l);
	
	// set back to what it was, calls this fn again but caught by ..==.. above
	i->setCurrentItem(l->statusMsg());
}
void EbQtMainWin::tryChangeLocalStatus(const QString & status,
	EbQtLocalAccount * l)
{
	(*sock) << (EbQtCommand("set_local_account_status")
		<< l->name()
		<< l->service()->name()
		<< status);
}

void EbQtMainWin::tryResolveErrorDlg()
{
	// sender is hopefully a button
	const QObject * b = QObject::sender();
	(*sock) << (EbQtCommand("resolve_dialog")
		<< b->parent()->name()); // tag
}
void EbQtMainWin::tryResolveYesNoDlg()
{
	// sender is hopefully a button
	const QObject * b = QObject::sender();
	(*sock) << (EbQtCommand("resolve_dialog")
		<< b->parent()->name()	// tag
		<< (strcmp(b->name(), "yes") ? "0" : "1")); // yes = 1
}
void EbQtMainWin::tryResolveListDlg()
{
	// sender is hopefully a button
	const QObject * b = QObject::sender();
//	int choice = ((EbQtDlgList *)(b->parent()))->choices->currentItem() + 1;
//	if (choice <= 0) return;  // leave the user with nothing having happened
	(*sock) << (EbQtCommand("resolve_dialog")
		<< b->parent()->name()	// tag
//		// 0 for no selection (_maybe_ what 0's for), in any case
//		// we force the first item selected initially, and then you
//		// can't deselect an item
//		<< QString::number(choice));
		<< ((EbQtDlgList *)(b->parent()))->choices->currentText());
}
void EbQtMainWin::tryResolveTextDlg()
{
	// sender is hopefully a button
	const QObject * b = QObject::sender();
	(*sock) << (EbQtCommand("resolve_dialog")
		<< b->parent()->name()	// tag
		<< ((EbQtDlgText *)(b->parent()))->text());
}

void EbQtMainWin::trySend(EbQtLocalAccount * l, EbQtAccount * a, const QString & msg)
{
	(*sock) << (EbQtCommand("message_send")
		<< a->contact()->group()->name()
		<< a->contact()->name()
		<< l->name()
		<< l->service()->name()
		<< a->name()
		<< msg);
}

void EbQtMainWin::tryGroupSend(const QString & msg, const QString & id)
{
	(*sock) << (EbQtCommand("group_chat_send")
		<< id
		<< msg);
}
void EbQtMainWin::tryGroupInvite(const QString & handle, const QString & id)
{
	(*sock) << (EbQtCommand("group_chat_invite")
		<< id
		<< handle);
}
void EbQtMainWin::tryGroupLeave(const QString & id)
{
	(*sock) << (EbQtCommand("close_group_chat")
		<< id);
}

void EbQtMainWin::tryOpenConv()
{
	tryOpenConv(buddyView->selectedItem());

}
void EbQtMainWin::tryOpenConv(QListViewItem * i)
{
	if (! i) return;
	QObject * o;
	if (! (o = ((EbQtBuddyViewItem *)i)->owner())) return;
	if (o->inherits("EbQtAccount")) {
		tryOpenConv((EbQtAccount *)o);
	} else if (o->inherits("EbQtContact")) {
		tryOpenConv((EbQtContact *)o);
	}
}
void EbQtMainWin::tryOpenConv(EbQtContact * c)
{
	EbQtConv * conv = getConv(c);
	conv->show();
	if (convsInTabs->isOn())
		convsTabWidget->showPage(conv);
}
void EbQtMainWin::tryOpenConv(EbQtAccount * a)
{
	EbQtConv * conv = getConv(a->contact());
	conv->show();
	if (convsInTabs->isOn())
		convsTabWidget->showPage(conv);
	conv->setCurrents(a->localAccount(), a);
}
void EbQtMainWin::tryOpenConv(const QString & u, EbQtConv * conv)
{
	if (! conv) return;
	EbQtLocalAccount * l = conv->localAcct();
	if (u == l->name())
		return;  // the user probably missed.
		// the only thing we might want here is, when we have e.g. an irc
		// server console, it could be another way to open that.
	EbQtAccount * a = l->account(u);
	if (a)
		tryOpenConv(a);
	else
		QMessageBox::warning(this, tr("Not on contact list"),
			tr("To hold a one-to-one conversation with someone,\n"
			   "you must first add them to your contact list."));
}

void EbQtMainWin::trySaveConfig()
{
	(*sock) << "save_config";  // auto-conversion... gotta have it :-)
}

void EbQtMainWin::selectLocalAcct(int row, int col)
{
	EbQtPrefPage * p;
	if (col != 0)
		localAcctsView->blockSignals(TRUE);
		localAcctsView->setCurrentCell(row, 0);
		localAcctsView->blockSignals(FALSE);
	if (! (p = prefPages["accounts"])) {
		qWarning("mainwin: no account prefs page!");
		return;
	}
	// handle _ service
	p->selectAnySubpage(localAcctsView->text(row,3) + '_'
		+ localAcctsView->text(row,4));
}
void EbQtMainWin::selectLocalAcct(EbQtPrefPage * page)
{
	if (! page) return;
	QDictIterator<EbQtService> si(allServices);
	for (; si.current(); ++si) {
		QDictIterator<EbQtLocalAccount> li = (*si)->localAccounts();
		for (; li.current(); ++li) {
			if (li.currentKey()+'_'+si.currentKey() == page->name()) {
				QTableItem * i = localAcctItems[*li];
				if (! i) {
					qWarning("mainwin: local acct %s has no item!",
						li.currentKey().ascii());
					continue;
				}
				localAcctsView->setCurrentCell(i->row(), 0);
				return;
			}
		}
	}
}

bool EbQtMainWin::eventFilter(QObject * o, QEvent * e)
{
	if (o == buddyView->viewport()) {
		if (e->type() == QEvent::MouseButtonDblClick) {
			tryOpenConv(buddyView->itemAt(((QMouseEvent *)e)->pos()));
			return TRUE;  // you can still expand/collapse with the +/- icons
		}
	} else if (o == addrCombo) {
		if (e->type() == QEvent::KeyPress) {
			QKeyEvent * k = (QKeyEvent *) e;
			if (k->key() == Qt::Key_Return) {
				k->accept();
				tryConnect();
				return TRUE;  // hide this from addrCombo
			}
		}
	} else if (o == launchCmdEdit) {
		if (e->type() == QEvent::KeyPress) {
			QKeyEvent * k = (QKeyEvent *) e;
			if (k->key() == Qt::Key_Return) {
				k->accept();
				launchAndConnect();
				return TRUE;  // hide this from launchCmdEdit
			}
		}
	} else if (convsTabWidget && o == convsTabWidget->topLevelWidget()) {
		bool gone = FALSE;
		switch(e->type()) {
			case QEvent::Close:
			if (groupConvsInTabs->isOn()) {
				for (QDictIterator<EbQtConv> gci(groupConvs);
					gci.current();) {
					EbQtConv * gc = gci.current();
					++gci;
					gc->tryClose();  // may delete
				}
			}
			if (convsInTabs->isOn()) {
				for (QPtrDictIterator<EbQtConv> sci(singleConvs);
					sci.current();) {
					EbQtConv * conv = sci.current();
					++sci;
					conv->tryClose();  // may delete
				}
			}
			((QCloseEvent *)e)->ignore();
			gone = TRUE;  // we probably deleted ctw just now
			case QEvent::Move:
			case QEvent::Resize:
			geoChangeTimer->start(1000, TRUE);  // and stop any pending timer
			default: break;
		}
		if (gone) return TRUE;
	} else if (e->type() == QEvent::Close) {
		if (o == addDlg) {
			addDlg->deleteLater();
			addDlg = NULL;
			return TRUE;
		} else if (dialogs[o->name()] == o) {
			// remote dialog: can only be closed by the server
			// (clicking eg OK _should_ prompt the server to do that)
			((QCloseEvent *)e)->ignore();
			return TRUE;
		} else if (o->inherits("QDialog")) {  // addDlg or prefs dialog
			o->deleteLater();
			return TRUE;  // might be gone when event gets there
		}
	}
	return EbQtMainWinUI::eventFilter(o, e);
}

void EbQtMainWin::updateAway()
{
	away->blockSignals(TRUE);
	away->setOn(lastAway);
	away->blockSignals(FALSE);
}
void EbQtMainWin::tryAway(bool)
{
	updateAway();
	// ignore isAway, use dialog
	EbQtAwayDlg * awayDlg = new EbQtAwayDlg(this, "awayDlg");
	awayDlg->setTitle(lastAwayTitle);
	awayDlg->setBody(lastAwayBody);
	if (awayDlg->exec() == QDialog::Accepted) {
		statusBar()->message(tr("Setting away..."), EBQT_STATUSBAR_DELAY);
		(*sock) << (EbQtCommand("set_away")
			<< awayDlg->title()
			<< awayDlg->body());
	} else {
		statusBar()->message(tr("Unsetting away..."), EBQT_STATUSBAR_DELAY);
		(*sock) << "unset_away";
	}
	delete awayDlg;
}

void EbQtMainWin::tryJoinGroup()
{
	EbQtJoinDlg * joinDlg = new EbQtJoinDlg(this, "joinDlg");
	joinDlg->refreshServices(&allServices);
	if (joinDlg->exec() == QDialog::Accepted) {
		statusBar()->message(tr("Joining chat..."), EBQT_STATUSBAR_DELAY);
		(*sock) << (EbQtCommand("join_group_chat")
			<< joinDlg->localCB->currentText()
			<< joinDlg->serviceCB->currentText()
			<< joinDlg->channelEdit->text());
	}
	delete joinDlg;
	joinDlg = NULL;
}

EbQtPrefPage * EbQtMainWin::newPrefPageInDialog(const QString & name)
{
	QDialog * d = new QDialog(this);
	(new QVBoxLayout(d, 3))->setAutoAdd(TRUE);
	d->installEventFilter(this);  // our event filter means close->delete for dialogs
	EbQtPrefPage * p;
	prefPages.insert(name, (p = new EbQtPrefPage(d, name)));
	connect(p, SIGNAL(destroyed(QObject *)), SLOT(removePrefPage(QObject *)));
	connect(p, SIGNAL(wantsRefresh(EbQtPrefPage *)),
		SLOT(refreshPrefPage(EbQtPrefPage *)));
	connect(p, SIGNAL(wantsValueSet(const QString &, const QString &, EbQtPrefPage *)),
		SLOT(trySetPref(const QString &, const QString &, EbQtPrefPage *)));
	connect(p, SIGNAL(wantsSave()), SLOT(trySaveConfig()));
	connect(p, SIGNAL(captionChanged(const QString &, EbQtPrefPage *)),
		// QWidget (->QDialog) is sensible: setCaption is a slot
		d, SLOT(setCaption(const QString &)));
	d->show();
	return p;
}

void EbQtMainWin::refreshAcctPrefs()
{
	refreshPrefPage("accounts");
}
void EbQtMainWin::refreshPrefPage(EbQtPrefPage * page)
{
	refreshPrefPage(page->name());
}
void EbQtMainWin::refreshPrefPage(const QString & page)
{
	(*sock) << (EbQtCommand("list_pref_page")
		<< page);
	// this will clear and refill the page, and when we hear about
	// subpages, those will be refreshed too, recursively
}

void EbQtMainWin::trySetPref(const QString & name, const QString & value,
	EbQtPrefPage * page)
{
	(*sock) << (EbQtCommand("set_pref_value")
		<< page->name()
		<< name
		<< value);
}

void EbQtMainWin::removeLocalAcctItem(QObject * o)
{
	// DON'T do this, it doesn't work in a destroyed() handler
	//if (! o->inherits("EbQtLocalAccount")) return;
	EbQtLocalAccount * l = (EbQtLocalAccount *)o;
	localAcctsView->removeRow(localAcctItems[l]->row());
	localAcctItems.remove(l);
}

void EbQtMainWin::closeEvent(QCloseEvent * ce)
{
	bool signOffFirst = FALSE;
	if (signOff->isEnabled()) {
		// at least one acct is signed on (so socket is connected)
		switch(QMessageBox::information(this, tr("Still signed on"),
			tr("There are some accounts still signed on. If\n"
			"you leave them signed on with no interface\n"
			"connected, they will be set as away, and the\n"
			"server will hold your messages.\n\n"
			"Do you want to sign off first?\n"
			"(cancel to keep EbQt open)"),
			QMessageBox::Yes,
			QMessageBox::No     | QMessageBox::Default,
			QMessageBox::Cancel | QMessageBox::Escape)) {
			case QMessageBox::Yes:  // sign off and close
			signOffFirst = TRUE;  // now fall through to
			case QMessageBox::No:  // close
			break;
			case QMessageBox::Cancel:  // actually don't close
			ce->ignore();
			return;
		}
	}
	sock->blockSignals(TRUE);
	if (signOffFirst)
		trySignOff(TRUE);
	sock->close();
	sock->blockSignals(FALSE);
	killLaunchedCore();
	ce->accept();
	qApp->quit();
}

EbQtConv * EbQtMainWin::getConv(EbQtContact * c)
{
	EbQtConv * conv;
	if (! (conv = singleConvs[c])) {
		if (toggleInvisibility->isOn())
			toggleInvisibility->setOn(FALSE);
		bool showPage = (convsInTabs->isOn() && convsTabWidget == NULL);
		conv = newConv(convsInTabs->isOn(),
			QString("conv %1").arg(c->name()), c);
		singleConvs.insert(c, conv);
		connect(conv, SIGNAL(convClosing(EbQtConv *)),
			SLOT(removeConversation(EbQtConv *)));
		connect(conv, SIGNAL(wantsToSend(EbQtLocalAccount *,
				EbQtAccount *, const QString &)),
			SLOT(trySend(EbQtLocalAccount *, EbQtAccount *,
				const QString &)));
		connect(conv, SIGNAL(wantsIgnore(EbQtContact *)),
			SLOT(tryIgnore(EbQtContact *)));
		conv->show();
		if (showPage)
			convsTabWidget->showPage(conv);
	}
	raiseAndActivate(conv->topLevelWidget());
	return conv;
}

void EbQtMainWin::removeConversation(EbQtConv * conv)
{
	if (! conv) return;
	singleConvs.remove(conv->contact());
	closedConv(conv);
}
void EbQtMainWin::removeGroupConv(EbQtConv * conv)
{
	if (! conv) return;
	groupConvs.remove(conv->id());
	closedConv(conv);
}
void EbQtMainWin::closedConv(EbQtConv * conv)
{
	prevSplitterSizes = topSplitter->sizes();  // for restoreSplitterSizes
	if (convsTabWidget)
		convsTabWidget->removePage(conv);
	delete conv;
	updateConvsTabWidget();
	// we don't want the splitter to drift left here
	// (testing has confirmed that this call is necessary)
	if (convsTabWidget && ! (convsTabWidget->count() == 0 ||
		ctwUndocked->isOn() || ctwRight->isOn()))
		QTimer::singleShot(0, this, SLOT(restoreSplitterSizes()));
}

void EbQtMainWin::tryRename(QListViewItem * item, int col, const QString & newName)
{
	QObject * o;
	QString existingName;
	if (! (o = ((EbQtBuddyViewItem *)item)->owner()) || col != 0) {
		qWarning("mainwin: martian cell renamed!");
		return;
	}
	if (o->inherits("EbQtContact")) {
		EbQtContact * c = (EbQtContact *)o;
		existingName = c->name();
		if (existingName == newName) return;
		statusBar()->message(tr("Renaming contact..."), EBQT_STATUSBAR_DELAY);
		(*sock) << (EbQtCommand("rename_contact")
			<< c->group()->name()
			<< c->name()
			<< newName);
	} else if (o->inherits("EbQtGroup")) {
		EbQtGroup * g = (EbQtGroup *)o;
		existingName = g->name();
		if (existingName == newName) return;
		statusBar()->message(tr("Renaming group..."), EBQT_STATUSBAR_DELAY);
		if (! allGroups[newName])
			tryCreateBuddyList(newName, QString::null, QString::null,
				QString::null, QString::null);
		for (QDictIterator<EbQtContact> ci = g->contacts();
			ci.current(); ++ci) {
			tryMoveContact(newName, *ci);
		}
		// force since tryRemove(g) thinks it's not empty...
		(*sock) << (EbQtCommand("del_group")
			<< g->name());
	} else {
		qWarning("mainwin: martian object renamed!");
		return;
	}
	trySaveConfig();
	// back to original for the moment
	item->setText(0, existingName.isEmpty() ? QString(o->name()) : existingName);
}

void EbQtMainWin::tryMoveContact(const QString & group, EbQtContact * contact)
{
	statusBar()->message(tr("Moving contact..."), EBQT_STATUSBAR_DELAY);
	(*sock) << (EbQtCommand("move_contact")
		<< contact->group()->name()
		<< contact->name()
		<< group);
	trySaveConfig();
}
void EbQtMainWin::tryMoveAccount(EbQtContact * contact, EbQtAccount * account)
{
	statusBar()->message(contact == account->contact()
		? tr("Changing account order...")
		: tr("Moving account..."),
		EBQT_STATUSBAR_DELAY);
	(*sock) << (EbQtCommand("move_account")
		<< account->localAccount()->name()
		<< account->service()->name()
		<< account->name()
		<< contact->group()->name()
		<< contact->name());
	trySaveConfig();
}

void EbQtMainWin::tryIgnore(bool, EbQtContact * c)
{
	if (c) {
		tryIgnore(c);
		return;
	}
	setSelectionActions();  // update button
	EbQtBuddyViewItem * item = (EbQtBuddyViewItem *)(buddyView->selectedItem());
	if (! item) return;
	QObject * tmp = item->owner();
	if (! tmp) return;
	if (! tmp->inherits("EbQtContact")) return;
	// don't use current state, use dialog
	tryIgnore((EbQtContact *)tmp);
}
void EbQtMainWin::tryIgnore(EbQtContact * c)
{
	EbQtIgnoreDlg * ignoreDlg = new EbQtIgnoreDlg(this, "ignoreDlg");
	ignoreDlg->setCaption(QString(tr("Ignore %1")).arg(c->name()));
	ignoreDlg->setResponse(c->ignoreMsg());
	if (ignoreDlg->exec() == QDialog::Accepted) {
		statusBar()->message(tr("Ignoring %1...").arg(c->name()),
			EBQT_STATUSBAR_DELAY);
		(*sock) << (EbQtCommand("ignore_contact")
			<< c->group()->name()
			<< c->name()
			<< ignoreDlg->response());
		trySaveConfig();
	} else {
		if (c->ignored()) {
			statusBar()->message(tr("Unignoring %1...").arg(c->name()),
				EBQT_STATUSBAR_DELAY);
			(*sock) << (EbQtCommand("unignore_contact")
				<< c->group()->name()
				<< c->name());
			trySaveConfig();
		}
	}
	delete ignoreDlg;
}

/*void EbQtMainWin::startSettings(QSettings & settings)
{
	// on windows, uses registry keys HKEY_.../Software/Everybuddy/EbQt/
	// everything else just uses ~/.qt/ebqtrc
	settings.insertSearchPath(QSettings::Windows, "/EveryBuddy");
}*/

void EbQtMainWin::tryChatAction(const QString & action,
	const QString & chat, EbQtLocalAccount *)
{
	(*sock) << (EbQtCommand("perform_action")
		<< "groupchat"
		<< chat
		<< action);
}
void EbQtMainWin::tryChatUserAction(const QString & user,
	const QString & action, const QString & chat) {
	(*sock) << (EbQtCommand("perform_action")
		<< "group_users"
		<< chat
		<< user
		<< action);
}

void EbQtMainWin::setInvisibility(bool on)
{
	static QPoint mainPos, tabsPos;
	static QMap<void *,QPoint> sglPos;
	static QMap<QString,QPoint> grpPos;
	if (on) {
		mainPos = topSplitter->pos();
		topSplitter->hide();
	} else {
		topSplitter->move(mainPos);
		topSplitter->show();
	}
	if (convsTabWidget != NULL && ctwUndocked->isOn()) {
		if (on) {
			tabsPos = convsTabWidget->topLevelWidget()->pos();
			convsTabWidget->topLevelWidget()->hide();
		} else {
			convsTabWidget->topLevelWidget()->move(tabsPos);
			convsTabWidget->topLevelWidget()->show();
		}
	}
	if (! convsInTabs->isOn()) {
		for (QPtrDictIterator<EbQtConv> sci(singleConvs); sci.current(); ++sci) {
			if (on) {
				sglPos.insert(sci.currentKey(), sci.current()->pos());
				sci.current()->hide();
			} else {
				if (sglPos.contains(sci.currentKey()))
					sci.current()->move(sglPos[sci.currentKey()]);
				sci.current()->show();
			}
		}
	}
	if (! groupConvsInTabs->isOn()) {
		for (QDictIterator<EbQtConv> gci(groupConvs); gci.current(); ++gci) {
			if (on) {
				grpPos.insert(gci.currentKey(), gci.current()->pos());
				gci.current()->hide();
			} else {
				if (grpPos.contains(gci.currentKey()))
					gci.current()->move(grpPos[gci.currentKey()]);
				gci.current()->show();
			}
		}
	}
	if (! on) {
		sglPos.clear();
		grpPos.clear();
	}
}
void EbQtMainWin::updateSummary()
{
#ifdef Q_WS_X11
	// tray icon is the entire point of this fn (for now at least)
	if (! trayIcon)
#endif
		return;
	QString summary;
	for (QDictIterator<EbQtGroup> gi(allGroups); (*gi)!=0; ++gi) {
		bool firstOnlineInGroup = TRUE;
		for (QDictIterator<EbQtContact> ci = (*gi)->contacts();
			(*ci)!=0; ++ci) {
			if ((*ci)->isOnline()) {
				summary += summary.isEmpty()
					? "Online contacts:\n"
					: "\n";
				if (firstOnlineInGroup) {
					summary += QString((*gi)->name()) + ":\n";
					firstOnlineInGroup = FALSE;
				}
				summary += QString("  ") + (*ci)->name();
			}
		}
	}
	if (summary.isEmpty())
		summary = "No contacts online";
#ifdef Q_WS_X11
	QToolTip::remove(trayIcon);
	QToolTip::add(trayIcon, summary);
#endif
}
void EbQtMainWin::doTrayMenu(const QPoint & where)
{
	QPopupMenu * p = trayMenu(this);
	connect(p, SIGNAL(aboutToHide()), p, SLOT(deleteLater()));
	p->exec(where);
}

QPopupMenu * EbQtMainWin::trayMenu(QWidget * parent)
{
	QPopupMenu * p = new QPopupMenu(parent, "trayMenu");
	signOn->addTo(p);
	signOff->addTo(p);
	away->addTo(p);
	p->insertSeparator();
	toggleInvisibility->addTo(p);
	fileQuit->addTo(p);
	return p;
}
QPopupMenu * EbQtMainWin::groupMenu(QWidget * parent, EbQtGroup * g)
{
	QPopupMenu * p = new QPopupMenu(parent, "groupMenu");
	bool doneSep;
	if (tabs->currentPage() == listTab && groupItems[g]->isVisible()) {
		p->setWhatsThis(p->insertItem(rename->iconSet(),
			rename->menuText(), g, SLOT(startRename()),
			rename->accel()), rename->whatsThis());
	}
	p->setWhatsThis(p->insertItem(remove->iconSet(), remove->menuText(),
		g, SLOT(remove()), remove->accel()), remove->whatsThis());
	doneSep = FALSE;
	for (unsigned int i=0; i < g->actions.count(); i++) {
		if (! doneSep) {
			p->insertSeparator();
			doneSep = TRUE;
		}
		p->setItemParameter(p->insertItem(g->actions[i],
			g, SLOT(tryGroupAction(int))), i);
	}
	doneSep = FALSE;
	for (QDictIterator<EbQtContact> ci = g->contacts();
		ci.current(); ++ci) {
		if (! doneSep) {
			p->insertSeparator();
			doneSep = TRUE;
		}
		p->insertItem((*ci)->name(), contactMenu(p, *ci, TRUE));
	}
	return p;
}
QPopupMenu * EbQtMainWin::contactMenu(QWidget * parent, EbQtContact * c,
	bool showAccts)
{
	QPopupMenu * p = new QPopupMenu(parent, "contactMenu");
	bool doneSep;
	p->setWhatsThis(p->insertItem(openConv->iconSet(), openConv->menuText(),
		c, SLOT(tryOpenConv()), openConv->accel()),
		openConv->whatsThis());
	int ignoreItem = p->insertItem(ignore->iconSet(), ignore->menuText(),
		c, SLOT(tryToggleIgnore()), ignore->accel());
	p->setWhatsThis(ignoreItem, ignore->whatsThis());
	p->setItemChecked(ignoreItem, c->ignored());
	if (tabs->currentPage() == listTab && contactItems[c]->isVisible()) {
		p->setWhatsThis(p->insertItem(rename->iconSet(), rename->menuText(),
			c, SLOT(startRename()), rename->accel()),
			rename->whatsThis());
	}
	p->setWhatsThis(p->insertItem(remove->iconSet(), remove->menuText(),
		c, SLOT(remove()), remove->accel()), remove->whatsThis());
	doneSep = FALSE;
	for (unsigned int i=0; i < c->actions.count(); i++) {
		if (! doneSep) {
			p->insertSeparator();
			doneSep = TRUE;
		}
		p->setItemParameter(p->insertItem(c->actions[i],
			c, SLOT(tryAction(int))), i);
	}
	if (showAccts) {
		doneSep = FALSE;
		for (QPtrListIterator<EbQtAccount> ai = c->accounts();
			ai.current(); ++ai) {
			if (! doneSep) {
				p->insertSeparator();
				doneSep = TRUE;
			}
			p->insertItem((*ai)->service()->pixmap(16,16,
				(*ai)->status() != EbQtAccount::Online),
				(*ai)->name(),
				accountMenu(p, *ai, FALSE, TRUE));
		}
	}
	return p;
}
QPopupMenu * EbQtMainWin::accountMenu(QWidget * parent, EbQtAccount * a,
	bool showContact, bool showLocal)
{
	QPopupMenu * p = new QPopupMenu(parent, "accountMenu");
	p->setWhatsThis(p->insertItem(openConv->iconSet(), openConv->menuText(),
		a, SLOT(tryOpenConv()), openConv->accel()),
		openConv->whatsThis());
	p->setWhatsThis(p->insertItem(makePreferred->iconSet(),
		makePreferred->menuText(), a, SLOT(makePreferred()),
		makePreferred->accel()), makePreferred->whatsThis());
	p->setWhatsThis(p->insertItem(remove->iconSet(), remove->menuText(),
		a, SLOT(remove()), remove->accel()), remove->whatsThis());
	QStringList actions = a->service()->buddyActions;
	if (a->actions.count() > 0) {
		p->insertSeparator();
		for (unsigned int i=0; i < a->actions.count(); i++) {
			p->setItemParameter(p->insertItem(a->actions[i],
				a, SLOT(tryGenericAction(int))), i);
		}
	}
	if (actions.count() > 0) {
		p->insertSeparator();
		for (unsigned int i=0; i < actions.count(); i++) {
			p->setItemParameter(
				p->insertItem(actions[i], a,
					SLOT(tryServiceAction(int))), i);
		}
	}
	if (showContact) {
		p->insertSeparator();
		p->insertItem(a->contact()->name(),
			contactMenu(p, a->contact(), FALSE));
	}
	if (showLocal) {
		p->insertSeparator();
		p->insertItem(a->localAccount()->service()->pixmap(16,16,
			! a->localAccount()->ready()),
			a->localAccount()->name(),
			localAccountMenu(p, a->localAccount(), FALSE));
	}
	return p;
}
QPopupMenu * EbQtMainWin::localAccountMenu(QWidget * parent,
	EbQtLocalAccount * l, bool showAccts)
{
	QPopupMenu * p = new QPopupMenu(parent, "localAccountMenu");
	for(unsigned int i=0; i < l->service()->states.count(); i++) {
		int id = p->insertItem(l->service()->states[i],
			l, SLOT(changeStatus(int)));
		p->setItemParameter(id, i);
		p->setItemChecked(id,
			l->statusMsg() == l->service()->states[i]);
	}
	if (tabs->currentPage() == localAcctsTab) {
		p->insertSeparator();
		p->setWhatsThis(p->insertItem(remove->iconSet(), remove->menuText(),
			l, SLOT(remove()), remove->accel()), remove->whatsThis());
	}
	if (showAccts) {
		bool doneSep = FALSE;
		for (QDictIterator<EbQtAccount> ai = l->accounts();
			ai.current(); ++ai) {
			if (! doneSep) {
				p->insertSeparator();
				doneSep = TRUE;
			}
			p->insertItem((*ai)->service()->pixmap(16,16,
				(*ai)->status() != EbQtAccount::Online),
				(*ai)->name(),
				accountMenu(p, *ai, TRUE, FALSE));
		}
	}
	return p;
}

void EbQtMainWin::doChatUserMenu(const QString & user, int userId,
	const QPoint & pos, EbQtConv * conv)
{
	EbQtLocalAccount * l = conv->localAcct();
	QStringList actions = l->service()->groupUserActions;
	EbQtAccount * a = l->account(user);
	QPopupMenu * p;
	QString lShortName = l->name();
	int atPos = lShortName.findRev('@');
	if (atPos > 0) lShortName.truncate(atPos);
	if (user == l->name() || user == lShortName) {
		p = localAccountMenu(conv, l, TRUE);
	} else {
		if (a == NULL) {
			p = new QPopupMenu(conv, "unknownAcctMenu");
			int addItem = p->insertItem(add->iconSet(), add->menuText(),
				conv, SLOT(tryAddFromGroup(int)), add->accel());
			p->setWhatsThis(addItem, add->whatsThis());
			p->setItemParameter(addItem, userId);
		} else {
			p = accountMenu(conv, a, TRUE, FALSE);
		}
		if (actions.count() > 0) {
			p->insertSeparator(0);
			for(unsigned int ai=0; ai < actions.count(); ai++) {
				p->setItemParameter(p->insertItem(actions[ai],
					conv, SLOT(tryChatUserAction(int)), 0, ai),
					// now, this is fun: we only get to pass one
					// int, but it has to indicate which action
					// to perform _AND_ which user (unknown user,
					// so no object to route through with a
					// slot-signal bounce)... the answer?
					// multiply, of course!
					// (num of possible actions is const)
					userId * actions.count() + ai);
			}
		}
	}
	connect(p, SIGNAL(aboutToHide()), p, SLOT(deleteLater()));
	p->exec(pos);
}

EbQtConv * EbQtMainWin::newConv(bool useTabs, const QString & name,
	EbQtContact * c)
{
	EbQtConv * conv;
	conv = new EbQtConv(this, name, c, Qt::WType_TopLevel);
	connect(conv, SIGNAL(wantsGeometrySave(EbQtConv *)),
		SLOT(putConvGeometry(EbQtConv *)));
	if (icon())  // give conv icon of mainwin
		conv->setIcon(QPixmap(*(icon())));
	if (useTabs) {
		addConvTab(conv);
	} else {
		conv->resize(conv->minimumSizeHint());
		conv->show();
		if (c)
			tryGetContactPref("geometry/conversation", c);
	}
	return conv;
}

QWidget * EbQtMainWin::newConvsTLW()
{
	QWidget * top = new QWidget(this, "convsWidget", Qt::WType_TopLevel);
	top->setCaption(tr("Conversations"));
	if (! tabsGeometry.isEmpty())
		top->setGeometry(tabsGeometry);
	top->installEventFilter(this);
	return top;
}

QTabWidget * EbQtMainWin::getConvsTabWidget()
{
	if (convsTabWidget == NULL) {
		if (ctwUndocked->isOn()) {
			QWidget * top = newConvsTLW();
			convsTabWidget = new QTabWidget(top, "convsTabWidget");
			convsTabWidget->setSizePolicy(QSizePolicy(
				QSizePolicy::MinimumExpanding,
				QSizePolicy::MinimumExpanding));
			(new QVBoxLayout(top))->addWidget(convsTabWidget);
			top->show();
		} else {
			convsTabWidget = new QTabWidget(topSplitter, "convsTabWidget");
			convsTabWidget->setSizePolicy(QSizePolicy(
				QSizePolicy::MinimumExpanding,
				QSizePolicy::MinimumExpanding));
			topSplitter->setResizeMode(convsTabWidget, QSplitter::Stretch);
			if (ctwRight->isOn())
				topSplitter->moveToLast(convsTabWidget);
			else
				topSplitter->moveToFirst(convsTabWidget);
				// addConvTab queues moveBackLeft
			convsTabWidget->setFocusPolicy(NoFocus);
			convsTabWidget->show();
		}
	}
	return convsTabWidget;
}

void EbQtMainWin::restoreRHSpos()
{
	int nx = topSplitter->geometry().left() -
		(topSplitter->geometry().right() - prevRHSpos);
	int min = topSplitter->geometry().left() - topSplitter->frameGeometry().left();
	if (nx < min) nx = min;
	topSplitter->setGeometry(QRect(QPoint(nx, topSplitter->geometry().top()),
		topSplitter->size()));
}

void EbQtMainWin::restoreSplitterSizes()
{
	topSplitter->setSizes(prevSplitterSizes);
}

void EbQtMainWin::updateConvsTabWidget()
{
	if (convsTabWidget) {
		if (convsTabWidget->count() == 0) {
			if (ctwUndocked->isOn())
				delete convsTabWidget->parentWidget();
			else {
				QRect mainGeo = QRect(mapToGlobal(QPoint(0,0)), size());
				delete convsTabWidget;
				topSplitter->setMinimumSize(topSplitter->minimumSizeHint());
				topSplitter->setGeometry(mainGeo);
			}
			convsTabWidget = NULL;
		} else if (convsTabWidget->count() == 1) {
			convsTabWidget->showPage(convsTabWidget->page(0));
		}
	}
	dockCTW->setEnabled(convsInTabs->isOn() || groupConvsInTabs->isOn());
}
void EbQtMainWin::toggleConvsInTabs(bool on, bool sendToServer)
{
	if (sendToServer)
		putGeneralPref("conversations_in_tabs", (on ? "1" : "0"));
	QPtrDictIterator<EbQtConv> it(singleConvs);
	for (it.toFirst(); it.current(); ++it)
		it.current()->hide();
	for (it.toFirst(); it.current(); ++it) {
		if (on)	addConvTab(it.current());
		else	removeConvTab(it.current());
	}
	if (! on)
		doCascadeConvs();
	// and now we might not need ctw anymore
	updateConvsTabWidget();
}
void EbQtMainWin::toggleGroupConvsInTabs(bool on, bool sendToServer)
{
	if (sendToServer)
		putGeneralPref("groupchats_in_tabs", (on ? "1" : "0"));
	QDictIterator<EbQtConv> it(groupConvs);
	for (it.toFirst(); it.current(); ++it)
		it.current()->hide();
	for (it.toFirst(); it.current(); ++it) {
		if (on)	addConvTab(it.current());
		else	removeConvTab(it.current());
	}
	if (! on)
		doCascadeGroupConvs();
	// and now we might not need ctw anymore
	updateConvsTabWidget();
}
void EbQtMainWin::doCascadeConvs()
{
	QPtrDictIterator<EbQtConv> it(singleConvs);
	if (it.count() == 0 || convsInTabs->isOn()) return;
	QRect allWins;
	for (it.toFirst(); it.current(); ++it) {
		allWins = allWins.isNull() ? it.current()->frameGeometry()
			: allWins.unite(it.current()->frameGeometry());
	}
	QSize s = it.toFirst()->frameGeometry().size();
	QRect r = cascadeRect(it.count(),
		QRect(allWins.center() - QPoint(s.width()/2, s.height()/2 - 1), s));
	for (it.toFirst(); it.current(); ++it) {
		it.current()->show();
		it.current()->move(r.topLeft());  // inc frame
		it.current()->resize(r.size() -  // exc frame
			// measure main frame since this probably
			// doesn't have one yet
			(frameSize() - size()));
		it.current()->raise();
		r.moveBy(EBQT_CASCADE_GAP_X,EBQT_CASCADE_GAP_Y);
	}
}
void EbQtMainWin::doCascadeGroupConvs()
{
	// I know, I know, it's identical to the above...
	// unfortunately QDictIterator and QPtrDictIterator have no common
	// ancestor. I would describe that as a Qt bug. :-/
	QDictIterator<EbQtConv> it(groupConvs);
	if (it.count() == 0 || groupConvsInTabs->isOn()) return;
	QRect allWins;
	for (it.toFirst(); it.current(); ++it) {
		allWins = allWins.isNull() ? it.current()->frameGeometry()
			: allWins.unite(it.current()->frameGeometry());
	}
	QSize s = it.toFirst()->size();
	QRect r = cascadeRect(it.count(),
		QRect(allWins.center() - QPoint(s.width()/2, s.height()/2), s));
	for (it.toFirst(); it.current(); ++it) {
		it.current()->show();
		it.current()->move(r.topLeft());  // inc frame
		it.current()->resize(r.size() -  // exc frame
			// measure main frame since this probably
			// doesn't have one yet
			(frameSize() - size()));
		r.moveBy(EBQT_CASCADE_GAP_X,EBQT_CASCADE_GAP_Y);
	}
}
QRect EbQtMainWin::cascadeRect(unsigned int count, const QRect & centerRect)
{
	// a size really: the spread of all the new window (top-left) positions
	QPoint span = QPoint(EBQT_CASCADE_GAP_X,EBQT_CASCADE_GAP_Y)
		* ((int)count - 1);
	// _try_ to center them on where ctw was
	QPoint p = centerRect.topLeft() - (span/2);
	QRect desktop = QApplication::desktop()->geometry();
	// furthest down and right the cascade can start:
	QPoint max = desktop.bottomRight() - span;
	max.rx() -= centerRect.width();  // prefer to try with original size
	max.ry() -= centerRect.height();
	if (p.x() > max.x())	p.setX(max.x());  // less important
	if (p.y() > max.y())	p.setY(max.y());  // and may be impossible (p goes -ve)
	if (p.x() < desktop.left())	p.setX(desktop.left()); // more important so 2nd
	if (p.y() < desktop.top())	p.setY(desktop.top());
	// now we know where it'll start, resize all so bottom-right win will fit
	// requires a QLayout to interpret (0,0) as minimumSizeHint()
	QPoint q = p + span;  // top-left of bottom-right win
	return QRect(p, centerRect.size().boundedTo(QSize(
		q.x() > desktop.right() ? 0 : desktop.right() - q.x(),
		q.y() > desktop.bottom() ? 0 : desktop.bottom() - q.y())));
}
void EbQtMainWin::addConvTab(EbQtConv * conv)
{
	prevRHSpos = topSplitter->geometry().right();  // for restoreRHSpos
	conv->reparent(getConvsTabWidget()->parentWidget(),
		0, QPoint(), FALSE);
	convsTabWidget->topLevelWidget()->setTabOrder(conv, convsTabWidget);
	bool firstTab = (convsTabWidget->count() == 0);
	convsTabWidget->addTab(conv, icon() ? *(conv->icon()) : QIconSet(),
		conv->caption());
	connect(conv, SIGNAL(captionChanged(const QPixmap &, const QString &, QWidget *)),
		SLOT(setConvsTabCaption(const QPixmap &, const QString &, QWidget *)));
	if (firstTab)
		conv->show();
	if (! (ctwUndocked->isOn() || ctwRight->isOn()))
		QTimer::singleShot(0, this, SLOT(restoreRHSpos()));
}
void EbQtMainWin::removeConvTab(EbQtConv * conv)
{
	disconnect(conv, SIGNAL(captionChanged(const QPixmap &, const QString &,
		QWidget *)),
		this, SLOT(setConvsTabCaption(const QPixmap &, const QString &,
		QWidget *)));
	if (convsTabWidget)
		convsTabWidget->removePage(conv);
	conv->reparent(this, Qt::WType_TopLevel,
		convsTabWidget->topLevelWidget()->pos(), TRUE);
	// after that reparenting, conv seems to need reminding of a few things...
	conv->setIcon(conv->normalIcon());  // like what its icon is...
	conv->setMinimumSize(conv->minimumSizeHint());  // what size it must be...
	conv->focusSendEdit();  // ...and where its towel^H^H^H^H^H focus is.
}
void EbQtMainWin::setConvsTabCaption(const QPixmap & pix, const QString & caption,
	QWidget * conv)
{
	if (! convsTabWidget)
		return;
	convsTabWidget->changeTab(conv, pix,
		QString(caption).replace(QRegExp("&"), "&&"));  // 3.0 only takes regexp
}

void EbQtMainWin::raiseAndActivate(QWidget * w)
{
	if (w->isMinimized())
		w->showNormal();  // FIXME: "Nothing happens."
	w->raise();
//	w->setActiveWindow();
}

void EbQtMainWin::tryGeneralAction(int which)
{
	(*sock) << (EbQtCommand("perform_action")
		<< "general"
		<< generalActions[which]);
}
void EbQtMainWin::tryGroupAction(int which, EbQtGroup * g)
{
	(*sock) << (EbQtCommand("perform_action")
		<< "group"
		<< g->actions[which]
		<< g->name());
}
void EbQtMainWin::tryContactAction(int which, EbQtContact * c)
{
	(*sock) << (EbQtCommand("perform_action")
		<< "contact"
		<< c->actions[which]
		<< c->group()->name()
		<< c->name());
}
void EbQtMainWin::tryAcctServiceAction(int which, EbQtAccount * a)
{
	(*sock) << (EbQtCommand("perform_action")
		<< "buddy"
		<< a->service()->buddyActions[which]
		<< a->localAccount()->name()
		<< a->service()->name()
		<< a->name());
}
void EbQtMainWin::tryAcctGenericAction(int which, EbQtAccount * a)
{
	(*sock) << (EbQtCommand("perform_action")
		<< "buddy"
		<< a->actions[which]
		<< a->localAccount()->name()
		<< a->service()->name()
		<< a->name());
}

void EbQtMainWin::getInitialPrefs()
{
	tryGetGeneralPref("show_toolbar");
	tryGetGeneralPref("show_statusbar");
	tryGetGeneralPref("show_offline_contacts");
	tryGetGeneralPref("show_contacts_expanded");
	tryGetGeneralPref("conversations_in_tabs");
	tryGetGeneralPref("groupchats_in_tabs");
	tryGetGeneralPref("dock_conversation_tabs");
	tryGetGeneralPref("geometry/main");
	tryGetGeneralPref("geometry/conversation_tabs");
}
void EbQtMainWin::gotGeneralPref(const QString & key, const QString & value)
{
	if (key == "ebqt/geometry/main") {
		QRect g = stringToRect(value);
		if (! g.isNull()) {
			mainGeometry = g;
			topSplitter->setGeometry(g);
		}
	} else if (key == "ebqt/geometry/conversation_tabs") {
		QRect g = stringToRect(value);
		if (! g.isNull()) {
			tabsGeometry = g;
			if (convsTabWidget != NULL && ctwUndocked->isOn())
				convsTabWidget->topLevelWidget()->setGeometry(g);
		}
	} else if (key == "ebqt/show_toolbar") {
		showToolBar->blockSignals(TRUE);
		showToolBar->setOn(value.toUInt() > 0);
		showToolBar->blockSignals(FALSE);
		toggleToolBar(showToolBar->isOn(), FALSE);
	} else if (key == "ebqt/show_statusbar") {
		showStatusBar->blockSignals(TRUE);
		showStatusBar->setOn(value.toUInt() > 0);
		showStatusBar->blockSignals(FALSE);
		toggleStatusBar(showStatusBar->isOn(), FALSE);
	} else if (key == "ebqt/conversations_in_tabs") {
		convsInTabs->blockSignals(TRUE);
		convsInTabs->setOn(value.toUInt() > 0);
		convsInTabs->blockSignals(FALSE);
		toggleConvsInTabs(convsInTabs->isOn(), FALSE);
	} else if (key == "ebqt/groupchats_in_tabs") {
		groupConvsInTabs->blockSignals(TRUE);
		groupConvsInTabs->setOn(value.toUInt() > 0);
		groupConvsInTabs->blockSignals(FALSE);
		toggleGroupConvsInTabs(groupConvsInTabs->isOn(), FALSE);
	} else if (key == "ebqt/dock_conversation_tabs") {
		if (value == "none") {
			ctwUndocked->setOn(TRUE);
		} else if (value == "left") {
			ctwLeft->setOn(TRUE);
		} else if (value == "right") {
			ctwRight->setOn(TRUE);
		}
	} else if (key == "ebqt/show_offline_contacts") {
		showOfflineContacts = value.toUInt() > 0;
		changeWho();
	} else if (key == "ebqt/show_contacts_expanded") {
		showContactsExpanded = value.toUInt() > 0;
		changeHow();
	} else {
		qWarning("mainwin: return_data for martian general pref!");
	}
}
void EbQtMainWin::tryGetGeneralPref(const QString & key)
{
	(*sock) << (EbQtCommand("get_data")
		<< QString("ebqt/%1").arg(key)
		<< "general");
}
void EbQtMainWin::putGeneralPref(const QString & key, const QString & value)
{
	(*sock) << (EbQtCommand("put_data")
		<< QString("ebqt/%1").arg(key)
		<< value
		<< "general");
	trySaveConfig();
}
void EbQtMainWin::gotContactPref(const QString & key, const QString & value,
	EbQtContact * c)
{
	if (key == "ebqt/geometry/conversation") {
		if (convsInTabs->isOn()) return;
		QRect g = stringToRect(value);
		if (! g.isNull()) {
			EbQtConv * conv;
			if ((conv = singleConvs[c]))
				conv->setGeometry(g);
		}
	} else {
		qWarning("mainwin: return_data for martian contact pref!");
	}
}
void EbQtMainWin::tryGetContactPref(const QString & key, EbQtContact * c)
{
	(*sock) << (EbQtCommand("get_data")
		<< QString("ebqt/%1").arg(key)
		<< "contact"
		<< c->group()->name()
		<< c->name());
}
void EbQtMainWin::putContactPref(const QString & key, const QString & value,
	EbQtContact * c)
{
	(*sock) << (EbQtCommand("put_data")
		<< QString("ebqt/%1").arg(key)
		<< value
		<< "contact"
		<< c->group()->name()
		<< c->name());
}

QRect EbQtMainWin::stringToRect(const QString & whxy)
{
	QRegExp geoRE("([0-9]+)x([0-9]+)([+-][0-9]+)([+-][0-9]+)");
	return geoRE.exactMatch(whxy)
		? QRect(geoRE.cap(3).toInt(),geoRE.cap(4).toInt(),
			geoRE.cap(1).toUInt(),geoRE.cap(2).toUInt())
		: QRect();  // null
}
QString EbQtMainWin::rectToString(const QRect & r)
{
	return QString::number(r.width()) + 'x' + QString::number(r.height())
		+ (x() < 0 ? "" : "+") + QString::number(r.x())
		+ (y() < 0 ? "" : "+") + QString::number(r.y());
}

void EbQtMainWin::moveEvent(QMoveEvent *)
{
	geoChangeTimer->start(1000, TRUE);  // and stop any pending timer
}
void EbQtMainWin::resizeEvent(QResizeEvent *)
{
	geoChangeTimer->start(1000, TRUE);  // and stop any pending timer
}

void EbQtMainWin::putGeometries()
{
	if (sock->state() != QSocket::Connected)
		return;
	bool changed = FALSE;
	if (! mainGeometry.isNull() && mainGeometry != geometry()) {
		mainGeometry = geometry();
		putGeneralPref("geometry/main", rectToString(mainGeometry));
		changed = TRUE;
	}
	// above/below here inconsistent, I know: main geometry is always
	// set _after_ show(), tabs geometry is always set _before_...
	// (see the Qt docs page on window geometry and X11 weirdness)
	// update: this below is weird. I know. such is x11.
	if (convsTabWidget != NULL && ctwUndocked->isOn()) {
		QWidget * w = convsTabWidget->topLevelWidget();
		QRect current(w->frameGeometry().topLeft(), w->geometry().size());
		if (tabsGeometry != current) {
			tabsGeometry = current;
			putGeneralPref("geometry/conversation_tabs",
				rectToString(tabsGeometry));
			changed = TRUE;
		}
	}
	if (changed)
		trySaveConfig();
}
void EbQtMainWin::putConvGeometry(EbQtConv * conv)
{
	if (sock->state() != QSocket::Connected)
		return;  // shouldn't happen, convs should be closed
	if (conv->contact() == NULL)
		return;  // this is only for single convs (can't save data on a group)
	if (convsInTabs->isOn())
		return;  // for tabs we save the geometry of the tabs
	putContactPref("geometry/conversation", rectToString(
		QRect(conv->frameGeometry().topLeft(), conv->geometry().size())),
		conv->contact());
}

void EbQtMainWin::notifyContactStatus(EbQtAccount::AccountStatus old, EbQtContact * c)
{
	if (tabs->currentPageIndex() != 0 ||
		// must go from offline to online or vice versa to be shown
		(old == EbQtAccount::Offline) == (c->status() == EbQtAccount::Offline))
		return;
	statusBar()->message(tr("%1 (%3) is now %2")
		.arg(c->name()).arg(QString(c->statusMsg())
		.replace(QRegExp("^\\((.*)\\)$"), "\\1")).arg(c->group()->name()),
		EBQT_STATUSBAR_DELAY);
}

void EbQtMainWin::setConvsDocked(bool right, bool on, bool sendToServer)
{
	if (! on) return;
	if (sendToServer)
		putGeneralPref("dock_conversation_tabs", right ? "right" : "left");
	if (! convsTabWidget || ! topSplitter)
		return;
	if (convsTabWidget->parentWidget() != topSplitter) {
		QRect mg = QRect(mapToGlobal(QPoint(0,0)), size());
		QRect tg = QRect(convsTabWidget->mapToGlobal(QPoint(0,0)),
			convsTabWidget->size());
		QPoint c = mg.unite(tg).center();
		int w = mg.width() + tg.width();
		int h = (mg.height() + tg.height()) / 2;
		QRect ng = QRect(c.x() - w/2, c.y() - h/2, w, h);
		QWidget * top = convsTabWidget->parentWidget();
		convsTabWidget->reparent(topSplitter, QPoint(), TRUE);
		topSplitter->setResizeMode(convsTabWidget, QSplitter::Stretch);
		delete top;
		topSplitter->setGeometry(ng);
	}
	if (right)
		topSplitter->moveToLast(convsTabWidget);
	else
		topSplitter->moveToFirst(convsTabWidget);
	convsTabWidget->show();
	// docs: "you should not need to call this function"...
	// ...but if I don't, the widgets don't swap, until I drag the divide.
	topSplitter->refresh();
}

void EbQtMainWin::setConvsUndocked(bool on, bool sendToServer)
{
	if (! on) return;
	if (sendToServer)
		putGeneralPref("dock_conversation_tabs", "none");
	if (! convsTabWidget || convsTabWidget->parentWidget() != topSplitter)
		return;
	QRect ctwGeo = QRect(convsTabWidget->mapToGlobal(QPoint(0,0)) -
		(topSplitter->geometry().topLeft() - topSplitter->frameGeometry().topLeft()),
		convsTabWidget->size());
	QRect mainGeo = QRect(mapToGlobal(QPoint(0,0)), size());
	QWidget * top = newConvsTLW();
	const QPixmap * i = convsTabWidget->icon();
	convsTabWidget->reparent(top, QPoint(), TRUE);
	(new QVBoxLayout(top))->addWidget(convsTabWidget);
	// after that reparenting, ctw seems to need reminding of a few things...
	// like what its icon is...
	if (i) convsTabWidget->setIcon(*i);
	// what size it must be...
	convsTabWidget->setMinimumSize(convsTabWidget->minimumSizeHint());
	top->setGeometry(ctwGeo);
	// (that goes for us too, while we're at it)
	topSplitter->setMinimumSize(topSplitter->minimumSizeHint());
	topSplitter->setGeometry(mainGeo);
	// ...and where its towel^H^H^H^H^H focus is: TODO?
	top->show();
}

