/* File     : windows.h
   Desc     : Graphical classes
   Authors  : Karine Proot (kproot@nerim.net)                              */

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef WINDOWS_H
#define WINDOWS_H

#include <curses.h>
#include <string>
#include "group.h"
using namespace std;

void finish(int sig);

class AccountsWin
{
 public:
  AccountsWin();
  int getFirst() const;
  int getHighlight() const;
  LocalAccount *getLocalAccountHighlighted() const;
  void downHighlight();
  void upHighlight();
  void resetHighlight();
  void highlightLocalAccount(LocalAccount *ca);
  void setLength(int length);
 private:
  int first;
  int highlight;
  LocalAccount *localAccountHighlighted;
  int length;
};

class ContactsWin
{
 public:
  ContactsWin();
  int getFirst() const;
  int getHighlight() const;
  Contact *getContactHighlighted() const;
  ContactAccount *getContactAccountHighlighted() const;
  void downHighlight();
  void upHighlight();
  void resetHighlight();
  void highlightContact(Contact *c);
  void highlightContactAccount(ContactAccount *ca);
  void setLength(int length);
 private:
  int first;
  int highlight;
  Contact *contactHighlighted;
  ContactAccount *contactAccountHighlighted;
  int length;
};

class Tab
{
 public:
  Tab(WINDOW *currentWin, char *name);
  Tab(WINDOW *currentWin, char *name, int type);
  Tab(WINDOW *currentWin, char *name, char *shortName);
  ~Tab();
  Tab *getPrevious() const;
  Tab *getNext() const;
  char *getName() const;
  const char *getShortName() const;
  int getActivity() const;
  int getType() const;
  void setPrevious(Tab *t);
  void setNext(Tab *t);
  void setCurrentWin(WINDOW *currentWin);
  void setActivity(int activity);
 private:
  Tab *previous;
  Tab *next;
  WINDOW *currentWin;
  char *name;
  char shortName[6];
  int activity;
  int type;
};

class ChatTab: public Tab
{
 protected:
  ChatTab(WINDOW *currentWin, char *name, int type);
 public:
  string getBuffer() const;
  void addToBuffer(char c);
  void addToBuffer(char* c);
  void addToBuffer(string s);
 private:
  string buffer;
};

class BuddyTab: public ChatTab
{
 public:
  BuddyTab(WINDOW *currentWin, Contact *contact, ContactAccount *contactAccount);
  Contact *getContact() const;
  ContactAccount *getContactAccount() const;
  void setContactAccount(ContactAccount *ca);
 private:
  Contact *contact;
  ContactAccount *contactAccount;
};

class GroupTab: public ChatTab
{
 public:
  GroupTab(WINDOW *currentWin, LocalAccount *localAccount, int chatId, char *name);
  LocalAccount *getLocalAccount() const;
  int getChatId() const;
 private:
  LocalAccount *localAccount;
  int chatId;
};

class TabWinList
{
 public:
  TabWinList();
  ~TabWinList();
  int getNumber() const;
  WINDOW *getTabsWin() const;
  WINDOW *getWindow(int index) const;
  Tab *getTab(int index) const;
  int getIndex(Tab *t) const;
  void setTab(int index, Tab *t);
  void removeTab(Tab *t, Tab *t2);
 private:
  int number;
  WINDOW *tabsWIN;
  WINDOW **tabWins;
  Tab **tabs;
};

class TabList
{
 public:
  TabList();
  ~TabList();
  Tab *getFirst() const;
  Tab *getCurrent() const;
  int getNumber() const;
  int getNumber(Tab *t) const;
  Tab *getTab(int index) const;
  BuddyTab *getBuddyTab(Contact *c) const;
  BuddyTab *getBuddyTab(ContactAccount *ca) const;
  GroupTab *getGroupTab(int chatId) const;
  Contact *getCurrentContact() const;
  ContactAccount *getCurrentContactAccount() const;
  int getCurrentChatId() const;
  void setCurrent(Tab *t);
  void setNext();
  void setPrevious();
  void leave(Tab *t) const;
  void enter(Tab *t);
  void add(char *name, char *shortName);
  void addBuddy(Contact *c);
  void addBuddy(Contact *c, ContactAccount *ca);
  void addGroup(LocalAccount *la, int chatId, char *name);
  void remove(Tab *t);
  void removeCurrent();
  void refresh() const;
 private:
  TabWinList *tabWinList;
  Tab *firstTab;
  Tab *currentTab;
};

class MainWin
{
 public:
  MainWin();
  ~MainWin();
  AccountsWin *getAccountsWin() const;
  ContactsWin *getContactsWin() const;
  void title(char *name) const;
  void showHelp() const;
  void showAccounts(LocalAccountList *localAccounts) const;
  void showContacts(GroupList *groups, bool online) const;
  void showTab(string chatBuffer) const;
  void inputGroupchat() const;
  void addLine(int color, char *message) const;
  void addLine(int color, string message) const;
  void addChar(char ch) const;
  void eraseChar() const;
  void clearSubLine() const;
  void drawSubline(char *buffer) const;
  void refresh() const;
 private:
  WINDOW *mainWIN;
  WINDOW *chatWIN;
  WINDOW *sublineWIN;
  WINDOW *groupWIN;
  AccountsWin *accountsWin;
  ContactsWin *contactsWin;
};

class EbncGUI
{
 public:
  EbncGUI(ServiceList *services, GroupList *groups, LocalAccountList *localAccounts);
  ~EbncGUI();
  void readInput();
  void updateAccounts() const;
  void updateContacts() const;
  void showBuddyStatus(ContactAccount *ca, int state, char *status) const;
  void receiveMessage(ContactAccount *ca, char *message) const;
  void sendMessage(ContactAccount *ca, char *message) const;
  void openGroupchat(LocalAccount *la, int chatId, char *name);
  void receiveGroupMessage(int chatId, char *handle, char *message) const;
  void receiveGroup3rd(int chatId, char *message) const;
  void sendGroupMessage(int chatId, char *message) const;
  void joinedGroup(int chatId, char *handle) const;
  void leftGroup(int chatId, char *handle) const;
  void closeGroup(int chatId);
  void draw() const;
 private:
  TabList *tabsWin;
  MainWin *mainWin;
  int currentWin;
  bool sublineMode;
  bool showOnlineOnly;
  char *buffer;
  int currentChar;
  ServiceList *services;
  GroupList *groups;
  LocalAccountList *localAccounts;
};

#endif // WINDOWS_H
