/* File     : windows.cpp
   Desc     : Graphical classes
   Authors  : Karine Proot (kproot@nerim.net)                              */

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <signal.h>
#include <errno.h>
#include <curses.h>
#include <string>
#include <iostream>
#include "GUIcomms.h"
#include "windows.h"

void finish(int sig)
{
  endwin();
  curs_set(1);          /* visible cursor */
  nl();
  echo();
  
  exit(sig);
}

/*** AccountsWin ***/

AccountsWin::AccountsWin()
{
  first = 1;
  highlight = 1;
  localAccountHighlighted = NULL;
  length = 0;
}

int AccountsWin::getFirst() const
{
  return first;
}

int AccountsWin::getHighlight() const
{
  return highlight;
}

LocalAccount *AccountsWin::getLocalAccountHighlighted() const
{
  return localAccountHighlighted;
}

void AccountsWin::upHighlight()
{
  highlight--;
  if (highlight < 1)
  {
    highlight = length;
    first = length - LINES + 6;
    if (first < 1) first = 1;
  }
  else if (highlight < first)
    first--;
}

void AccountsWin::downHighlight()
{
  highlight++;
  if (highlight > length)
  {
    highlight = 1;
    first = 1;
  }
  else if (highlight > first + LINES - 6)
    first++;
}

void AccountsWin::resetHighlight()
{
  highlight = 1;
}

void AccountsWin::highlightLocalAccount(LocalAccount *la)
{
  localAccountHighlighted = la;
}

void AccountsWin::setLength(int length)
{
  this->length = length;
}

/*** ContactsWin ***/

ContactsWin::ContactsWin()
{
  first = 1;
  highlight = 1;
  contactHighlighted = NULL;
  contactAccountHighlighted = NULL;
  length = 0;
}

int ContactsWin::getFirst() const
{
  return first;
}

int ContactsWin::getHighlight() const
{
  return highlight;
}

Contact *ContactsWin::getContactHighlighted() const
{
  return contactHighlighted;
}

ContactAccount *ContactsWin::getContactAccountHighlighted() const
{
  return contactAccountHighlighted;
}

void ContactsWin::upHighlight()
{
  highlight--;
  if (highlight < 1)
  {
    highlight = length;
    first = length - LINES + 6;
    if (first < 1) first = 1;
  }
  else if (highlight < first)
    first--;
}

void ContactsWin::downHighlight()
{
  highlight++;
  if (highlight > length)
  {
    highlight = 1;
    first = 1;
  }
  else if (highlight > first + LINES - 6)
    first++;
}

void ContactsWin::resetHighlight()
{
  highlight = 1;
}

void ContactsWin::highlightContact(Contact *c)
{
  contactHighlighted = c;
}

void ContactsWin::highlightContactAccount(ContactAccount *ca)
{
  contactAccountHighlighted = ca;
}

void ContactsWin::setLength(int length)
{
  this->length = length;
}

/*** Tab ***/

Tab::Tab(WINDOW *currentWin, char *name)
{
  this->currentWin = currentWin;
  this->name = name;
  strncpy(shortName, name, 5);
  shortName[5] = '\0';
  activity = 0;
  type = 0;
}

Tab::Tab(WINDOW *currentWin, char *name, int type)
{
  this->currentWin = currentWin;
  this->name = name;
  strncpy(shortName, name, 5);
  shortName[5] = '\0';
  activity = 0;
  this->type = type;
}

Tab::Tab(WINDOW *currentWin, char *name, char *shortName)
{
  this->currentWin = currentWin;
  this->name = name;
  strncpy(this->shortName, shortName, 5);
  this->shortName[5] = '\0';
  activity = 0;
  type = 0;
}

Tab::~Tab()
{
}

Tab *Tab::getPrevious() const
{
  return previous;
}

Tab *Tab::getNext() const
{
  return next;
}

char *Tab::getName() const
{
  return name;
}

const char *Tab::getShortName() const
{
  return shortName;
}

int Tab::getActivity() const
{
  return activity;
}

int Tab::getType() const
{
  return type;
}

void Tab::setPrevious(Tab *t)
{
  previous = t;
}

void Tab::setNext(Tab *t)
{
  next = t;
}

void Tab::setCurrentWin(WINDOW *currentWin)
{
  this->currentWin = currentWin;
}

void Tab::setActivity(int activity)
{
  this->activity = activity;
  wclear(currentWin);
  if (activity)
  {
    if (has_colors())
      wattron(currentWin, COLOR_PAIR(activity));
    waddstr(currentWin, shortName);
    if (has_colors())
      wattroff(currentWin, COLOR_PAIR(activity));
  }
}

/*** ChatTab ***/

ChatTab::ChatTab(WINDOW *currentWin, char *name, int type)
  : Tab(currentWin, name, type)
{
  buffer = "";
}

string ChatTab::getBuffer() const
{
  return buffer;
}

void ChatTab::addToBuffer(char c)
{
  buffer += c;
}

void ChatTab::addToBuffer(char *c)
{
  buffer += c;
}

void ChatTab::addToBuffer(string s)
{
  buffer += s;
}

/*** BuddyTab ***/

BuddyTab::BuddyTab(WINDOW *currentWin, Contact *contact, ContactAccount *contactAccount)
  : ChatTab(currentWin, contact->getName(), 1)
{
  this->contact = contact;
  this->contactAccount = contactAccount;
}

Contact *BuddyTab::getContact() const
{
  return contact;
}

ContactAccount *BuddyTab::getContactAccount() const
{
  return contactAccount;
}

void BuddyTab::setContactAccount(ContactAccount *ca)
{
  contactAccount = ca;
}

/*** GroupTab ***/

GroupTab::GroupTab(WINDOW *currentWin, LocalAccount *localAccount, int chatId, char *name)
  : ChatTab(currentWin, name, 2)
{
  this->localAccount = localAccount;
  this->chatId = chatId;
}

LocalAccount *GroupTab::getLocalAccount() const
{
  return localAccount;
}

int GroupTab::getChatId() const
{
  return chatId;
}

/*** TabWinList ***/

TabWinList::TabWinList()
{
  number = LINES / 2;
  tabsWIN = subwin(stdscr, LINES, 7, 0, 0);
  tabWins = new WINDOW*[number];
  tabs = new Tab*[number];
  int i;
  WINDOW *w;
  for (i=0; i<number; i++)
  {
    w = subwin(tabsWIN, 1, 5, 2*i+1, 1);
    tabWins[i] = w;
    tabs[i] = NULL;
  }
  if (has_colors())
  {
    wattron(tabsWIN, A_BOLD);
    wattron(tabsWIN, COLOR_PAIR(COLOR_BLUE));
  }
}

TabWinList::~TabWinList()
{
  for (int i=0; i<number; i++)
  {
    if (tabWins[i])
    {
      delwin(tabWins[i]);
      tabWins[i] = NULL;
    }
  }
  if (tabWins)
  {
    delete[] tabWins;
    tabWins = NULL;
  }
  if (tabs)
  {
    delete[] tabs;
    tabs = NULL;
  }
}

int TabWinList::getNumber() const
{
  return number;
}

WINDOW *TabWinList::getTabsWin() const
{
  return tabsWIN;
}

WINDOW *TabWinList::getWindow(int index) const
{
  return tabWins[index];
}

Tab *TabWinList::getTab(int index) const
{
  return tabs[index];
}

int TabWinList::getIndex(Tab *t) const
{
  int result = 0;
  while (result < number && tabs[result] != t)
    result++;
  if (result == number)
    return -1;
  else
    return result;
}

void TabWinList::setTab(int index, Tab *t)
{
  tabs[index] = t;
}

void TabWinList::removeTab(Tab *t, Tab *t2)
{
  //t2 is the first tab unseen before removal (NULL if all tabs were on screen)
  int i = getIndex(t);
  if (i > -1)
  {
    for (int j=i; j < number-1; j++)
    {
      wclear(tabWins[j]);
      tabs[j] = tabs[j+1];
      if (tabs[j])
      {
	tabs[j]->setCurrentWin(tabWins[j]);
        mvwaddstr(tabWins[j], 0, 0, tabs[j]->getShortName());
      }
    }
    wclear(tabWins[number-1]);
    tabs[number-1] = t2;
    if (tabs[number-1])
    {
      tabs[number-1]->setCurrentWin(tabWins[number-1]);
      mvwaddstr(tabWins[number-1], 0, 0, tabs[number-1]->getShortName());
    }
  }
}

/*** TabList ***/

TabList::TabList()
{
  tabWinList = new TabWinList();
  firstTab = NULL;
  add("Help", "F2");
  add("Accounts", "F3");
  add("Buddies", "F4");
  currentTab = firstTab;
  WINDOW *w = tabWinList->getTabsWin();
  waddch(w, ACS_ULCORNER);
  whline(w, ACS_HLINE, 6);
  mvwaddch(w, 1, 0, ACS_VLINE);
  mvwaddch(w, 1, 6, ' ');
  mvwaddch(w, 2, 0, ACS_LTEE);
  whline(w, ACS_HLINE, 5);
  mvwaddch(w, 2, 6, ACS_URCORNER);
  mvwaddch(w, 3, 0, ACS_VLINE);
  mvwaddch(w, 4, 0, ACS_LTEE);
  whline(w, ACS_HLINE, 5);
  mvwaddch(w, 4, 6, ACS_RTEE);
  mvwaddch(w, 5, 0, ACS_VLINE);
  mvwaddch(w, 6, 0, ACS_LLCORNER);
  whline(w, ACS_HLINE, 5);
  mvwaddch(w, 6, 6, ACS_RTEE);
  w = tabWinList->getWindow(0);
  if (has_colors())
    wattron(w, A_BOLD);
  waddstr(w, tabWinList->getTab(0)->getShortName());
  if (has_colors())
    wattroff(w, A_BOLD);
  w = tabWinList->getWindow(1);
  waddstr(w, tabWinList->getTab(1)->getShortName());
  w = tabWinList->getWindow(2);
  waddstr(w, tabWinList->getTab(2)->getShortName());  
}

TabList::~TabList()
{
  while (firstTab->getPrevious() != firstTab)
    remove(firstTab->getPrevious());
  if (firstTab)
  {
    delete firstTab;
    firstTab = NULL;
  }
  if (tabWinList)
  {
    delete tabWinList;
    tabWinList = NULL;
  }
}

Tab *TabList::getFirst() const
{
  return firstTab;
}

Tab *TabList::getCurrent() const
{
  return currentTab;
}

int TabList::getNumber() const
{
  if (firstTab == NULL) return 0;
  int result = 1;
  Tab *t = firstTab->getNext();
  while (t != firstTab)
  {
    result++;
    t = t->getNext();
  }
  return result;
}

int TabList::getNumber(Tab *t) const
{
  if (firstTab == NULL) return -1;
  int result = 0;
  Tab *tempTab = firstTab;
  while (tempTab != t)
  {
    result++;
    tempTab = tempTab->getNext();
  }
  return result;  
}

Tab *TabList::getTab(int index) const
{
  return tabWinList->getTab(index);
}

BuddyTab *TabList::getBuddyTab(Contact *c) const
{
  Tab *t = firstTab->getNext()->getNext()->getNext();
  BuddyTab *bt;
  while (t != firstTab)
  {
    if (t->getType() == 1)
    {
      bt = static_cast<BuddyTab*>(t);
      if (bt->getContact() == c)
	return bt;
    }
    t = t->getNext();
  }
  return NULL;
}

BuddyTab *TabList::getBuddyTab(ContactAccount *ca) const
{
  Tab *t = firstTab->getNext()->getNext()->getNext();
  BuddyTab *bt;
  while (t != firstTab)
  {
    if (t->getType() == 1)
    {
      bt = static_cast<BuddyTab*>(t);
      if (bt->getContactAccount() == ca)
	return bt;
    }
    t = t->getNext();
  }
  return NULL;
}

GroupTab *TabList::getGroupTab(int chatId) const
{
  Tab *t = firstTab->getNext()->getNext()->getNext();
  GroupTab *gt;
  while (t != firstTab)
  {
    if (t->getType() == 2)
    {
      gt = static_cast<GroupTab*>(t);
      if (gt->getChatId() == chatId)
	return gt;
    }
    t = t->getNext();
  }
  return NULL;
}

Contact *TabList::getCurrentContact() const
{
  return static_cast<BuddyTab*>(currentTab)->getContact();
}

ContactAccount *TabList::getCurrentContactAccount() const
{
  return static_cast<BuddyTab*>(currentTab)->getContactAccount();
}

int TabList::getCurrentChatId() const
{
  return static_cast<GroupTab*>(currentTab)->getChatId();
}

void TabList::setCurrent(Tab *t)
{
  leave(currentTab);
  enter(t);
}

void TabList::setNext()
{
  leave(currentTab);
  enter(currentTab->getNext());
}

void TabList::setPrevious()
{
  leave(currentTab);
  enter(currentTab->getPrevious());
}

void TabList::leave(Tab *t) const
{
  int n = getNumber(t);
  WINDOW *w = tabWinList->getWindow(n);
  mvwaddstr(w, 0, 0, t->getShortName());
  w = tabWinList->getTabsWin();
  if (n == 0)
    mvwaddch(w, 0, 6, ACS_TTEE);
  else
    mvwaddch(w, 2*n, 6, ACS_RTEE);
  mvwaddch(w, 2*n+1, 6, ACS_VLINE);
  mvwaddch(w, 2*n+2, 6, ACS_RTEE);
}

void TabList::enter(Tab *t)
{
  currentTab = t;
  t->setActivity(0);
  int n = getNumber(t);
  WINDOW *w = tabWinList->getWindow(n);
  if (has_colors())
    wattron(w, A_BOLD);
  mvwaddstr(w, 0, 0, t->getShortName());
  if (has_colors())
    wattroff(w, A_BOLD);
  w = tabWinList->getTabsWin();
  if (n == 0)
    mvwaddch(w, 0, 6, ACS_HLINE);
  else
    mvwaddch(w, 2*n, 6, ACS_LRCORNER);
  mvwaddch(w, 2*n+1, 6, ' ');
  mvwaddch(w, 2*n+2, 6, ACS_URCORNER);
}

void TabList::add(char *name, char *shortName)
{
  int index = getNumber();
  Tab *t = new Tab(tabWinList->getWindow(index), name, shortName);
  tabWinList->setTab(index, t);
  if (firstTab == NULL)
  {
    firstTab = t;
    firstTab->setPrevious(t);
    firstTab->setNext(t);
  }
  else
  {
    firstTab->getPrevious()->setNext(t);
    t->setPrevious(firstTab->getPrevious());
    firstTab->setPrevious(t);
    t->setNext(firstTab);
  }
}

void TabList::addBuddy(Contact *c)
{
  addBuddy(c, NULL);
}

void TabList::addBuddy(Contact *c, ContactAccount *ca)
{
  int index = getNumber();
  BuddyTab *bt = new BuddyTab(tabWinList->getWindow(index), c, ca);
  tabWinList->setTab(index, bt);
  firstTab->getPrevious()->setNext(bt);
  bt->setPrevious(firstTab->getPrevious());
  firstTab->setPrevious(bt);
  bt->setNext(firstTab);
  WINDOW *w = tabWinList->getTabsWin();
  mvwaddch(w, 2*index, 0, ACS_LTEE);
  mvwaddch(w, 2*index+1, 0, ACS_VLINE);
  mvwaddch(w, 2*index+1, 6, ACS_VLINE);
  mvwaddch(w, 2*index+2, 0, ACS_LLCORNER);
  whline(w, ACS_HLINE, 5);
  mvwaddch(w, 2*index+2, 6, ACS_RTEE);
}

void TabList::addGroup(LocalAccount *la, int chatId, char *name)
{
  int index = getNumber();
  GroupTab *gt = new GroupTab(tabWinList->getWindow(index), la, chatId, name);
  tabWinList->setTab(index, gt);
  firstTab->getPrevious()->setNext(gt);
  gt->setPrevious(firstTab->getPrevious());
  firstTab->setPrevious(gt);
  gt->setNext(firstTab);
  WINDOW *w = tabWinList->getTabsWin();
  mvwaddch(w, 2*index, 0, ACS_LTEE);
  mvwaddch(w, 2*index+1, 0, ACS_VLINE);
  mvwaddch(w, 2*index+1, 6, ACS_VLINE);
  mvwaddch(w, 2*index+2, 0, ACS_LLCORNER);
  whline(w, ACS_HLINE, 5);
  mvwaddch(w, 2*index+2, 6, ACS_RTEE);
}

void TabList::remove(Tab *t)
{
  if (t)
  {
    //lastSeen is the down-most Tab you can see on the screen
    //lastSeen is NULL if there were less Tabs than space allows
    Tab *lastSeen = tabWinList->getTab(tabWinList->getNumber()-1);
    if (lastSeen && (lastSeen->getNext() != firstTab))
      tabWinList->removeTab(t, lastSeen->getNext());
    else
    {
      int n = getNumber();
      WINDOW *w = tabWinList->getWindow(n-1);
      wclear(w);
      w = tabWinList->getTabsWin();
      mvwaddch(w, 2*n-2, 0, ACS_LLCORNER);
      mvwaddch(w, 2*n-2, 6, ACS_RTEE);
      mvwaddch(w, 2*n-1, 0, ' ');
      mvwaddch(w, 2*n-1, 6, ACS_VLINE);
      mvwhline(w, 2*n, 0, ' ', 6);
      mvwaddch(w, 2*n, 6, ACS_VLINE);
      tabWinList->removeTab(t, NULL);
    }
    t->getPrevious()->setNext(t->getNext());
    t->getNext()->setPrevious(t->getPrevious());
    enter(t->getNext());
    delete t;
    t = NULL;
  }  
}

void TabList::removeCurrent()
{
  remove(currentTab);
}

void TabList::refresh() const
{
  wrefresh(tabWinList->getTabsWin());
  WINDOW *w;
  for (int i=0; i<tabWinList->getNumber(); i++)
  {
    w = tabWinList->getWindow(i);
    wrefresh(w);
  }
}

/*** MainWin ***/

MainWin::MainWin()
{
  mainWIN = subwin(stdscr, LINES, COLS-6, 0, 6);
  chatWIN = subwin(mainWIN, LINES-4, COLS-8, 1, 7);
  sublineWIN = subwin(mainWIN, 1, COLS-8, LINES-2, 7);
  groupWIN = NULL;
  accountsWin = new AccountsWin();
  contactsWin = new ContactsWin();

  scrollok(chatWIN, true);
  if (has_colors())
  {
    wattron(mainWIN, A_BOLD);
    wattron(mainWIN, COLOR_PAIR(COLOR_BLUE));
  }
  box(mainWIN, ACS_VLINE, ACS_HLINE);
  mvwaddch(mainWIN, LINES-3, 0, ACS_LTEE);
  whline(mainWIN, ACS_HLINE, COLS-8);
  mvwaddch(mainWIN, LINES-3, COLS-7, ACS_RTEE);
  mvwhline(mainWIN, LINES-1, 1, ACS_HLINE, COLS-8);

  waddstr(chatWIN, "test");
}

MainWin::~MainWin()
{
  if (chatWIN)
  {
    delete chatWIN;
    chatWIN = NULL;
  }
  if (sublineWIN)
  {
    delete sublineWIN;
    sublineWIN = NULL;
  }
  if (accountsWin)
  {
    delete accountsWin;
    accountsWin = NULL;
  }
  if (contactsWin)
  {
    delete contactsWin;
    contactsWin = NULL;
  }
}

AccountsWin *MainWin::getAccountsWin() const
{
  return accountsWin;
}

ContactsWin *MainWin::getContactsWin() const
{
  return contactsWin;
}

void MainWin::title(char *name) const
{
  mvwhline(mainWIN, 0, 1, ACS_HLINE, COLS-8);
  if (has_colors())
    wattroff(mainWIN, COLOR_PAIR(COLOR_BLUE));
  //center the title
  wmove(mainWIN, 0, (COLS-8-strlen(name))/2);
  waddstr(mainWIN, name);
  if (has_colors())
    wattron(mainWIN, COLOR_PAIR(COLOR_BLUE));
}

void MainWin::showHelp() const
{
  wclear(chatWIN);
  mvwaddstr(chatWIN, 0, 0, "Welcome to EbNcurses,");
  mvwaddstr(chatWIN, 1, 0, "the ncurses UI for EB-lite.");
  mvwaddstr(chatWIN, 3, 0, "F2 displays this window");
  mvwaddstr(chatWIN, 4, 0, "F3 displays local accounts and manages connection");
  mvwaddstr(chatWIN, 5, 0, "F4 displays groups of buddies");
  mvwaddstr(chatWIN, 6, 0, "F9 closes a buddy tab");
  mvwaddstr(chatWIN, 7, 0, "F12 quits.");
  mvwaddstr(chatWIN, 8, 0, "PageDown and PageUp move through tabs");

  wclear(sublineWIN);
  if (has_colors())
    wattron(sublineWIN, COLOR_PAIR(COLOR_CYAN));
  waddstr(sublineWIN, "F2 Help - F3 Accounts - F4 Contacts - F12 Quit  ");
  if (has_colors())
    wattroff(sublineWIN, COLOR_PAIR(COLOR_CYAN));
}

void MainWin::showAccounts(LocalAccountList *localAccounts) const
{
  wclear(chatWIN);
  mvwaddstr(chatWIN, 0, 0, "Hit c to Connect to all, d to Disconnect from all, g to join a Groupchat.");
  LocalAccountList *la = localAccounts;
  int line = 1;
  int fromLine = accountsWin->getFirst();
  int toLine = fromLine + LINES - 6;
  while (la->getFirst() != NULL)
  {
    if (line >= fromLine && line <= toLine)
    {
      if (line == accountsWin->getHighlight())
      {
	wattron(chatWIN, A_REVERSE);
	accountsWin->highlightLocalAccount(la->getFirst());
      }
      if (has_colors())
	wattron(chatWIN, COLOR_PAIR(la->getFirst()->getService()->getNearestColour()));
      wmove(chatWIN, line, 0);
      if (la->getFirst()->isReady())
	waddstr(chatWIN, "O ");
      else if (la->getFirst()->isConnected())
	waddstr(chatWIN, "o ");
      else
	waddstr(chatWIN, ". ");
      waddstr(chatWIN, la->getFirst()->getUsername());
      if (has_colors())
	wattroff(chatWIN, COLOR_PAIR(la->getFirst()->getService()->getNearestColour()));
      if (line == accountsWin->getHighlight())
	wattroff(chatWIN, A_REVERSE);
    }
    line++;
    la = la->getNext();
  }
  accountsWin->setLength(line - 1);

  wclear(sublineWIN);
  if (has_colors())
    wattron(sublineWIN, COLOR_PAIR(COLOR_CYAN));
  waddstr(sublineWIN, "F2 Help - F3 Accounts - F4 Contacts - F12 Quit  ");
  if (has_colors())
    wattroff(sublineWIN, COLOR_PAIR(COLOR_CYAN));
}

void MainWin::showContacts(GroupList *groups, bool showOnlineOnly) const
{
  wclear(chatWIN);
  mvwaddstr(chatWIN, 0, 0, "Hit a to display All buddies, o for Online only.");
  GroupList *g = groups;
  ContactList *c;
  ContactAccountList *ca;
  int line = 1;
  int fromLine = contactsWin->getFirst();
  int toLine = fromLine + LINES - 6;
  while (g->getFirst() != NULL)
  {
    if (line >= fromLine && line <= toLine)
    {
      if (line == contactsWin->getHighlight())
      {
	wattron(chatWIN, A_REVERSE);
	contactsWin->highlightContact(NULL);
	contactsWin->highlightContactAccount(NULL);
      }
      waddstr(chatWIN, "\n");
      wattron(chatWIN, A_UNDERLINE);
      waddstr(chatWIN, g->getFirst()->getName());
      wattroff(chatWIN, A_UNDERLINE);
      if (line == contactsWin->getHighlight())
	wattroff(chatWIN, A_REVERSE);
    }
    line++;
    c = g->getFirst()->getContacts();
    while (c->getFirst() != NULL)
    {
      if (!showOnlineOnly || c->getFirst()->isLogged())
      {
	if (line >= fromLine && line <=toLine)
	{
	  if (line == contactsWin->getHighlight())
	  {
	    wattron(chatWIN, A_REVERSE);
	    contactsWin->highlightContact(c->getFirst());
	    contactsWin->highlightContactAccount(NULL);
	  }
	  waddstr(chatWIN, "\n  ");
	  switch (c->getFirst()->getStatus())
	  {
	  case 0:
	    waddstr(chatWIN, ". ");
	    break;
	  case 1:
	    waddstr(chatWIN, "O ");
	    break;
	  case 2:
	    waddstr(chatWIN, "x ");
	    break;
	  }
	  waddstr(chatWIN, c->getFirst()->getName());
	  if (showOnlineOnly && c->getFirst()->getStatusString() && strlen(c->getFirst()->getStatusString()) > 0)
	  {
	    waddstr(chatWIN, " (");
	    waddstr(chatWIN, c->getFirst()->getStatusString());
	    waddstr(chatWIN, ")");
	  }
	  if (line == contactsWin->getHighlight())
	    wattroff(chatWIN, A_REVERSE);
	}
	line++;
      }
      if (!showOnlineOnly)
      {
	ca = c->getFirst()->getAccounts();
	while (ca->getFirst() != NULL)
	{
	  if (line >= fromLine && line <=toLine)
	  {
	    if (line == contactsWin->getHighlight())
	    {
	      wattron(chatWIN, A_REVERSE);
	      contactsWin->highlightContact(NULL);
	      contactsWin->highlightContactAccount(ca->getFirst());
	    }
	    if (has_colors())
	      wattron(chatWIN, COLOR_PAIR(ca->getFirst()->getService()->getNearestColour()));
	    waddstr(chatWIN, "\n    ");
	    switch (ca->getFirst()->getStatus())
	    {
	    case 0:
	      waddstr(chatWIN, ". ");
	      break;
	    case 1:
	      waddstr(chatWIN, "O ");
	      break;
	    case 2:
	      waddstr(chatWIN, "x ");
	      break;
	    }
	    waddstr(chatWIN, ca->getFirst()->getName());
	    if (ca->getFirst()->getStatusString() && strlen(ca->getFirst()->getStatusString()) > 0)
	    {
	      waddstr(chatWIN, " (");
	      waddstr(chatWIN, ca->getFirst()->getStatusString());
	      waddstr(chatWIN, ")");
	    }
	    if (has_colors())
	      wattroff(chatWIN, COLOR_PAIR(ca->getFirst()->getService()->getNearestColour()));
	    if (line == contactsWin->getHighlight())
	      wattroff(chatWIN, A_REVERSE);
	  }
	  line++;
	  ca = ca->getNext();
	}
      }
      c = c->getNext();
    }
    g = g->getNext();
  }
  contactsWin->setLength(line - 1);

  wclear(sublineWIN);
  if (has_colors())
    wattron(sublineWIN, COLOR_PAIR(COLOR_CYAN));
  waddstr(sublineWIN, "F2 Help - F3 Accounts - F4 Contacts - F12 Quit  ");
  if (has_colors())
    wattroff(sublineWIN, COLOR_PAIR(COLOR_CYAN));
}

void MainWin::showTab(string chatBuffer) const
{
  wclear(chatWIN);
  string::size_type pos1 = 0;
  string::size_type pos2 = chatBuffer.find("\n", pos1);
  int color1 = 0;
  int color2;
  string currentString;
  while (pos2 != string::npos)
  {
    pos2++;
    color2 = strtol(chatBuffer.substr(pos1, 1).c_str(), NULL, 10);
    // test if the first char was really a number... 
    // otherwise the \n was in the same paragraph
    if (color2 == 0)
      currentString = chatBuffer.substr(pos1 , pos2 - pos1 );
    else
    {
      currentString = chatBuffer.substr(pos1 + 1, pos2 - pos1 - 1);
      if (has_colors())
      {
	//remove color only when when we have a real new line
	wattroff(chatWIN, COLOR_PAIR(color1));
	wattron(chatWIN, COLOR_PAIR(color2));
      }
    }
    waddstr(chatWIN, currentString.c_str());
    if (color2 > 0)
      color1 = color2;
    pos1 = pos2;
    pos2 = chatBuffer.find("\n", pos1);
  }
  wattroff(chatWIN, COLOR_PAIR(color1));
}

void MainWin::inputGroupchat() const
{
  wclear(sublineWIN);
  waddstr(sublineWIN, "Join groupchat: ");
}

void MainWin::addLine(int color, char *message) const
{
  if (has_colors())
    wattron(chatWIN, COLOR_PAIR(color));
  waddstr(chatWIN, message);
  if (has_colors())
    wattroff(chatWIN, COLOR_PAIR(color));
}

void MainWin::addLine(int color, string message) const
{
  if (has_colors())
    wattron(chatWIN, COLOR_PAIR(color));
  waddstr(chatWIN, message.c_str());
  if (has_colors())
    wattroff(chatWIN, COLOR_PAIR(color));
}

void MainWin::addChar(char ch) const
{
  waddch(sublineWIN, ch);
}

void MainWin::eraseChar() const
{
  int x, y;
  getyx(sublineWIN, y, x);
  wmove(sublineWIN, y, x-1);
  wdelch(sublineWIN);
}

void MainWin::clearSubLine() const
{
  wclear(sublineWIN);
}

void MainWin::drawSubline(char *buffer) const
{
  wclear(sublineWIN);
  waddstr(sublineWIN, buffer);
}

void MainWin::refresh() const
{
  wrefresh(mainWIN);
  wrefresh(chatWIN);
  wrefresh(sublineWIN);
}

/*** EbncGUI ***/

EbncGUI::EbncGUI(ServiceList *services, GroupList *groups, LocalAccountList *localAccounts)
{
  (void) signal(SIGINT, finish);      /* arrange interrupts to terminate */
  (void) initscr();      /* initialize the curses library */
  (void) nonl();         /* tell curses not to do NL->CR/NL on output */
  (void) cbreak();       /* take input chars one at a time, no wait for \n */
  //(void) raw();          /* do not interprete signals */
  (void) noecho();       /* don't echo input */
  keypad(stdscr, TRUE); /* enable keyboard mapping and special keys */
  nodelay(stdscr, TRUE);/* getch won't block */
  curs_set(0);          /* invisible cursor */
  
  if (has_colors())
  {
    start_color();
    init_pair(COLOR_BLACK, COLOR_BLACK, COLOR_BLACK);
    init_pair(COLOR_RED, COLOR_RED, COLOR_BLACK);
    init_pair(COLOR_GREEN, COLOR_GREEN, COLOR_BLACK);
    init_pair(COLOR_YELLOW, COLOR_YELLOW, COLOR_BLACK);
    init_pair(COLOR_BLUE, COLOR_BLUE, COLOR_BLACK);
    init_pair(COLOR_MAGENTA, COLOR_MAGENTA, COLOR_BLACK);
    init_pair(COLOR_CYAN, COLOR_CYAN, COLOR_BLACK);
    init_pair(COLOR_WHITE, COLOR_WHITE, COLOR_BLACK);
  }

  mainWin = new MainWin();
  tabsWin = new TabList();
  currentWin = 0;
  draw();
  sublineMode = false;
  showOnlineOnly = true;
  buffer = new char[256];
  currentChar = 0;
  this->services = services;
  this->groups = groups;
  this->localAccounts = localAccounts;
}

EbncGUI::~EbncGUI()
{
  if (tabsWin)
  {
    delete tabsWin;
    tabsWin = NULL;
  }
  if (mainWin)
  {
    delete mainWin;
    mainWin = NULL;
  }

  finish(0);
}

void EbncGUI::readInput()
{
  int ch = getch(); //this calls refresh() ... 8-/
  // returns instead of breaks so I can handle letters key at the end
  switch (ch)
  {
  case KEY_PPAGE:
    tabsWin->setPrevious();
    tabsWin->refresh();
    if (currentWin == 0)
      currentWin = tabsWin->getNumber() - 1;
    else
      currentWin--;
    draw();
    return;
  case KEY_NPAGE:
    tabsWin->setNext();
    tabsWin->refresh();
    if (currentWin == tabsWin->getNumber() - 1)
      currentWin = 0;
    else
      currentWin++;
    draw();
    return;
  case KEY_F(2):
    tabsWin->setCurrent(tabsWin->getFirst());
    tabsWin->refresh();
    currentWin = 0;
    draw();
    return;
  case KEY_F(3):
    tabsWin->setCurrent(tabsWin->getFirst()->getNext());
    tabsWin->refresh();
    currentWin = 1;
    draw();
    return;
  case KEY_F(4):
    tabsWin->setCurrent(tabsWin->getFirst()->getNext()->getNext());
    tabsWin->refresh();
    currentWin = 2;
    draw();
    return;
  case KEY_F(9):
    if (currentWin >= 3)
    {
      if (tabsWin->getCurrent()->getType() == 2)
      {
	char *cmd[3];
	char tmp[12];
	cmd[0] = "close_group_chat";
	sprintf(tmp, "%d", static_cast<GroupTab*>(tabsWin->getCurrent())->getChatId());
	cmd[1] = tmp;
	cmd[2] = NULL;
	send_gui_message(cmd, 2);
      }
      else
      {
	if (currentWin == tabsWin->getNumber() - 1)
	  currentWin = 0;
	tabsWin->removeCurrent();
	tabsWin->refresh();
	draw();
      }
    }
    return;
  case KEY_F(12):
    finish(0);
    //delete this; //best way to do it ??
    return;
  case KEY_UP:
    if (currentWin == 1)
    {
      mainWin->getAccountsWin()->upHighlight();
      mainWin->showAccounts(localAccounts);
    }
    else if (currentWin == 2)
    {
      mainWin->getContactsWin()->upHighlight();
      mainWin->showContacts(groups, showOnlineOnly);
    }
    mainWin->refresh();
    return;
  case KEY_DOWN:
    if (currentWin == 1)
    {
      mainWin->getAccountsWin()->downHighlight();
      mainWin->showAccounts(localAccounts);
    }
    else if (currentWin == 2)
    {
      mainWin->getContactsWin()->downHighlight();
      mainWin->showContacts(groups, showOnlineOnly);
    }
    mainWin->refresh();
    return;
  case KEY_BACKSPACE: // ?
  case 127: // my BACKSPACE key...
    if (currentWin >= 3)
      if (currentChar > 0)
      {
	buffer[currentChar] = '\0';
	currentChar--;
	mainWin->eraseChar();
	mainWin->refresh();
      }
    return;
  case '\n': // ?
  case '\r': // keypad ENTER
  case KEY_ENTER: // numpad ENTER
    if (currentWin == 1)
    {
      if (sublineMode)
      {
	LocalAccount *la = mainWin->getAccountsWin()->getLocalAccountHighlighted();	
	char *cmd[5];
	cmd[0] = "join_group_chat";
	cmd[1] = la->getUsername();
	cmd[2] = la->getService()->getName();
	cmd[3] = buffer;
	cmd[4] = NULL;
	send_gui_message(cmd, 4);
	memset(buffer, '\0', 255);
	currentChar = 0;
	sublineMode = false;
	mainWin->refresh();
      }
      else
      {
	LocalAccount *la = mainWin->getAccountsWin()->getLocalAccountHighlighted();
// 	char *cmd[4];
// 	cmd[1] = la->getUsername();
// 	cmd[2] = la->getService()->getName();
// 	cmd[3] = NULL;
	if (la->isConnected())
	{
	  //doesn't seem to work...
//	  cmd[0] = "local_account_logout";
	  char *cmd[5];
	  cmd[0] = "set_local_account_status";
	  cmd[1] = la->getUsername();
	  cmd[2] = la->getService()->getName();
	  cmd[3] = "Offline";
	  cmd[4] = NULL;
	  send_gui_message(cmd, 4);
	}
	else
	{
	  char *cmd[4];
	  cmd[0] = "local_account_login";
	  cmd[1] = la->getUsername();
	  cmd[2] = la->getService()->getName();
	  cmd[3] = NULL;
	  send_gui_message(cmd, 3);
	}
      }
    }
    else if (currentWin == 2)
    {
      Contact *c;
      ContactAccount *ca;
      if (mainWin->getContactsWin()->getContactAccountHighlighted())
      {
	ca = mainWin->getContactsWin()->getContactAccountHighlighted();
	c = ca->getContact();
      }
      else if (mainWin->getContactsWin()->getContactHighlighted())
      {
	c = mainWin->getContactsWin()->getContactHighlighted();
	ca = NULL;
      }
      else
	return;
      BuddyTab *buddyTab = tabsWin->getBuddyTab(c);
      if (buddyTab != NULL)
      {
	tabsWin->setCurrent(buddyTab);
	buddyTab->setContactAccount(ca);
	currentWin = tabsWin->getNumber(buddyTab);
      }
      else
      {
	tabsWin->addBuddy(c, ca);
	tabsWin->setCurrent(tabsWin->getFirst()->getPrevious());
	currentWin = tabsWin->getNumber() - 1;
      }
      tabsWin->refresh();
      draw();
    }
    else if (currentWin >= 3)
    {
      if (tabsWin->getTab(currentWin)->getType() == 1)
      {
	Contact *c = tabsWin->getCurrentContact();
	ContactAccount *ca = tabsWin->getCurrentContactAccount();
	char *cmd[8];
	cmd[0] = "message_send";
	cmd[1] = c->getGroup()->getName();
	cmd[2] = c->getName();
	cmd[6] = buffer;
	cmd[7] = NULL;
	if (ca)
	{
	  cmd[3] = ca->getLocalAccount()->getUsername();
	  cmd[4] = ca->getService()->getName();
	  cmd[5] = ca->getName();
	}
	else
	{
	  cmd[3] = "";
	  cmd[4] = "";
	  cmd[5] = "";
	}
	send_gui_message(cmd, 7);
      }
      else
      {
	char *cmd[4];
	char tmp[12];
	cmd[0] = "group_chat_send";
	sprintf(tmp, "%d", tabsWin->getCurrentChatId());
	cmd[1] = tmp;
	cmd[2] = buffer;
	cmd[3] = NULL;
	send_gui_message(cmd, 3);
      }
      memset(buffer, '\0', 255);
      currentChar = 0;
      mainWin->clearSubLine();
      mainWin->refresh();
    }
    return;
  case 'c': // Connect all
    if (currentWin < 3 && !sublineMode)
    {
      char *cmd[2];
      cmd[0] = "sign_on_all";
      cmd[1] = NULL;
      send_gui_message(cmd, 1);;
      tabsWin->setCurrent(tabsWin->getFirst()->getNext());
      tabsWin->refresh();
      currentWin = 1;
      draw();
      return;
    }
    break;
  case 'd': // Disconnect all
    if (currentWin < 3 && !sublineMode)
    {
      char *cmd[2];
      cmd[0] = "sign_off_all";
      cmd[1] = NULL;
      send_gui_message(cmd, 1);;
      tabsWin->setCurrent(tabsWin->getFirst()->getNext());
      tabsWin->refresh();
      currentWin = 1;
      draw();
      return;
    }
    break;
  case 'g': // Join groupchat
    if (currentWin == 1 && !sublineMode)
    {
      sublineMode = true;
      mainWin->inputGroupchat();
      mainWin->refresh();
      return;
    }
  case 'a': // All buddies
    if (currentWin < 3 && !sublineMode)
    {
      mainWin->getContactsWin()->resetHighlight();
      showOnlineOnly = false;
      tabsWin->setCurrent(tabsWin->getFirst()->getNext()->getNext());
      tabsWin->refresh();
      currentWin = 2;
      draw();
      return;
    }
    break;
  case 'o': // Online buddies only
    if (currentWin < 3 && !sublineMode)
    {
      mainWin->getContactsWin()->resetHighlight();
      showOnlineOnly = true;
      tabsWin->setCurrent(tabsWin->getFirst()->getNext()->getNext());
      tabsWin->refresh();
      currentWin = 2;
      draw();
      return;
    }
    break;
  case 'n':
    if (currentWin < 3 && !sublineMode)
    {
      ;//will handle things later
      return;
    }
    break;
  case KEY_RESIZE:
    draw();
    break;
  }
  
  // If the function has not reached any return so far...
  if (currentWin >= 3 || sublineMode)
  {
    // 	  char *intkey = new char [12];
    // 	  sprintf(intkey, "%d", ch);
    // 	  recv(false, "KEY", intkey);
    // 	  delete intkey;
    if (currentChar < 255)
    {
      mainWin->addChar(ch);
      mainWin->refresh();
      buffer[currentChar] = ch;
      currentChar++;
    }
  }
}

void EbncGUI::updateAccounts() const
{
  if (currentWin == 1)
    draw();
  else
  {
    tabsWin->getFirst()->getNext()->setActivity(COLOR_RED);
    tabsWin->refresh();
  }
}

void EbncGUI::updateContacts() const
{
  if (currentWin == 2)
    draw();
  else
  {
    tabsWin->getFirst()->getNext()->getNext()->setActivity(COLOR_RED);
    tabsWin->refresh();
  }
}

void EbncGUI::showBuddyStatus(ContactAccount *ca, int state, char *status) const
{
  BuddyTab *buddyTab = tabsWin->getBuddyTab(ca);
  if (buddyTab != NULL)
  {
    string message = ca->getName();
    message += " is now ";
    switch (state)
    {
    case 0:
      message += "offline";
      break;
    case 1:
      message += "online";
      break;
    case 2:
      message += "away";
      break;
    }
    if (state > 0 && status != "")
    {
      message += " (";
      message += status;
      message += ")";
    }
    else
      message += ".";
    message += "\n";
    
    if (currentWin >= 3 && static_cast<BuddyTab*>(tabsWin->getCurrent()) == buddyTab)
    {
      mainWin->addLine(COLOR_MAGENTA, message);
      mainWin->refresh();
    }
    else if (buddyTab->getActivity() != COLOR_RED)
    {
      buddyTab->setActivity(COLOR_MAGENTA);
      tabsWin->refresh();
    }
    buddyTab->addToBuffer(char(COLOR_MAGENTA + '0'));
    buddyTab->addToBuffer(message);  
  }
}

void EbncGUI::receiveMessage(ContactAccount *ca, char *message) const
{
  BuddyTab *buddyTab = tabsWin->getBuddyTab(ca->getContact());
  string msg = ca->getName();
  msg += ": ";
  msg += message;
  msg += "\n";
  if (buddyTab != NULL)
  {
    buddyTab->setContactAccount(ca);
    if (currentWin >= 3 && tabsWin->getCurrent()->getType() == 1 && static_cast<BuddyTab*>(tabsWin->getCurrent()) == buddyTab)
    {
      mainWin->addLine(COLOR_WHITE, msg.c_str());
      mainWin->refresh();
    }
    else
    {
      buddyTab->setActivity(COLOR_RED);
      tabsWin->refresh();
    }
  }
  else
  {
    tabsWin->addBuddy(ca->getContact(), ca);
    buddyTab = static_cast<BuddyTab*>(tabsWin->getFirst()->getPrevious());
    buddyTab->setActivity(COLOR_RED);
    tabsWin->refresh();
  }
  buddyTab->addToBuffer(char(COLOR_WHITE + '0'));
  buddyTab->addToBuffer(msg);
}

void EbncGUI::sendMessage(ContactAccount *ca, char *message) const
{
  BuddyTab *buddyTab = tabsWin->getBuddyTab(ca->getContact());
  string msg = ca->getLocalAccount()->getUsername();
  msg += ": ";
  msg += message;
  msg += "\n";
  if (buddyTab != NULL)
  {
    buddyTab->setContactAccount(ca);
    if (currentWin >= 3 && tabsWin->getCurrent()->getType() == 1 && static_cast<BuddyTab*>(tabsWin->getCurrent()) == buddyTab)
    {
      mainWin->addLine(COLOR_CYAN, msg.c_str());
      mainWin->refresh();
    }
    else
    {
      buddyTab->setActivity(COLOR_RED);
      tabsWin->refresh();
    }
  }
  else
  {
    tabsWin->addBuddy(ca->getContact(), ca);
    buddyTab = static_cast<BuddyTab*>(tabsWin->getFirst()->getPrevious());
    buddyTab->setActivity(COLOR_RED);
    tabsWin->refresh();
  }
  buddyTab->addToBuffer(char(COLOR_CYAN + '0'));
  buddyTab->addToBuffer(msg);
}

void EbncGUI::openGroupchat(LocalAccount *la, int chatId, char *name)
{
  tabsWin->addGroup(la, chatId, strdup(name));
  tabsWin->setCurrent(tabsWin->getFirst()->getPrevious());
  tabsWin->refresh();
  currentWin = tabsWin->getNumber() - 1;
  draw();
}

void EbncGUI::receiveGroupMessage(int chatId, char *handle, char *message) const
{
  GroupTab *groupTab = tabsWin->getGroupTab(chatId);
  string msg = handle;
  msg += ": ";
  msg += message;
  msg += "\n";
  if (currentWin >= 3 && tabsWin->getCurrent()->getType() == 2 && static_cast<GroupTab*>(tabsWin->getCurrent()) == groupTab)
  {
    mainWin->addLine(COLOR_WHITE, msg.c_str());
    mainWin->refresh();
  }
  else
  {
    groupTab->setActivity(COLOR_RED);
    tabsWin->refresh();
  }
  groupTab->addToBuffer(char(COLOR_WHITE + '0'));
  groupTab->addToBuffer(msg);  
}

void EbncGUI::receiveGroup3rd(int chatId, char *message) const
{
  GroupTab *groupTab = tabsWin->getGroupTab(chatId);
  string msg = "* ";
  msg += message;
  msg += "\n";
  if (currentWin >= 3 && tabsWin->getCurrent()->getType() == 2 && static_cast<GroupTab*>(tabsWin->getCurrent()) == groupTab)
  {
    mainWin->addLine(COLOR_MAGENTA, msg.c_str());
    mainWin->refresh();
  }
  else
  {
    groupTab->setActivity(COLOR_RED);
    tabsWin->refresh();
  }
  groupTab->addToBuffer(char(COLOR_MAGENTA + '0'));
  groupTab->addToBuffer(msg);
}

void EbncGUI::sendGroupMessage(int chatId, char *message) const
{
  GroupTab *groupTab = tabsWin->getGroupTab(chatId);
  string msg = groupTab->getLocalAccount()->getUsername();
  msg += ": ";
  msg += message;
  msg += "\n";
  if (currentWin >= 3 && tabsWin->getCurrent()->getType() == 2 && static_cast<GroupTab*>(tabsWin->getCurrent()) == groupTab)
  {
    mainWin->addLine(COLOR_CYAN, msg.c_str());
    mainWin->refresh();
  }
  else
  {
    groupTab->setActivity(COLOR_RED);
    tabsWin->refresh();
  }
  groupTab->addToBuffer(char(COLOR_CYAN + '0'));
  groupTab->addToBuffer(msg);
}

void EbncGUI::joinedGroup(int chatId, char *handle) const
{
  GroupTab *groupTab = tabsWin->getGroupTab(chatId);
  string msg = handle;
  msg += " has joined ";
  msg += groupTab->getName();
  msg += "\n";
  if (currentWin >= 3 && tabsWin->getCurrent()->getType() == 2 && static_cast<GroupTab*>(tabsWin->getCurrent()) == groupTab)
  {
    mainWin->addLine(COLOR_BLUE, msg.c_str());
    mainWin->refresh();
  }
  else
  {
    groupTab->setActivity(COLOR_MAGENTA);
    tabsWin->refresh();
  }
  groupTab->addToBuffer(char(COLOR_BLUE + '0'));
  groupTab->addToBuffer(msg);  
}

void EbncGUI::leftGroup(int chatId, char *handle) const
{
  GroupTab *groupTab = tabsWin->getGroupTab(chatId);
  string msg = handle;
  msg += " has left ";
  msg += groupTab->getName();
  msg += "\n";
  if (currentWin >= 3 && tabsWin->getCurrent()->getType() == 2 && static_cast<GroupTab*>(tabsWin->getCurrent()) == groupTab)
  {
    mainWin->addLine(COLOR_BLUE, msg.c_str());
    mainWin->refresh();
  }
  else
  {
    groupTab->setActivity(COLOR_MAGENTA);
    tabsWin->refresh();
  }
  groupTab->addToBuffer(char(COLOR_BLUE + '0'));
  groupTab->addToBuffer(msg);  
}

void EbncGUI::closeGroup(int chatId)
{
  GroupTab *groupTab = tabsWin->getGroupTab(chatId);
  if (tabsWin->getCurrent() == groupTab)
  {
    if (currentWin == tabsWin->getNumber() - 1)
      currentWin = 0;
    tabsWin->removeCurrent();
  }
  else
  {
    Tab *cur = tabsWin->getCurrent();
    if (tabsWin->getNumber(groupTab) < currentWin)
      currentWin--;
    tabsWin->remove(groupTab);
    tabsWin->setCurrent(cur);
  }
  tabsWin->refresh();
  draw();
}

void EbncGUI::draw() const
{
  switch (currentWin)
  {
  case 0:
    mainWin->showHelp();
    break;
  case 1:
    mainWin->showAccounts(localAccounts);
    break;
  case 2:
    mainWin->showContacts(groups, showOnlineOnly);
    break;
  default:
    mainWin->showTab(static_cast<ChatTab*>(tabsWin->getTab(currentWin))->getBuffer());
    mainWin->drawSubline(buffer);
  }
  mainWin->title(tabsWin->getTab(currentWin)->getName());
  mainWin->refresh();
}
