/* File     : service.cpp
   Desc     : Service classes
   Authors  : Karine Proot (kproot@nerim.net)                              */

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <iostream>
#include <string.h>
#include <stdlib.h>
#include "service.h"

/*** Action ***/

Action::Action(char *name)
{
  this->name = strdup(name);
}

Action::~Action()
{
  if (name)
  {
    free(name);
    name = NULL;
  }
}

/*** ActionList ***/

ActionList::ActionList()
{
  first = NULL;
  next = NULL;
}

ActionList::~ActionList()
{
  if (first)
  {
    delete first;
    first = NULL;
  }
  if (next)
  {
    delete next;
    next = NULL;
  }
}

void ActionList::add(Action *a)
{
  if (first == NULL)
  {
    first = a;
    next = new ActionList();
  }
  else
    next->add(a);
}

/*** ServiceAction ***/

ServiceAction::ServiceAction(char *type, char *numActions)
{
  this->type = strdup(type);
  this->numActions = strtol(numActions, NULL, 10);
  actions = new ActionList();
}

ServiceAction::~ServiceAction()
{
  if (type)
  {
    free(type);
    type = NULL;
  }
  if (actions)
  {
    delete actions;
    actions = NULL;
  }
}

void ServiceAction::addAction(Action *a)
{
  actions->add(a);
}

/*** ServiceActionList ***/

ServiceActionList::ServiceActionList()
{
  first = NULL;
  next = NULL;
}

ServiceActionList::~ServiceActionList()
{
  if (first)
  {
    delete first;
    first = NULL;
  }
  if (next)
  {
    delete next;
    next = NULL;
  }
}

void ServiceActionList::add(ServiceAction *sa)
{
  if (first == NULL)
  {
    first = sa;
    next = new ServiceActionList();
  }
  else
    next->add(sa);
}

/*** State ***/

State::State(char *name)
{
  this->name = strdup(name);
}

State::~State()
{
  if (name)
  {
    free(name);
    name = NULL;
  }
}

/*** StateList ***/

StateList::StateList()
{
  first = NULL;
  next = NULL;
}

StateList::~StateList()
{
  if (first)
  {
    delete first;
    first = NULL;
  }
  if (next)
  {
    delete next;
    next = NULL;
  }
}

void StateList::add(State *s)
{
  if (first == NULL)
  {
    first = s;
    next = new StateList();
  }
  else
    next->add(s);
}

/*** Service ***/

Service::Service(char *name, char *colour, char *capabilities)
{
  this->name = strdup(name);
  this->colour = convert(colour);
  this->capabilities = strtol(capabilities, NULL, 10);
  serviceActions = new ServiceActionList();
  numStates = 0;
  states = new StateList();
}

Service::~Service()
{
  if (name)
  {
    free(name);
    name = NULL;
  }
  if (serviceActions)
  {
    delete serviceActions;
    serviceActions = NULL;
  }
  if (states)
  {
    delete states;
    states = NULL;
  }
}

char *Service::getName() const
{
  return name;
}

int Service::getR() const
{
  return colour.r;
}

int Service::getG() const
{
  return colour.g;
}

int Service::getB() const
{
  return colour.b;
}

int Service::getNearestColour() const // (c) Beloved penguin of mine.
{
  // What a coincidence, ncurses colours are in the right order to make bitwise operations!
  // colour.x & 0x80 takes the most significant bit to see whether the colour component is
  //   over 128 or not, then I shift it to the right in bgr order to match ncurses order.
  // colour.r doesn't need to be masked out as it will be shifted by seven, all other bits
  //   disappear.
  return (colour.r >> 7) | ((colour.g & 0x80) >> 6) | ((colour.b & 0x80) >> 5);
}

void Service::setNumStates(int n)
{
  numStates = n;
}

void Service::setNumStates(char *n)
{
}

void Service::addServiceAction(ServiceAction *sa)
{
  serviceActions->add(sa);
}

void Service::addState(State *s)
{
  states->add(s);
}

RgbColour Service::convert(char *colour) const // (c) Beloved penguin of mine.
{
  // Warning: the color string will be modified after this!
  RgbColour result;
  // get blue component
  result.b = strtoul(colour + 5, NULL, 16);
  colour[5] = '\0';
  // get green component
  result.g = strtoul(colour + 3, NULL, 16);
  colour[3] = '\0';
  // get red component
  result.r = strtoul(colour + 1, NULL, 16);
  return result;
}

/*** ServiceList ***/

ServiceList::ServiceList()
{
  first = NULL;
  next = NULL;
}

ServiceList::~ServiceList()
{
  if (first)
  {
    delete first;
    first = NULL;
  }
  if (next)
  {
    delete next;
    next = NULL;
  }
}

int ServiceList::count() const
{
  if (first == NULL)
    return 0;
  else
    return 1 + next->count();
}

Service *ServiceList::getFirst() const
{
  return first;
}

void ServiceList::setFirst(Service *s)
{
  first = s;
}

void ServiceList::add(Service *s)
{
  if (first == NULL)
  {
    first = s;
    next = new ServiceList();
  }
  else
    next->add(s);
}

Service *ServiceList::find(char *name) const
{
  if (first == NULL)
    return NULL;
  else if (!strcmp(first->getName(), name))
    return first;
  else if (next == NULL)
    return NULL;
  else
    return next->find(name);
}

