/* File     : group.cpp
   Desc     : Group classes
   Authors  : Karine Proot (kproot@nerim.net)                              */

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <iostream>
#include <string.h>
#include <stdlib.h>
#include "service.h"
#include "account.h"
#include "group.h"

/*** ContactAccount ***/

ContactAccount::ContactAccount(LocalAccount *localAccount, char *name, char *isBlocked)
{
  this->localAccount = localAccount;
  this->name = name ? strdup(name) : NULL;
  statusString = NULL;
  if (isBlocked == NULL)
    this->isBlocked = 0;
  else
    this->isBlocked = strtol(isBlocked, NULL, 10);
  status = 0;
  contact = NULL;
}

ContactAccount::~ContactAccount()
{
  if (name)
  {
    free(name);
    name = NULL;
  }
  if (statusString)
  {
    free(statusString);
    statusString = NULL;
  }
}

LocalAccount *ContactAccount::getLocalAccount() const
{
  return localAccount;
}

Service *ContactAccount::getService() const
{
  return localAccount->getService();
}

char *ContactAccount::getName() const
{
  return name;
}

char *ContactAccount::getStatusString() const
{
  return statusString;
}

int ContactAccount::getStatus() const
{
  return status;
}

bool ContactAccount::isLogged() const
{
  return (status > 0);
}

Contact *ContactAccount::getContact() const
{
  return contact;
}

Group *ContactAccount::getGroup() const
{
  return contact->getGroup();
}

void ContactAccount::updateStatus(int status, char *statusString)
{
  this->status = status;
  if (this->statusString)
    free(this->statusString);
  this->statusString = statusString ? strdup(statusString): NULL;
  contact->updateStatusAccount(this);
}

void ContactAccount::login()
{
  updateStatus(1, "");
  contact->login();
}

void ContactAccount::logout()
{
  updateStatus(0, NULL);
  contact->logout();
}

void ContactAccount::setContact(Contact *c)
{
  contact = c;
}

/*** ContactAccountList ***/

ContactAccountList::ContactAccountList()
{
  first = NULL;
  next = NULL;
}

ContactAccountList::~ContactAccountList()
{
  if (first)
  {
    delete first;
    first = NULL;
  }
  if (next)
  {
    delete next;
    next = NULL;
  }
}

ContactAccount *ContactAccountList::getFirst() const
{
  return first;
}

ContactAccountList *ContactAccountList::getNext() const
{
  return next;
}

void ContactAccountList::add(ContactAccount *ca)
{
  if (first == NULL)
  {
    first = ca;
    next = new ContactAccountList();
  }
  else
    next->add(ca);
}

/*** Contact ***/

Contact::Contact(char *name, char *isIgnored, char *ignoreMessage)
{
  this->name = name ? strdup(name) : NULL;
  statusAccount = NULL;
  if (isIgnored == NULL)
    this->isIgnored = 0;
  else
    this->isIgnored = strtol(isIgnored, NULL, 10);
  this->ignoreMessage = ignoreMessage;
  contactAccounts = new ContactAccountList();
  group = NULL;
}

Contact::~Contact()
{
  if (name)
  {
    free(name);
    name = NULL;
  }
  if (statusAccount)
  {
    // Don't free or delete this one, it's just a reference
    statusAccount = NULL;
  }
  if (contactAccounts)
  {
    delete contactAccounts;
    contactAccounts = NULL;
  }
}

char *Contact::getName() const
{
  return name;
}

int Contact::getStatus() const
{
  if (!statusAccount)
    return 0;
  return statusAccount->getStatus();
}

char *Contact::getStatusString() const
{
  if (!statusAccount)
    return NULL;
  return statusAccount->getStatusString();
}

bool Contact::isLogged() const
{
  return (getStatus() > 0);
}

ContactAccountList *Contact::getAccounts() const
{
  return contactAccounts;
}

Group *Contact::getGroup() const
{
  return group;
}

void Contact::setName(char *name)
{
  if (this->name)
    free(this->name);
  this->name = name ? strdup(name) : NULL;
}

void Contact::updateStatusAccount(ContactAccount * updatedAccount)
{
  // Check all the contact accounts to find the most logged-in status
  // It is possible that we should use the preferred account, instead
  // but I'm not sure we know which one it is
  // We will start with the current status account as the best one
  // Note that when we implement moving accounts from one contact to another,
  // we should check if the account that is being removed is the current
  // status account, we should call this with NULL
  ContactAccount * bestAccount = statusAccount;
  ContactAccountList *accountList = contactAccounts;

  while (accountList && accountList->getFirst())
  {
    if (bestAccount == NULL)
    {
      // We had no status, so anything is better
      bestAccount = accountList->getFirst();
    }
    else
    {
      switch (bestAccount->getStatus())
      {
      case 0: // offline, any other status is higher than this
	if (accountList->getFirst()->getStatus() > 0)
	  bestAccount = accountList->getFirst();
	break;
      case 1: // online, no status is higher than this
	break;
      case 2: // away, only online is higher than this
	if (accountList->getFirst()->getStatus() == 1)
	  bestAccount = accountList->getFirst();
	break;
      }
    }
    accountList = accountList->getNext();
  }

  if (bestAccount == NULL)
  {
    // there were apparently no accounts to be found
    statusAccount = NULL;
  }
  else if (updatedAccount == NULL)
  {
    // a NULL argument means we just want the best status available to us
    statusAccount = bestAccount;
  }
  else if (statusAccount == NULL)
  {
    // we had no previous status, the new one is certainly better
    statusAccount = updatedAccount;
  }
  else if (updatedAccount->getStatus() == bestAccount->getStatus())
  {
    // if the new status is the same as the 'best' status, we want to use it
    // Because of the way we do this, the new status will never actually be
    // better, seeing as it would be one of the ones used to choose the best
    statusAccount = updatedAccount;
  }
  else if (updatedAccount == statusAccount)
  {
    // the new status must be lower than the best one, and we are updating the
    // account which was our previous best one, which means that we should
    // just use the best one, instead
    statusAccount = bestAccount;
  }
}

void Contact::login()
{
}

void Contact::logout()
{
}

void Contact::addAccount(ContactAccount *ca)
{
  contactAccounts->add(ca);
  ca->setContact(this);
}

void Contact::setGroup(Group *g)
{
  group = g;
}

/*** ContactList ***/

ContactList::ContactList()
{
  first = NULL;
  next = NULL;
}

ContactList::~ContactList()
{
  if (first)
  {
    delete first;
    first = NULL;
  }
  if (next)
  {
    delete next;
    next = NULL;
  }
}

Contact *ContactList::getFirst() const
{
  return first;
}

ContactList *ContactList::getNext() const
{
  return next;
}

void ContactList::add(Contact *c)
{
  if (first == NULL)
  {
    first = c;
    next = new ContactList();
  }
  else
    next->add(c);
}

void ContactList::del(char *name)
{
  if (first && !strcmp(first->getName(), name))
  {
    delete first;
    first = NULL;
  }
  else
    next->del(name);
}

Contact *ContactList::find(char *name) const
{
  if (!strcmp(first->getName(), name))
    return first;
  else
  {
    if (next == NULL)
      return NULL;
    else
      return next->find(name);
  }
}

/*** Group ***/

Group::Group(char *name)
{
  this->name = name ? strdup(name) : NULL;
  contacts = new ContactList();
}

Group::~Group()
{
  if (name)
  {
    free(name);
    name = NULL;
  }
  if (contacts)
  {
    delete contacts;
    contacts = NULL;
  }
}

char *Group::getName() const
{
  return name;
}

ContactList *Group::getContacts() const
{
  return contacts;
}

void Group::addContact(Contact *c)
{
  contacts->add(c);
  c->setGroup(this);
}

void Group::delContact(char *name)
{
  contacts->del(name);
}

Contact *Group::findContact(char *name) const
{
  return contacts->find(name);
}

/*** GroupList ***/

GroupList::GroupList()
{
  first = NULL;
  next = NULL;
}

GroupList::~GroupList()
{
  if (first)
  {
    delete first;
    first = NULL;
  }
  if (next)
  {
    delete next;
    next = NULL;
  }
}

Group *GroupList::getFirst() const
{
  return first;
}

GroupList *GroupList::getNext() const
{
  return next;
}

void GroupList::add(Group *g)
{
  if (first == NULL)
  {
    first = g;
    next = new GroupList();
  }
  else
    next->add(g);
}

void GroupList::del(char *name)
{
  if (first && !strcmp(first->getName(), name))
  {
    delete first;
    first = NULL;
  }
  else
    next->del(name);
}
	
Group *GroupList::find(char *name) const
{
  if (!strcmp(first->getName(), name))
    return first;
  else
  {
    if (next == NULL)
      return NULL;
    else
      return next->find(name);
  }
}

void GroupList::moveContact(char *name, Group *group1, Group *group2)
{
	
}
