/* File     : ebncurses.cpp
   Desc     : All the calls to the ncurses library are here
   Authors  : Karine Proot (kproot@nerim.net)                              */

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <iostream>
#include <string>
#include "GUIcomms.h"
#include "service.h"
#include "group.h"
#include "account.h"
#include "windows.h"
#include "ebncurses.h"

/*** EbNcurses ***/

EbNcurses::EbNcurses()
{
  services = new ServiceList();
  groups = new GroupList();
  localAccounts = new LocalAccountList();
  gui = new EbncGUI(services, groups, localAccounts);
}

EbNcurses::~EbNcurses()
{
  if (services)
  {
    delete services;
    services = NULL;
  }
  if (groups)
  {
    delete groups;
    groups = NULL;
  }
  if (localAccounts)
  {
    delete localAccounts;
    localAccounts = NULL;
  }
  if (gui)
  {
    delete gui;
    gui = NULL;
  }
}

EbncGUI *EbNcurses::getGUI() const
{
  return gui;
}

void EbNcurses::readCommand(char **command, int numParams) const
{
  int i = 0;
  /*** If the cookie is accepted, list all services ***/
  // "cookie_accepted"
  if (!strcmp(command[0], "cookie_accepted"))
  {
    char *cmd[] = {"list_services", NULL};
    send_gui_message(cmd, 1);
  }
  // "cookie_rejected"
  else if (!strcmp(command[0], "cookie_rejected"))
  {
    // TODO
  }
  // "list_service" "Service name" "Themed color" capabilities  
  else if (!strcmp(command[0], "list_service"))
  {
    Service *service = new Service(command[1], command[2], command[3]);
    services->add(service);
    services->find(command[1]);
  }
  // "list_service_actions" "Service name" "buddy" "Number of actions" "Action 1" "Action 2" ...
  // "list_service_actions" "Service name" "groupchat" "Number of actions" "Action 1" "Action 2" ...
  // "list_service_actions" "Service name" "group_users" "Number of actions" "Action 1" "Action 2" ...
  else if (!strcmp(command[0], "list_service_actions"))
  {
    Service *service = services->find(command[1]);
    ServiceAction *serviceAction = new ServiceAction(command[2], command[3]);
    for (i=4; i<numParams; i++)
    {
      Action *action = new Action(command[i]);
      serviceAction->addAction(action);
    }
    service->addServiceAction(serviceAction);
  }
  // "list_service_states" "Service name" "Number of states" "State 1" "State 2" ...
  else if (!strcmp(command[0], "list_service_states"))
  {
    Service *service = services->find(command[1]);
    service->setNumStates(command[2]);
    for (i=3; i<numParams; i++)
    {
      State *state = new State(command[i]);
      service->addState(state);
    }
  }
  /*** When services are listed, list all local accounts ***/
  // "list_services_done"
  else if (!strcmp(command[0], "list_services_done"))
  {
    char *cmd[] = {"list_local_accounts", NULL};
    send_gui_message(cmd, 1);
  }
  // "list_local_account" "Local username" "Service name"
  else if (!strcmp(command[0], "list_local_account"))
  {
    Service *service = services->find(command[2]);
    LocalAccount *localAccount = new LocalAccount(command[1], service);
    localAccounts->add(localAccount);
  }
  /*** When local accounts are listed, list all contacts ***/
  // "list_local_accounts_done"
  else if (!strcmp(command[0], "list_local_accounts_done"))
  {
    char *cmd[] = {"list_contacts", NULL};
    send_gui_message(cmd, 1);
  }
  // "list_group" "Group name"
  else if (!strcmp(command[0], "list_group"))
  {
    Group *group = new Group(command[1]);
    groups->add(group);
  }
  // "list_contact" "Group name" "Contact name" is_ignored "Ignore message"
  else if (!strcmp(command[0], "list_contact"))
  {
    Group *group = groups->find(command[1]);
    Contact *contact = new Contact(command[2], command[3], command[4]);
    group->addContact(contact);
  }
  // "list_account" "Group name" "Contact name" "Local buddy account" "Service name" "Account name" is_blocked
  else if (!strcmp(command[0], "list_account"))
  {
    Group *group = groups->find(command[1]);
    Contact *contact = group->findContact(command[2]);
    Service *service = services->find(command[4]);
    LocalAccount *localAccount = localAccounts->find(service, command[3]);
    ContactAccount *contactAccount = new ContactAccount(localAccount, command[5], command[6]);
    contact->addAccount(contactAccount);
    localAccount->addContactAccount(contactAccount);
  }
  /*** When contacts are listed, list all opened groupchats, unset away and show help ***/
  // "list_contacts_done"
  else if (!strcmp(command[0], "list_contacts_done"))
  {
    char *cmd1[] = {"list_group_chats", NULL};
    send_gui_message(cmd1, 1);
    char *cmd2[] = {"html_strip", "1", NULL};
    send_gui_message(cmd2, 2);
    char *cmd3[] = {"unset_away", NULL};
    send_gui_message(cmd3, 1);
    char *cmd4[] = {"message_hold", "0", NULL}; // !!! HAVE TO REMOVE THIS FAST !!!
    send_gui_message(cmd4, 2);
    gui->draw();
  }

  
  /*** Actions from other UI : groups, contacts, accounts ***/
  // "add_group" "Name"
  else if (!strcmp(command[0], "add_group"))
  {
    Group *group = new Group(command[1]);
    groups->add(group);
    gui->updateContacts();
  }
  // "add_contact" "Group" "Name"
  else if (!strcmp(command[0], "add_contact"))
  {
    Group *group = groups->find(command[1]);
    Contact *contact = new Contact(command[2], "0", NULL);
    group->addContact(contact);
    gui->updateContacts();
  }
  // "add_account" "Group" "Contact" "Local buddy account" "Service" "Account handle"
  else if (!strcmp(command[0], "add_account"))
  {
    Group *group = groups->find(command[1]);
    Contact *contact = group->findContact(command[2]);
    Service *service = services->find(command[4]);
    LocalAccount *localAccount = localAccounts->find(service, command[3]);
    ContactAccount *contactAccount = new ContactAccount(localAccount, command[5], command[6]);
    contact->addAccount(contactAccount);
    localAccount->addContactAccount(contactAccount);
    gui->updateContacts();
  }
  // "del_group" "Name"
  else if (!strcmp(command[0], "del_group"))
  {
    // TODO : Before deleting a group or contact, it must be empty
    groups->del(command[1]);
    gui->updateContacts();
  }
  // "del_contact" "Group" "Name"
  else if (!strcmp(command[0], "del_contact"))
  {
    // TODO : Before deleting a group or contact, it must be empty
    Group *group = groups->find(command[1]);
    group->delContact(command[2]);
    gui->updateContacts();
  }
  // "del_account" "Local handle" "Service" "Buddy handle"
  else if (!strcmp(command[0], "del_account"))
  {
    // TODO
  }
  // "rename_contact" "Group name" "Contact name" "New contact name"
  else if (!strcmp(command[0], "rename_contact"))
  {
    Group *group = groups->find(command[1]);
    Contact *contact = group->findContact(command[2]);
    contact->setName(command[3]);
    gui->updateContacts();
  }
  // "move_contact" "Group name" "Contact name" "New group name"
  else if (!strcmp(command[0], "move_contact"))
  {
    // TODO : To move a contact or account, the group or contact to which
    // they are moving must exist first.
    Group *group1 = groups->find(command[1]);
    Group *group2 = groups->find(command[3]);
    groups->moveContact(command[2], group1, group2);
    gui->updateContacts();
  }
  // "move_account" "Local handle" "Service" "Buddy handle" "New group name" "New contact name"
  else if (!strcmp(command[0], "move_account"))
  {
    // TODO
    // To move a contact or account, the group or contact to which
    // they are moving must exist first.
  }
  
  /*** Actions from other UI : ignoring ***/
  // "ignore_contact" "Group name" "Contact name" "Ignore message"
  else if (!strcmp(command[0], "ignore_contact"))
  {
    // TODO
  }
  // "unignore_contact" "Group name" "Contact name"
  else if (!strcmp(command[0], "unignore_contact"))
  {
    // TODO
  }
  
  /*** Actions from other UI : local accounts ***/
  // "add_local_account" "Local username" "Service name"
  else if (!strcmp(command[0], "add_local_account"))
  {
    // TODO
  }
  // "del_local_account" "Local username" "Service name"
  else if (!strcmp(command[0], "delete_local_account"))
  {
    // TODO
  }
  
  /*** Actions from other UI : away ***/
  // "set_away" "title" "body"
  else if (!strcmp(command[0], "set_away"))
  {
    // TODO
  }
  // "unset_away"
  else if (!strcmp(command[0], "unset_away"))
  {
    // TODO
  }
  
  /*** Dialogs ***/
  // "error_dialog" "tag code" "message_title" "message_body"
  else if (!strcmp(command[0], "error_dialog"))
  {
    // TODO
  }
  // "yesno_dialog" "tag code" "title" "message"
  else if (!strcmp(command[0], "yesno_dialog"))
  {
    // TODO
  }
  // "list_dialog" "tag code" "title" "message" "number of options" "option 1" "option 2" ...
  else if (!strcmp(command[0], "list_dialog"))
  {
    // TODO
  }
  // "text_dialog" "tag code" "current contents"
  else if (!strcmp(command[0], "text_dialog"))
  {
    // TODO
  }
  // "dialog_resolved" "tag code"
  else if (!strcmp(command[0], "dialog_resolved"))
  {
    // TODO
  }
  // "client_error" "error message"
  else if (!strcmp(command[0], "client_error"))
  {
    // TODO
  }

  /*** Status notification ***/
  // "buddy_logout" "Local buddy account" "Service name" "Buddy handle"
  else if (!strcmp(command[0], "buddy_logout"))
  {
    Service *service = services->find(command[2]);
    LocalAccount *localAccount = localAccounts->find(service, command[1]);
    ContactAccount *contactAccount = localAccount->findAccount(command[3]);
    contactAccount->logout();
    gui->updateContacts();
  }
  // "buddy_login" "Local buddy account" "Service name" "Buddy handle"
  else if (!strcmp(command[0], "buddy_login"))
  {
    Service *service = services->find(command[2]);
    LocalAccount *localAccount = localAccounts->find(service, command[1]);
    ContactAccount *contactAccount = localAccount->findAccount(command[3]);
    contactAccount->login();
    gui->updateContacts();
  }
  
  /*** Local account status ***/
  // "buddy_status" "Local buddy account" "Service name" "Buddy handle" state "Status string"
  else if (!strcmp(command[0], "buddy_status"))
  {
    Service *service = services->find(command[2]);
    LocalAccount *localAccount = localAccounts->find(service, command[1]);
    ContactAccount *contactAccount = localAccount->findAccount(command[3]);
    int state = strtol(command[4], NULL, 10);
    switch (state)
    {
    case 0:
      contactAccount->logout();
      break;
    case 1:
    case 2:
    default:
      contactAccount->updateStatus(state, command[5]);
      break;
    }
    gui->updateContacts();
    gui->showBuddyStatus(contactAccount, state, command[5]);
  }
  // "local_account_update" "Local account handle" "Service name" connected ready "Status string"
  else if (!strcmp(command[0], "local_account_update"))
  {
    Service *service = services->find(command[2]);
    LocalAccount *localAccount = localAccounts->find(service, command[1]);
    localAccount->update(command[3], command[4], command[5]);
    gui->updateAccounts();
  }

  /*** Sending and receiving messages ***/
  // "message_receive" "Group name" "Contact name" "Local handle" "Service name" "Buddy handle" "Message"
  else if (!strcmp(command[0], "message_receive"))
  {
    // unused???
    //Group *group = groups->find(command[1]);
    //Contact *contact = group->findContact(command[2]);
    Service *service = services->find(command[4]);
    LocalAccount *localAccount = localAccounts->find(service, command[3]);
    ContactAccount *contactAccount = localAccount->findAccount(command[5]);
    gui->receiveMessage(contactAccount, command[6]);
  }
  // "notify_3rdperson" "Group name" "Contact name" "Text"
  else if (!strcmp(command[0], "notify_3rdperson"))
  {
//     Group *group = groups->find(command[1]);
//     Contact *contact = group->findContact(command[2]);
//     gui->receive3rd(contact, command[3]);
  }
  // "message_send" "Group name" "Contact name" "Local handle" "Service name" "Buddy handle" "Message"
  else if (!strcmp(command[0], "message_send"))
  {
    // unused ???
    //Group *group = groups->find(command[1]);
    //Contact *contact = group->findContact(command[2]);
    Service *service = services->find(command[4]);
    LocalAccount *localAccount = localAccounts->find(service, command[3]);
    ContactAccount *contactAccount = localAccount->findAccount(command[5]);
    gui->sendMessage(contactAccount, command[6]);
  }
  
  /*** Queuing ***/
  // "message_hold" "1"/"0"
  else if (!strcmp(command[0], "message_hold"))
  {
    // TODO
  }
  // "held_message" time "Local handle" "Service" "Buddy handle" "Message"  
  // "held_sent_message" time "Local handle" "Service" "Buddy handle" "Message"
  // "held_3rdperson" time "Group" "Contact" "Message"
  // "held_messages_done"
  // "message_waiting"

  /*** Group chat ***/
  // "new_group_chat" "Local handle" "Service name" chat_id "Title"
  else if (!strcmp(command[0], "new_group_chat"))
  {
    Service *service = services->find(command[2]);
    LocalAccount *localAccount = localAccounts->find(service, command[1]);
    gui->openGroupchat(localAccount, strtol(command[3], NULL, 10), command[4]);
  }
  // "list_group_user" chat_id "Username"
  // "list_group_chat_done"
  // "group_chat_send" chat_id "Message as sent"
  else if (!strcmp(command[0], "group_chat_send"))
  {
    gui->sendGroupMessage(strtol(command[1], NULL, 10), command[2]);
  }
  // "group_chat_recv" chat_id "Sending handle" "Message"
  else if (!strcmp(command[0], "group_chat_recv"))
  {
    gui->receiveGroupMessage(strtol(command[1], NULL, 10), command[2], command[3]);
  }
  // "group_chat_3rdperson" chat_id "Message"
  else if (!strcmp(command[0], "group_chat_3rdperson"))
  {
    gui->receiveGroup3rd(strtol(command[1], NULL, 10), command[2]);
  }
  // "group_chat_update" chat_id "Title"
  // "group_chat_joined" chat_id "handle"
  else if (!strcmp(command[0], "group_chat_joined"))
  {
    gui->joinedGroup(strtol(command[1], NULL, 10), command[2]);
  }
  // "group_chat_left" chat_id "handle"
  else if (!strcmp(command[0], "group_chat_left"))
  {
    gui->leftGroup(strtol(command[1], NULL, 10), command[2]);
  }
  // "close_group_chat" chat_id
  else if (!strcmp(command[0], "close_group_chat"))
  {
    gui->closeGroup(strtol(command[1], NULL, 10));
  }

  /*** Preferences ***/
  // "list_pref_page" "page name" "page title"
  // "list_subpages" "page name"
  // "list_subpage" "page name" "subpage name" "subpage title"
  // "list_subpages_done"
  // "list_components" "page name"
  // "list_component" "page name" "type" "name" "title" "value"
  // "option_data" "page name" "component name" num_options "Option 0" "Option 1" ...
  // "list_components_done"
  // "list_pref_page_done"

  /*** Actions ***/
  // "list_actions" "general" num_actions "action 1" "action 2" "action 3" ...
  // "list_actions" "group" num_actions "action 1" "action 2" ...
  // "list_actions" "contact" num_actions "action 1" "action 2" ...
  // "list_actions" "buddy_generic" num_actions "action 1" "action 2" ...
  // "list_actions_done"

  /*** Persistent data storage ***/
  // "return_data" "name" "value" "keyword" [keyword parameters]  
}
