/***************************************************************************
                       util.c - Global utility functions
                             -------------------
                     (C) 2002 by the Everybuddy team
                            www.everybuddy.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#ifdef __MINGW32__
#include <winsock2.h>
#endif
#include "elist.h"
#include "gui_comms.h"
#include "debug.h"
#include "dialog.h"
#include "groupchat.h"
#include "util.h"
#include "globals.h"
#include "message_parse.h"


// EB global utility functions

void eb_sign_on_all(void)
{
  EList * n;

  eb_debug(DBG_CORE, "signing on all\n");

  for(n=local_accounts; n!=NULL; n=n->next)
  {
    eb_local_account * account=(eb_local_account *)n->data;

    eb_debug(DBG_CORE, "Examining %s\n", account->handle);
    if(account->connected) { eb_debug(DBG_CORE, "Skipping %s\n", account->handle); continue; }

    if(account->service->sc->login)
    { eb_debug(DBG_CORE, "Logging in\n"); account->service->sc->login(account); }

    eb_debug(DBG_CORE, "Finished login\n");
  }

  eb_debug(DBG_CORE, "Finished sign on all\n");
}

void eb_sign_off_all(void)
{
  EList * n;

  for(n=local_accounts; n!=NULL; n=n->next)
  {
    eb_local_account * account=(eb_local_account *)n->data;

    if(!account->connected) { continue; }

    eb_log_out_account(account);
  }
}

void eb_log_out_account(eb_local_account * account)
{
  EList * n;

  if(account->service->sc->logout)
  { account->service->sc->logout(account); }

  for(n=account->buddies; n!=NULL; n=n->next)
  {
    eb_account * acc=(eb_account *)n->data;

    if(acc->status!=EB_ACCOUNT_OFFLINE)
    { eb_buddy_logout(acc); }
  }

  eb_local_account_update(account);
}

eb_group * eb_get_group(char * name)
{
  EList * n;

  for(n=groups; n!=NULL; n=n->next)
  {
    eb_group * group=(eb_group *)n->data;

    if(!strcmp(group->name, name)) { return group; }
  }

  return NULL;
}

eb_contact * eb_get_contact(char * group_name, char * contact_name)
{
  EList * n;
  eb_group * group=eb_get_group(group_name);
  if(group==NULL) { return NULL; }

  for(n=group->contacts; n!=NULL; n=n->next)
  {
    eb_contact * contact=(eb_contact *)n->data;

    if(!strcmp(contact->name, contact_name)) { return contact; }
  }

  return NULL;
}

eb_account * eb_get_account(char * buddy_of, char * service_name, char * handle)
{
  EList * n;
  eb_local_account * la=eb_get_local_account(buddy_of, service_name);

  if(la==NULL) { return NULL; }

  for(n=la->buddies; n!=NULL; n=n->next)
  {
    eb_account * account=(eb_account *)n->data;

    if(!strcmp(account->handle, handle))
    {
      return account;
    }
  }

  return NULL;
}

eb_local_account * eb_get_local_account(char * handle, char * service)
{
  EList * n;

  for(n=local_accounts; n!=NULL; n=n->next)
  {
    eb_local_account * acc=(eb_local_account *)n->data;

    if(!strcmp(acc->handle, handle) && !strcmp(acc->service_name, service))
    {
      return acc;
    }
  }

  return NULL;
}

eb_service * eb_get_service(char * name)
{
  EList * n;

  for(n=services; n!=NULL; n=n->next)
  {
    eb_service * thisserv=(eb_service *)n->data;

    if(!strcmp(thisserv->name, name)) { return thisserv; }
  }

  return NULL;
}

eb_group_chat * eb_get_group_chat(int id)
{
  EList * n;

  for(n=group_chats; n!=NULL; n=n->next)
  {
    eb_group_chat * chat=(eb_group_chat *)n->data;

    if(chat->id==id) { return chat; }
  }

  return NULL;
}

eb_account * eb_get_message_account(eb_contact * cont, char * buddy_of, char * service, char * handle, int capabilities)
{
  EList * l;
  eb_account * acc=NULL;
  eb_account * backup=NULL;
  
  for(l=cont->accounts; l!=NULL; l=l->next)
  {
    eb_account * thisacc=(eb_account *)l->data;

    if(thisacc->status==EB_ACCOUNT_OFFLINE
      && !(thisacc->buddy_of->service->capabilities & SERVICE_CAN_OFFLINE))
    { continue; }

    if(backup==NULL || (backup->status==EB_ACCOUNT_OFFLINE && thisacc->status!=EB_ACCOUNT_OFFLINE))
    { backup=thisacc; }

    if(!strcmp(thisacc->buddy_of->handle, buddy_of)
     && !strcmp(thisacc->buddy_of->service_name, service)
     && !strcmp(thisacc->handle, handle)
     && (capabilities==0 || thisacc->buddy_of->service->capabilities & capabilities))
    {
      acc=thisacc;
      break;
    }

    if(!strcmp(thisacc->buddy_of->service_name, service)
      || thisacc->buddy_of->service->capabilities & capabilities)
    {
      acc=thisacc; // if there's an exact match later on, it will be overridden, but if not, use it!
    }
  }

  return (acc!=NULL)?(acc):(backup);
}

eb_group * eb_add_group(char * name)
{
  char * add_command[]={"add_group", name};
  eb_registry_key * key;
  eb_group * group;
  char * mangled_name=eb_escape_reg_key_name(name);

  group=(eb_group *)malloc(sizeof(eb_group));
  key=eb_get_key(registry, "contact_list");
  key=eb_get_key(key, mangled_name);
  free(mangled_name);
  eb_put_value(key, "key_prop", ""); // just holds the key open

  group->name=strdup(name);
  group->contacts=NULL;
  group->config_key=key;
  group->locked=0;

  groups=e_list_append(groups, group);

  eb_gui_broadcast(add_command, 2);

  return group;
}

void eb_del_group(eb_group * group)
{
  char * del_command[]={"del_group", group->name};
  EList * n;
  EList * to_destroy=NULL;
  eb_registry_key * key;

  for(n=group->contacts; n!=NULL; n=n->next)
  {
    eb_contact * contact=(eb_contact *)n->data;

    to_destroy=e_list_append(to_destroy, contact);
  }

  for(n=to_destroy; n!=NULL; n=n->next)
  {
    eb_contact * contact=(eb_contact *)n->data;

    eb_del_contact(contact);
  }

  e_list_free(to_destroy);

  key=eb_get_key(registry, "contact_list");
  eb_del_key(key, group->config_key);

  groups=e_list_remove(groups, group);

  eb_gui_broadcast(del_command, 2);

  free(group->name);
  free(group);
}

eb_contact * eb_add_contact(eb_group * group, char * name)
{
  char * add_command[]={"add_contact", group->name, name};
  eb_registry_key * key;
  eb_contact * contact;
  char * mangled_name=eb_escape_reg_key_name(name);

  key=eb_get_key(group->config_key, mangled_name);
  free(mangled_name);

  eb_put_value(key, "key_prop", ""); // just holds the key open

  contact=(eb_contact *)malloc(sizeof(eb_contact));

  contact->name=strdup(name);
  contact->group=group;
  contact->default_service=strdup("");
  contact->accounts=NULL;
  contact->config_key=key;
  contact->attribs=NULL;
  contact->next_away_msg=0;
  contact->last_message=0;
  contact->log_begins=0;
  contact->logfile=0;
  contact->ignore=NULL;
  contact->locked=0;

  group->contacts=e_list_append(group->contacts, contact);

  eb_gui_broadcast(add_command, 3);

  return contact;
}

void eb_rename_contact(eb_contact * contact, char * newname)
{
  char * notify_cmd[]={"rename_contact", contact->group->name, contact->name, newname};

  if(eb_get_contact(contact->group->name, newname)!=NULL)
  {
    eb_show_error("Cannot rename this contact, because a contact of that name already exists", "Cannot rename contact");
    return;
  }

  free(contact->config_key->name);
  contact->config_key->name=eb_escape_reg_key_name(newname);

  contact->name=strdup(newname);

  eb_gui_broadcast(notify_cmd, 4);
  free(notify_cmd[2]);
}

void eb_move_contact(eb_contact * contact, eb_group * dest)
{
  char * notify_cmd[]={"move_contact", contact->group->name, contact->name, dest->name};
  eb_contact * existing=eb_get_contact(dest->name, contact->name);

  if(existing!=NULL)
  {
    // There's one there already? No problem! Just merge them...
    EList * n;
    EList * n2=NULL;

    // This little reverse maneuver is so that buddies end up in the new
    // contact in the correct order, to preserve preferredness

    for(n=contact->accounts; n!=NULL; n=n->next) { n2=n; }

    if(n2==NULL) // no accounts in this contact
    { eb_del_contact(contact); return; }

    for(n=n2; n!=NULL; n=n->prev)
    {
      eb_account * acc=(eb_account *)n->data;

      eb_move_account(acc, existing);
    }

    // don't delete the contact afterwards, the last move_account() does it for us!

    return;
  }

  eb_gui_broadcast(notify_cmd, 4);

  contact->group->config_key->subkeys=e_list_remove(contact->group->config_key->subkeys, contact->config_key);
  dest->config_key->subkeys=e_list_append(dest->config_key->subkeys, contact->config_key);

  contact->group->contacts=e_list_remove(contact->group->contacts, contact);
  dest->contacts=e_list_append(dest->contacts, contact);
  contact->group=dest;
}

void eb_del_contact(eb_contact * contact)
{
  char * del_command[]={"del_contact", contact->group->name, contact->name};

  if(contact->accounts!=NULL)
  {
    eb_show_error("Cannot delete a non-empty contact", "Everybuddy error");
    return;
  }

  contact->group->contacts=e_list_remove(contact->group->contacts, contact);

  eb_gui_broadcast(del_command, 3);

  eb_del_key(registry, contact->config_key);

  free(contact->name);
  free(contact->default_service); // hmm, getting obsolete...
  free(contact);
}

eb_account * eb_add_account(eb_contact * contact, eb_local_account * buddy_of, eb_account * account)
{
  char * add_command[]={"add_account", contact->group->name, contact->name, buddy_of->handle, buddy_of->service_name, account->handle};
  char * tmp;
  char * tmp2;
  eb_registry_key * key;

  key=contact->config_key;

  tmp=(char *)malloc(strlen(buddy_of->handle)+strlen(buddy_of->service_name)+strlen(account->handle)+3);
  strcpy(tmp, buddy_of->handle);
  strcat(tmp, "_");
  strcat(tmp, buddy_of->service_name);
  strcat(tmp, "_");
  strcat(tmp, account->handle);

  tmp2=eb_escape_reg_key_name(tmp);
  key=eb_get_key(key, tmp2);

  free(tmp);
  free(tmp2);

  eb_put_value(key, "handle", account->handle);
  eb_put_value(key, "service", buddy_of->service_name);
  eb_put_value(key, "buddy_of", buddy_of->handle);

  account->contact=contact;
  account->config_key=key;
  account->buddy_of=buddy_of;
  account->protocol_data=NULL;
  account->blocked=0;
  account->locked=0;
  account->status=EB_ACCOUNT_OFFLINE;

  contact->accounts=e_list_append(contact->accounts, account);
  buddy_of->buddies=e_list_append(buddy_of->buddies, account);

  eb_gui_broadcast(add_command, 6);

  return account;
}

eb_account * eb_add_account_plus(char * gname, char * cname, eb_local_account * buddy_of, char * aname)
{
  eb_group * g=eb_get_group(gname);
  eb_contact * c;
  eb_account * a;
  
  if(g==NULL)
  {
    g=eb_add_group(gname);
    c=eb_add_contact(g, cname);
  } else {
    c=eb_get_contact(gname, cname);
    if(c==NULL)
    { c=eb_add_contact(g, cname); }
  }
  
  a=(eb_account *)malloc(sizeof(eb_account));
  a->handle=strdup(aname);
  a->status_string=strdup("Offline");
  a->status=EB_ACCOUNT_OFFLINE;
  
  eb_add_account(c, buddy_of, a);
  
  return a;
}

void eb_move_account(eb_account * account, eb_contact * dest)
{
  char * notify_cmd[]={"move_account", account->buddy_of->handle, account->buddy_of->service_name, account->handle, dest->group->name, dest->name};

  account->contact->config_key->subkeys=e_list_remove(account->contact->config_key->subkeys, account->config_key);
  dest->config_key->subkeys=e_list_insert(dest->config_key->subkeys, dest->config_key->subkeys, account->config_key);

  account->contact->accounts=e_list_remove(account->contact->accounts, account);
  dest->accounts=e_list_insert(dest->accounts, dest->accounts, account);

  eb_gui_broadcast(notify_cmd, 6);

  if(account->contact->accounts==NULL)
  {
    eb_del_contact(account->contact);
  }

  account->contact=dest;
}

void eb_del_account(eb_account * account)
{
  char * del_command[]={"del_account", account->buddy_of->handle, account->buddy_of->service_name, account->handle};

  account->contact->accounts=e_list_remove(account->contact->accounts, account);
  account->buddy_of->buddies=e_list_remove(account->buddy_of->buddies, account);

  eb_gui_broadcast(del_command, 4);

  eb_del_key(registry, account->config_key);

  free(account->handle);
  free(account->status_string);
  
  if(account->contact->accounts==NULL) { eb_del_contact(account->contact); }
  
  free(account);
}

void eb_set_away(char * title, char * body)
{
  char * notify_command[]={"set_away", NULL, NULL};
  EList * n;
  eb_registry_key * key;
  char * mangled_short;
  char * prev_long;

  if(away_msg!=NULL)
  { free(away_msg); free(away_short); }

  key=eb_get_key(registry, "config/away_msgs");
  mangled_short=eb_escape_reg_key_name(title);
  key=eb_get_key(key, mangled_short);
  prev_long=eb_get_value(key, "message");
  
  if(body[0]=='\0' && prev_long[0]!='\0')
  {
    away_msg=strdup(prev_long);
  } else {
    eb_put_value(key, "message", body);
    away_msg=strdup(body);
  }
  away_short=strdup(title);
  free(mangled_short);
  
  for(n=local_accounts; n!=NULL; n=n->next)
  {
    eb_local_account * acc=(eb_local_account *)n->data;

    if(acc->ready)
    {
      if(acc->service->sc->set_away)
      { acc->service->sc->set_away(acc, away_short, away_msg); }
    }
  }

  for(n=groups; n!=NULL; n=n->next)
  {
    EList * n2;
    for(n2=((eb_group *)n->data)->contacts; n2!=NULL; n2=n2->next)
    {
      eb_contact * contact=(eb_contact *)n2->data;
      contact->next_away_msg=0;
    }
  }

  notify_command[1]=away_short;
  notify_command[2]=away_msg;
  eb_gui_broadcast(notify_command, 3);
}

void eb_unset_away(void)
{
  char * notify_command[]={"unset_away"};
  EList * n;

  if(away_msg!=NULL)
  { free(away_msg); free(away_short); away_msg=NULL; away_short=NULL; }

  for(n=local_accounts; n!=NULL; n=n->next)
  {
    eb_local_account * acc=(eb_local_account *)n->data;

    if(acc->ready)
    {
      if(acc->service->sc->unset_away)
      { acc->service->sc->unset_away(acc); }
    }
  }

  eb_gui_broadcast(notify_command, 1);
}

char * eb_escape_reg_key_name(char * name)
{
  char * retval=(char *)malloc(2*strlen(name)+1);
  int rd=0, wr=0;

  while(1)
  {
    if(name[rd]=='/')
    {
      retval[wr++]='\\';
      retval[wr]='|';
    } else if(name[rd]=='\\') {
      retval[wr++]='\\';
      retval[wr]='\\';
    } else {
      retval[wr]=name[rd];
      if(retval[wr]=='\0') { break; }
    }
    rd++;
    wr++;
  }

  return retval;
}

char * eb_unescape_reg_key_name(char * name)
{
  char * retval=strdup(name);
  int rd=0, wr=0;

  while(1)
  {
    if(name[rd]=='\\')
    {
      rd++;
      if(name[rd]=='|') { retval[wr]='/'; }
      if(name[rd]=='\\') { retval[wr]='\\'; }
    } else {
      retval[wr]=name[rd];
      if(name[rd]=='\0') { break; }
    }
    rd++;
    wr++;
  }

  return retval;
}

char * eb_ascii2utf(unsigned char * s)
{
  char * retval=(char *)malloc(2*strlen(s)+1);
  int rpos=0, wpos=0;
  
  while(s[rpos]!='\0')
  {
    if(s[rpos]<128)
    {
      retval[wpos++]=s[rpos];
    } else {
      retval[wpos++]=(char)((s[rpos] >> 6) | 192);
      retval[wpos++]=(char)((s[rpos] & 63) | 128);
    }
    rpos++;
  }
  
  retval[wpos]='\0';
  
  return retval;
}

char * eb_utf2ascii(unsigned char * s)
{
  char * retval=strdup(s);
  int rpos=0, wpos=0;
  
  while(s[rpos]!='\0')
  {
    if(s[rpos]<128)
    {
      retval[wpos++]=s[rpos];
    } else {
      if(s[rpos+1]=='\0') { break; }
      retval[wpos++]=(char)((s[rpos] << 6) | (s[rpos+1] & 63));
      rpos++;
    }
    rpos++;
  }
  
  retval[wpos]='\0';
  
  return retval;
}

void eb_hold_messages(int hold)
{
  char * notify_cmd[]={"message_hold", "0"};

  if(!hold)
  {
    
    if(!eb_holding_messages)
    { return; }

    eb_gui_get_held_messages(NULL, NULL, NULL, 0);
    eb_holding_messages=0;
    eb_gui_broadcast(notify_cmd, 2);
  } else {
    if(eb_holding_messages) { return; }

    eb_holding_messages=1;
    notify_cmd[1]="1";
    eb_gui_broadcast(notify_cmd, 2);
  }
}

int eread(int fd, void * data, int len)
{
  int pos=0;
  unsigned char * d=(unsigned char *)data;
  
  while(pos<len)
  {
    int result=read(fd, d+pos, len-pos);
    if(result==0) { printf("Read %d chars\n", pos); return pos; }
    if(result<0) { printf("Whu-oh\n"); return -1; }
    pos+=result;
    if(pos<len) { printf("Partial read (%d/%d bytes)\n", pos, len); }
  }
  
  return pos;
}

int ewrite(int fd, void * data, int len)
{
  int pos=0;
  unsigned char * d=(unsigned char *)data;
  
  while(pos<len)
  {
    int result=write(fd, d+pos, len-pos);
    if(result==0) { printf("Wrote %d chars\n", pos); return pos; }
    if(result<0) { printf("Whu-oh\n"); return -1; }
    pos+=result;
    if(pos<len) { printf("Partial write (%d/%d bytes)\n", pos, len); }
  }
  
  return pos;
}

char * eb_get_filename(char * subdir)
{
  char * buf=(char *)malloc(strlen(eb_config_dir)+strlen(subdir)+32);
  int i=0;
  time_t now=time(NULL);
  
  sprintf(buf, "%s%s", eb_config_dir, subdir);
  mkdir(buf, 0700);
  
  while(1)
  {
    int fd;
    
    sprintf(buf, "%s%s/%lu_%d", eb_config_dir, subdir, now, i);
    if((fd=open(buf, O_RDONLY))<0)
    {
      if(errno==ENOENT) { return buf; } else { perror("eb_get_filename failed"); free(buf); return NULL; }
    }
    close(fd);
    i++;
  }
}

char * eb_url_encode(char * s)
{
  char * rptr;
  char * wptr;
  char * retval;

  wptr=retval=(char *)malloc(sizeof(char)*strlen(s)*3+1);
  rptr=s;

  while(1)
  {
    if(*rptr=='\0')
    { *wptr='\0'; break; }
    if(!(isalpha(*rptr) || isdigit(*rptr)))
    {
      sprintf(wptr, "%%%2x", (int)(*rptr));

      rptr++;
      wptr+=3;
      continue;
    }

    *wptr=*rptr;
    wptr++;
    rptr++;
  }

  return retval;
}

void eb_save_config(void)
{
  char * tmp_name=malloc(strlen(eb_config_dir)+32);
  strcpy(tmp_name, eb_config_dir);
  strcat(tmp_name, "everybuddy.conf");
  eb_save_key(registry, tmp_name);
  free(tmp_name);
}
