/***************************************************************************
                          stream.c - UI streaming
                             -------------------
                     (C) 2002 by the Everybuddy team
                            www.everybuddy.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#ifdef __MINGW32__
#include <winsock2.h>
#define EWOULDBLOCK    WSAEWOULDBLOCK
#endif
#include "elist.h"
#include "plugin_api.h"
#include "stream.h"

char * eb_stream_errors[]={ 	// Indexes in this array are EB_STREAM_ERR_xxx constants
  "Success",
  "Transfer aborted",
  "Read error"};

int eb_stream_next=0;

static void eb_stream_out(void * data, int fd, int conditions);
static void eb_stream_buffer_write_ready(void * data, int fd, int conditions);

eb_stream * eb_stream_file(eb_gui * gui, char * filename, int direction, void (*cb)(int success, void * data), void * cb_data, int id)
{
  int fd;

  if(direction==EB_STREAM_OUT)
  {
    fd=open(filename, O_RDONLY);
  } else {
    fd=open(filename, O_WRONLY|O_CREAT|O_TRUNC, S_IRUSR|S_IWUSR);
  }

  if(fd<0)
  {
    return NULL;
  } else {
    return eb_stream_fd(gui, fd, direction, cb, cb_data, id);
  }
}

static void eb_stream_set(void * data, int fd, int conditions)
{
  *((int *)data)=1;
}

eb_stream * eb_stream_fd(eb_gui * gui, int fd, int direction, void (*cb)(int success, void * data), void * cb_data, int id)
{
  eb_stream  * stream=(eb_stream *)malloc(sizeof(eb_stream));

  stream->gui=gui;
  stream->fd=fd;
  stream->direction=direction;
  stream->cb=cb;
  stream->cb_data=cb_data;
  stream->id=id;
  stream->wready=0;
  stream->buffer=NULL;
  if(direction==EB_STREAM_OUT)
  {
    stream->wr_tag=eb_input_add(gui->write_fd, EB_INPUT_WRITE, eb_stream_set, &(stream->wready));
    stream->rd_tag=eb_input_add(fd, EB_INPUT_READ, eb_stream_out, stream);
  } else {
    stream->wr_tag=stream->rd_tag=-1;
  }

  gui->streams=e_list_append(gui->streams, stream);

  return stream;
}

eb_stream * eb_stream_buffer(eb_gui * gui, char * buffer, int len, int direction, void (*cb)(int error, void * data), void * cb_data, int id)
{
  eb_stream  * stream=(eb_stream *)malloc(sizeof(eb_stream));
  
  stream->gui=gui;
  stream->fd=-1;
  stream->direction=direction;
  stream->cb=cb;
  stream->id=id;
  stream->wready=0;
  stream->wr_tag=stream->rd_tag=-1;
  if(direction==EB_STREAM_OUT)
  {
    stream->wr_tag=eb_input_add(gui->write_fd, EB_INPUT_WRITE, eb_stream_buffer_write_ready, stream);
  }
}

static void eb_stream_buffer_write_ready(void * data, int fd, int conditions)
{
  eb_stream * stream=(eb_stream *)data;
  
  
}

void eb_stream_cleanup(eb_stream * stream, int err)
{
  char buf[16], buf2[16];
  char * notify_cmd[]={"stream_ends", buf, buf2};

  sprintf(buf, "%d", stream->id);
  sprintf(buf2, "%d", err);
  
  if(err)
  { notify_cmd[0]=(stream->direction==EB_STREAM_IN)?("up_stream_error"):("down_stream_error"); }

  if(stream->direction==EB_STREAM_OUT || err!=EB_STREAM_SUCCESS)
  { eb_gui_send(stream->gui, notify_cmd, err?3:2); }

  if(stream->cb!=NULL)
  { stream->cb(err, stream->cb_data); }

  close(stream->fd);

  if(stream->rd_tag>=0)
  { eb_input_remove(stream->rd_tag); }
  if(stream->wr_tag>=0)
  { eb_input_remove(stream->wr_tag); }
  stream->gui->streams=e_list_remove(stream->gui->streams, stream);
  free(stream);
}

void eb_stream_abort(eb_stream * stream)
{
  eb_stream_cleanup(stream, EB_STREAM_ABORT);
}

static void eb_stream_out(void * data, int fd, int conditions)
{
  eb_stream * stream=(eb_stream *)data;
  char buf[4096];
  char tb[16];
  int len;

  if(!stream->wready) { return; }
  stream->wready=0;

  sprintf(tb, "%d", stream->id);

  len=read(stream->fd, buf, 4096);
  if(len<0)
  {
    if(errno==EAGAIN || errno==EWOULDBLOCK) { return; } // bizarre. It said ready for reading!
    // Otherwise, nasty error
    perror("Read failed");
    eb_stream_cleanup(stream, EB_STREAM_IO);
  } else if(len==0) {
    eb_stream_cleanup(stream, EB_STREAM_SUCCESS);
  } else {
    int lengths[]={5, strlen(tb), len};
    char * data_cmd[]={"sdata", tb, buf};
    eb_gui_send_data(stream->gui, data_cmd, lengths, 3);
  }
}

eb_stream * eb_stream_get(eb_gui * gui, int id, int direction)
{
  EList * n;
  
  for(n=gui->streams;  n!=NULL; n=n->next)
  {
    eb_stream * s=(eb_stream *)n->data;
    if(s->id==id && s->direction==direction) { return s; }
  }
  return NULL;
}

void eb_stream_incoming(eb_stream * stream, void * data, int len)
{
  int pos=0;
  while(pos<len)
  {
    int rv=write(stream->fd, data, len-pos);
    if(rv<0) { eb_stream_cleanup(stream, EB_STREAM_IO); }
    pos+=rv;
  }
}

void eb_stream_finished(eb_stream * stream)
{
  eb_stream_cleanup(stream, EB_STREAM_SUCCESS);
}
