// services.h

#ifndef __SERVICES_H__
#define __SERVICES_H__

#include "contactlist.h"
#include "registry.h"
#include "actions.h"
#include "groupchat.h"

// Note - anything marked as CONDEMNED is scheduled to be removed from the
// codebase as soon as I finish my latest tinker and have the time to take
// the (usually useless) references to it from all modules

typedef struct _eb_service_callbacks {
        // ANY of the below can be set to NULL if not applicable
        // (all NULLs will give the same effect as the stub plugin -
        // pretty useless, but it won't segfault)


        /*callback that will establish connection to server, and set up
          any timers and listeners necessary */
        void (*login) (eb_local_account *account);

        /*callback to disable any timers and disconnect from server*/
        void (*logout) (eb_local_account *account);

        /* send message from one account to another, return the message as the user will see it
           (if you return something other than the original message pointer, please be sure
           to free() the original, and be aware that whatever you return will also be
           free()ed). Return NULL if you do not want the user to be notified of a sent message
           at all (eg IRC's /me, where a 3rd-person notification will be sent instead) */
        char * (*send_im) (eb_account *account_to,
                         char *message);

	/* send a data message, which encompasses everything from inline images to file transfer.
	   The arguments correspond to values in GUI_SPEC, apart from the filename, which is
	   the location of the data to send (by the time this function is called, the stream has
	   been completed). Note that reported_length is what the UI sent, not what stat() says. */
	
	void (*send_data_message)(eb_account *account_to, char * filename, char * contenttype, char * disposition, int reported_length);

        /* send a typing notification - the return value is the number
           of seconds to wait before sending again
        */
        int (*send_typing) (eb_account *account_to);

        /* reads local account information from the registry key supplied, which
         can be organised as the service wishes. It is, however, courteous to
         prefix all your plugin-specific config with something, so another module
         for the same service doesn't clash. */
        void (*setup_local_account) (eb_local_account * account);

        /* Free all the protocol-specific bits */
        void (*release_local_account) (eb_local_account * account);

        /* reads service-specific contact information from registry
          (the key is linked to by the account object) */
        void (*setup_buddy_account) (eb_account * account);

        /* Free all protocol-specific bits without removing the buddy */
        void (*release_buddy_account) (eb_account * account);

        /*Sets the online status for the given account*/
        void (*set_current_state)(eb_local_account * account, char * state);

        /*Sets as away. Both the short and long descriptions of the reason are supplied
         - this function MAY receive more than one successive call WITHOUT an intervening
         call to unset_away()*/
        void (*set_away)(eb_local_account * account, char * short_msg, char * long_msg);

        /*Unsets a previously-set "away" state, and restores the account to its previous
          state - you may want to store this when set_away() is called, but do beware that
          you may already be set as away because of a previous call - unset_away() should
          restore you to the state you were in before the first call the set_away()

           Another thing to be sure to take care of - if the account isn't "ready" when
           the set-as-away happens, it doesn't get set_away() called, so unset_away()
           can come "out of the blue"*/
        void (*unset_away)(eb_local_account * account);

        /*Checks validity (used in account editor) - return NULL if OK,
          and if not, a string saying why

          CONDEMNED - do not bother implementing, it won't be used... */
        char * (*check_login)(char * login, char * pass);

        /*Informs the service of an account that it needs to manage*/
        void (*add_user)(eb_local_account * buddy_of, eb_account * account);

        /*Notifies the service that it doesn't need to track an account,
          and cleans up the protocol-specific bits of the object*/
        void (*del_user)(eb_account * account);

        /*Block the user at a protocol level. If this function does anything, it should
          set the "blocked" attribute of the eb_account to true*/
        void (*block_user)(eb_account * account);

        /*Notifies the service that the user is no longer to be blocked. If this function
          does anything, it should set the "blocked" attribute to false (zero)*/
        void (*unblock_user)(eb_account * account);

        /*Notifies the service that an account's group has changed
          can be used to inform the server of the group change. */
        void (*change_group)(eb_account * account, char *new_group);


        /*set the idle time */

	/*send a message to a chat room*/

	char * (*group_chat_send)(eb_group_chat * chat, char * message);
	// Return values as for ordinary IM send

	/*these are used to join and leave a chat room */

	void (*join_group_chat)(eb_local_account * account, char * name);
	void (*leave_group_chat)(eb_group_chat * chat);

	/*this is to invite somebody into the chat room*/

	void (*group_chat_invite)( eb_group_chat * room, char * user);

	/* gets called for each account a contact has when a chat window is closed

	   AS YET UNUSED - don't rely on its existence...
	 */

	void (*terminate_chat)(eb_account * account );

} eb_service_callbacks;

/*
  What's missing from the original EB plugin API:

  query_service_callbacks() - now obsolete as we have a plugin init function.
    The service itself should initialise the callback structure in the
    eb_service_info structure

  eb_new_account() - not necessary - the core is perfectly capable of creating
    a NULLed-out eb_account structure with the correct handle and service pointer
    - further processing should be carried out by add_user()

  get_prefs() - a new prefs system has been introduced, and preference groups can be added
    to the prefs list by the plugin's init() function

  is_suitable() - remote accounts are now listed by local account, not by service,
    rendering this redundant

  read_local_account_config() - The registry deprecates this

  add_importers() - this was neither used nor implemented by anything at all!

  get_status_string() - now part of the buddy account object

  invitation responses - it's the service's responsibility to put up a yes/no dialog if applicable.
    the core should only ever know about a group chat once you join it

  a service ID - this has been replaced by a pointer to the eb_service structure itself
*/


#define SERVICE_CAN_OFFLINE   0x1
#define SERVICE_CAN_GROUPCHAT 0x2
#define SERVICE_CAN_NAME_CHAT 0x4
#define SERVICE_CAN_SEND_DATA 0x8
#define SERVICE_CAN_BLOCK     0x10
// For services that don't support the concept of blocking a user, EB will
// send an explanatory "you are being ignored" message in response to any
// message it receives to a blocked account. This avoids puzzling other users!

typedef struct _eb_service {
  char * name;
  char * description;
  eb_service_callbacks * sc;

  // Information the UI will want to know
  int capabilities;
  char * color; // in HTML form, eg AIM is "#0000dd"
  EList * states; // a list of char *s

  // Actions
  EList * buddy_actions; // actions that can be performed on buddy accounts
  EList * groupchat_actions; // actions that can be performed on a groupchat
  EList * group_user_actions; // actions that can be performed on someone *in* a groupchat
} eb_service;

extern EList * services;

extern eb_service dummy_service;

#endif
