/***************************************************************************
                   registry.c - Manage the EB registry
                             -------------------
                     (C) 2002 by the Everybuddy team
                            www.everybuddy.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "elist.h"

#include "registry.h"

eb_registry_key staticreg={"ROOT", NULL, NULL};

eb_registry_key * registry=&staticreg;

void free_key(eb_registry_key * key);

char * retrieve_value(eb_registry_key * key, char * name); // no path translation
void enter_value(eb_registry_key * key, char * name, char * value); // ditto

void load_key(eb_registry_key * key, FILE * dat);
void save_key(eb_registry_key * key, FILE * dat, char * path);


char * eb_get_value(eb_registry_key * key, char * path)
{
  int s=strlen(path)-1;

  while(1)
  {
    if(s<=0) { return retrieve_value(key, path); }
    if(path[s]=='/')
    {
      char * npath=(char *)strdup(path);
      char * retval;
      npath[s]='\0';
      retval=retrieve_value(eb_get_key(key, npath), path+s+1);
      free(npath);
      return retval;
    }

    s--;
  }
}

char * retrieve_value(eb_registry_key * key, char * name)
{
  EList * n;

  for(n=key->values; n!=NULL; n=n->next)
  {
    eb_registry_value * thisval=(eb_registry_value *)n->data;

    if(!strcmp(name, thisval->name)) { return thisval->value; }
  }

  // failing that...
  return "";
}

eb_registry_key * eb_get_key(eb_registry_key * key, char * cpath)
{
  EList * n;
  char * path=(char *)strdup(cpath);
  char * nextpath=NULL;
  int pos=0;

  eb_registry_key * newkey;

  while(1)
  {
    if(path[pos]=='/')
    {
      path[pos]='\0';
      nextpath=path+pos+1;
      break;
    }
    if(path[pos]=='\0') { break; }

    pos++;
  }

  for(n=key->subkeys; n!=NULL; n=n->next)
  {
    eb_registry_key * thiskey=(eb_registry_key *)n->data;

    if(!strcmp(thiskey->name, path))
    {
      if(nextpath!=NULL)
      {
        eb_registry_key * retval;
        retval=eb_get_key(thiskey, nextpath);
        free(path);
        return retval;
      } else {
        free(path);
        return thiskey;
      }
    }
  }

  // Uh-oh, better create one

  newkey=(eb_registry_key *)malloc(sizeof(eb_registry_key));
  newkey->name=(char *)strdup(path);
  newkey->subkeys=NULL;
  newkey->values=NULL;

  key->subkeys=e_list_append(key->subkeys, newkey);

  if(nextpath!=NULL)
  {
    eb_registry_key * retval;
    retval=eb_get_key(newkey, nextpath);
    free(path);
    return retval;
  } else {
    free(path);
    return newkey;
  }
}

void eb_del_value(eb_registry_key * key, char * name)
{
  EList * n;
  for(n=key->values; n!=NULL; n=n->next)
  {
    eb_registry_value * val=(eb_registry_value *)n->data;
    if(!strcmp(name, val->name))
    {
      free(val->name);
      free(val->value);
      free(val);
      key->values=e_list_remove_link(key->values, n);
      e_list_free_1(n);
      return;
    }
  }
}

void eb_del_key(eb_registry_key * skey, eb_registry_key * target)
{
  EList * n;

  for(n=skey->subkeys; n!=NULL; n=n->next)
  {
    eb_registry_key * key=(eb_registry_key *)n->data;

    if(key==target)
    {
      free_key(target);
      skey->subkeys=e_list_remove_link(skey->subkeys, n);
      e_list_free_1(n);
      return;
    }
  }

  // If we got here, we didn't find it, so let's search everything
  // downstream of us for it too

  for(n=skey->subkeys; n!=NULL; n=n->next)
  {
    eb_registry_key * key=(eb_registry_key *)n->data;

    eb_del_key(key, target);
  }
}

void free_key(eb_registry_key * key)
{
  EList * n;

  for(n=key->subkeys; n!=NULL; n=n->next)
  {
    eb_registry_key * tbd=(eb_registry_key *)n->data;
    free_key(tbd);
  }

  for(n=key->values; n!=NULL; n=n->next)
  {
    eb_registry_value * val=(eb_registry_value *)n->data;

    free(val->name);
    free(val->value);
    free(val);
  }

  e_list_free(key->subkeys);
  e_list_free(key->values);
  free(key->name);
  free(key);
}

void eb_put_value(eb_registry_key * key, char * path, char * value)
{
  int s=strlen(path)-1;

  if(strlen(path)>65535 || strlen(value)>65535)
  {
    fprintf(stderr, "Registry key too long, aborting");
    exit(1); // UGLY!
  }

  while(1)
  {
    if(s<=0) { enter_value(key, path, value); return; }
    if(path[s]=='/')
    {
      char * npath=(char *)strdup(path);
      npath[s]='\0';
      enter_value(eb_get_key(key, npath), path+s+1, value);
      free(npath);
      return;
    }

    s--;
  }
}

void enter_value(eb_registry_key * key, char * name, char * value)
{
  EList * n;
  eb_registry_value * thisval;

  for(n=key->values; n!=NULL; n=n->next)
  {
    thisval=(eb_registry_value *)n->data;

    if(!strcmp(thisval->name, name))
    {
      free(thisval->value);
      thisval->value=(char *)strdup(value);
      return;
    }
  }

  // OK, there's no existing value, let's add a new one

  thisval=(eb_registry_value *)malloc(sizeof(eb_registry_value));
  thisval->name=(char *)strdup(name);
  thisval->value=(char *)strdup(value);

  key->values=e_list_append(key->values, thisval);
}

void eb_put_default(eb_registry_key * key, char * name, char * value)
{
  if(eb_get_value(key, name)[0]=='\0')
  { eb_put_value(key, name, value); }
}

void eb_load_key(eb_registry_key * key, char * fnam)
{
  // Right, from sheer laziness I'm going to use STDIO

  FILE * dat;

  if((dat=fopen(fnam, "r"))==NULL)
  {
    perror("Could not load registry");
    return;
  }

  load_key(key, dat);

  fclose(dat);
}

void eb_save_key(eb_registry_key * key, char * fnam)
{
  FILE * dat;

  if((dat=fopen(fnam, "w"))==NULL)
  {
    perror("Could not save registry");
    return;
  }

  save_key(key, dat, "");

  fclose(dat);
}

void load_key(eb_registry_key * key, FILE * dat)
{
  int a, len;
  char * name;
  char * value;

  while(1)
  {
    len=fgetc(dat)*256+fgetc(dat);
    if(feof(dat)) { break; }
    name=(char *)malloc(len+1);

    for(a=0; a<len; a++)
    { name[a]=fgetc(dat); }
    name[len]='\0';

    len=fgetc(dat)*256+fgetc(dat);
    value=(char *)malloc(len+1);

    for(a=0; a<len; a++)
    { value[a]=fgetc(dat); }
    value[len]='\0';

    eb_put_value(key, name+1, value);
  }
}

void save_key(eb_registry_key * key, FILE * dat, char * path)
{
  EList * n;
  int a, len;
  char * newpath;

  for(n=key->subkeys; n!=NULL; n=n->next)
  {
    eb_registry_key * thiskey=(eb_registry_key *)n->data;
    newpath=(char *)malloc(strlen(path)+strlen(thiskey->name)+2);
    strcpy(newpath, path);
    strcat(newpath, "/");
    strcat(newpath, thiskey->name);
    save_key(thiskey, dat, newpath);
    free(newpath);
  }

  for(n=key->values; n!=NULL; n=n->next)
  {
    eb_registry_value * thisval=(eb_registry_value *)n->data;

    len=strlen(path)+strlen(thisval->name)+1;
    fputc(len/256, dat);
    fputc(len%256, dat);
    for(a=0; a<strlen(path); a++) { fputc(path[a], dat); }
    fputc('/', dat);
    for(a=0; a<strlen(thisval->name); a++) { fputc(thisval->name[a], dat); }

    len=strlen(thisval->value);
    fputc(len/256, dat);
    fputc(len%256, dat);
    for(a=0; a<len; a++) { fputc(thisval->value[a], dat); }
  }

}
