/***************************************************************************
                          proxy.c - Connection wrappers
                             -------------------
                     (C) 2002 by the Everybuddy team
                            www.everybuddy.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#ifndef __MINGW32__
#include <sys/socket.h>
#include <sys/stat.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <netdb.h>
#else
#include <winsock2.h>
#define ECONNREFUSED    WSAECONNREFUSED
#define EINPROGRESS     WSAEINPROGRESS
#endif

#include "plugin_api.h"
#include "dialog.h"

typedef struct _eb_rawtcp_data {
  void (*cb)(int s, void * data);
  void * data;
  int itag;
} eb_rawtcp_data;

typedef struct _eb_socks5_data {
  char * host;
  int port;
  void (*cb)(int s, void * data);
  void * data;
  int nextstage;
  int itag;
} eb_socks5_data;

typedef struct _eb_http_data {
  int stage;
  
  void (*cb)(int s, int code, EList * headers, void * data);
  void * data;
  char * url;
  char * host;
  char * send_headers;
  int port;
  int method;
  int code;
  int tag;
} eb_http_data;

typedef struct _eb_http_connect_data {
  void (*cb)(int s, void * data);
  void * data;
} eb_http_connect_data;

char * eb_http_methods[]={"GET", "POST", "CONNECT"};

void eb_rawtcp_finish(void * cdata, int s, int conditions)
{
  eb_rawtcp_data * data=(eb_rawtcp_data *)cdata;
  int err;
  int val=0, len=4;

  if((err=getsockopt(s, SOL_SOCKET, SO_ERROR, &val, &len))<0 || val!=0)
  {
    close(s);
    data->cb(-1, data->data);
  } else {
#ifndef __MINGW32__
    fcntl(s, F_SETFL, 0);
#endif
    data->cb(s, data->data);
  }

  eb_input_remove(data->itag);

  free(data);
}

int eb_rawtcp_connect(char * hostname, int port, void (*cb)(int s, void * data), void * cdata)
{
  eb_rawtcp_data * edata;
  struct sockaddr_in sa;
  struct hostent     *hp;
  int s;

  // This should be asynchronous. I could *murder* the fool who reckoned that
  // making gethostbyname() synchronous-only was a good idea...

  if ((hp= gethostbyname(hostname)) == NULL) { /* do we know the host's */
    errno= ECONNREFUSED;                       /* address? */
    cb(-1, cdata);                                /* no */
    return -1;
  }

  memset(&sa,0,sizeof(sa));
  memcpy((char *)&sa.sin_addr,hp->h_addr,hp->h_length);     /* set address */
  sa.sin_family= hp->h_addrtype;
  sa.sin_port= htons((u_short)port);

  if ((s= socket(hp->h_addrtype,SOCK_STREAM,0)) < 0)     /* get socket */
  {
    cb(-1, cdata);
    return -1;
  }

  fcntl(s, F_SETFL, O_NONBLOCK);

  if (connect(s,(struct sockaddr *)&sa,sizeof sa) >= 0) { /* connect */
    // wow, instant response! Cool beans...
#ifndef __MINGW32__
    fcntl(s, F_SETFL, 0);
#endif
    cb(s, cdata);
    return s;
  }
  // Otherwise, we're just gonna have to wait for it
  if(errno!=EINPROGRESS)
  {
    close(s);
    cb(-1, cdata);
    return -1;
  }

  edata=(eb_rawtcp_data *)malloc(sizeof(eb_rawtcp_data));
  edata->data=cdata;
  edata->cb=cb;
  edata->itag=eb_input_add(s, EB_INPUT_READ | EB_INPUT_WRITE |  EB_INPUT_EXCEPTION, eb_rawtcp_finish, edata);
  return s;
}

/*****************************************************************************/

void eb_socks5_incoming(void * vdata, int s, int conditions)
{
  eb_socks5_data * data=(eb_socks5_data *)vdata;
  char incoming[16]={0x66, 0x66};
  unsigned char * buf;
  int a;

  if(eread(s, incoming, 2)<2 || incoming[1]!=0)
  {
    printf("Whoops! incoming=[%02x,%02x]\n", incoming[0], incoming[1]);
    eb_input_remove(data->itag);
    close(s);
    data->cb(-1, data->data);
    free(data->host);
    free(data);
    return;
  }

  printf("Socks: incoming data\n");
  
  if(data->nextstage)
  {
    eread(s, incoming, 2);
    switch(incoming[1]) {
      case(1) : { // IPV4
        eread(s, incoming, 6);
        break;
      }

      case(3) : { // DNS
        int a;
        unsigned char n;

        eread(s, &n, 1);
        for(a=0; a<n+2; a++)
        {
          eread(s, incoming, 1);
        }
        break;
      }

      case(4) : {
        eread(s, incoming, 16);
        break;
      }

      default : {
        eb_show_error("The SOCKS proxy gave an unknown address type", "Proxy error");
        eb_input_remove(data->itag);
        close(s);
        data->cb(-1, data->data);
        free(data->host);
        free(data);
        return;
      }
    }
    // OK folks, that's it! This TCP connection now belongs to the server and the routine which asked
    // for it. We just need to get out of the way.

    printf("Connection success (fd %d)\n", s);
    eb_input_remove(data->itag);
    data->cb(s, data->data);
    free(data->host);
    free(data);
    return;
  }

  buf=(char *)malloc(strlen(data->host)+7);
  buf[0]=5; // version
  buf[1]=1; // "CONNECT" command
  buf[2]=0; // reserved
  buf[3]=3; // this is a DNS addy
  buf[4]=strlen(data->host);
  for(a=0; data->host[a]!=0; a++)
  {
    buf[5+a]=data->host[a];
  }
  buf[5+a++]=(data->port)/256;
  buf[5+a++]=(data->port)%256;

  write(s, buf, a+5);

  free(buf);
  data->nextstage=1;
}

void eb_socks5_connect_cb(int s, void * vdata)
{
  eb_socks5_data * data=(eb_socks5_data *)vdata;
  char send[]={5, 1, 0}; // version 5, 1 authentication method supported, unauthenticated
  printf("Socks connect got TCP connection\n");

  if(s==-1)
  {
    data->cb(-1, data->data);
    free(data->host);
    free(data);
    return;
  }

  write(s, send, 3);
  data->nextstage=0;

  data->itag=eb_input_add(s, EB_INPUT_READ | EB_INPUT_EXCEPTION, eb_socks5_incoming, data);
}

int eb_socks5_connect(char * hostname, int port, void (*cb)(int s, void * data), void * cdata)
{
  eb_socks5_data * data=(eb_socks5_data *)malloc(sizeof(eb_socks5_data));
  data->host=strdup(hostname);
  data->port=port;
  data->cb=cb;
  data->data=cdata;

  printf("Socks connect making TCP connection to %s:%d\n", hostname, port);
  
  return eb_rawtcp_connect(eb_get_value(registry, "config/proxy/socks5/host"), atoi(eb_get_value(registry, "config/proxy/socks5/port")), eb_socks5_connect_cb, data);
}


/*****************************************************************************/

void eb_sync_socket_cb(int s, void * data)
{
  *((int *)data)=s;
}

int eb_connect_socket(char * hostname, int port, void (*cb)(int s, void * data), void * cdata)
{
  char * type=eb_get_value(registry, "config/proxy/type");
  
  if(!strcmp(type, "SOCKS 5"))
  {
    return eb_socks5_connect(hostname, port, cb, cdata);
  } else if(!strcmp(type, "HTTP Proxy")) {
    return eb_http_connect(hostname, port, cb, cdata);
  } else {
    return eb_rawtcp_connect(hostname, port, cb, cdata);
  }
}

int eb_connect_sync_socket(char * hostname, int port)
{
  int done=-2;

  int fd = eb_connect_socket(hostname, port, eb_sync_socket_cb, &done);

  while(done==-2)
  { eb_polling_loop(); } // my soul's going to rot in hell for this...be CAREFUL, people!

  return done;
}

/*****************************************************************************/

char * eb_http_readline(int s)
{
  char buf[2048];
  int pos=0;
  
  while(pos<2048)
  {
    if(read(s, buf+pos, 1)==0) { buf[pos]='\n'; }
    if(buf[pos]=='\r') { continue; }
    if(buf[pos]=='\n') { buf[pos]='\0'; return strdup(buf); }
    pos++;
  }
  
  return strdup(buf);
}

void eb_http_data_cb(void * data, int s, int conditions)
{
  eb_http_data * hdata=(eb_http_data *)data;
  char * h;
  int code;
  EList * headers=NULL;
  EList * n;
  
  h=eb_http_readline(s);
  sscanf(h, "HTTP/1.1 %d ", &code);
  free(h);

  while(1)
  {
    h=eb_http_readline(s);
    if(h[0]=='\0') { break; }
    headers=e_list_append(headers, h);
  }
  
  hdata->cb(s, code, headers, hdata->data);

  for(n=headers; n!=NULL; n=n->next)
  { free(n->data); }
  e_list_free(headers);
    
  eb_input_remove(hdata->tag);
  free(hdata->host);
  free(hdata->url);
  free(hdata->send_headers);
  free(hdata);
}

void eb_http_connect_cb(int s, void * data)
{
  eb_http_data * hdata=(eb_http_data *)data;
  char * request;
  int len;

  if(s>=0)
  {
    request=malloc(64+strlen(hdata->url)+strlen(hdata->host));
    len=sprintf(request, "%s %s HTTP/1.1\r\nHost: %s:%d\r\n%s\r\n",
      eb_http_methods[hdata->method], hdata->url, hdata->host,
      hdata->port, hdata->send_headers);

    write(s, request, len);
    
    hdata->tag=eb_input_add(s, EB_INPUT_READ, eb_http_data_cb, hdata);
  } else {
    hdata->cb(-1, 0, NULL, hdata->data);
  
    free(hdata->host);
    free(hdata->url);
    free(hdata->send_headers);
    free(hdata);
  }
}

int eb_http_request(char * host, int port, char * url, int method, EList * extra_headers, void (*cb)(int s, int code, EList * headers, void * data), void * data)
{
  eb_http_data * hdata=(eb_http_data *)malloc(sizeof(eb_http_data));
  int len=0;
  EList * n;
  EList * headers=NULL;
  char * proxyauth=NULL;
  if(!strcmp(eb_get_value(registry, "config/proxy/type"), "HTTP Proxy"))
  {
    hdata->host=strdup(eb_get_value(registry, "config/proxy/http/host"));
    hdata->port=atoi(eb_get_value(registry, "config/proxy/http/port"));
    if(method==EB_HTTP_CONNECT)
    {
      hdata->url=(char *)malloc(strlen(host)+16);
      sprintf(hdata->url, "%s:%d", host, port);
    } else {
      hdata->url=(char *)malloc(strlen(url)+strlen(host)+16);
      sprintf(hdata->url, "http://%s:%d%s", host, port, url);
    }
    
    if(eb_get_value(registry, "config/proxy/http/doauth"))
    {
      char * user=eb_get_value(registry, "config/proxy/http/username");
      char * pass=eb_get_value(registry, "config/proxy/http/password");
      char * toenc=(char *)malloc(strlen(user)+strlen(pass)+2);
      int len=sprintf(toenc, "%s:%s", user, pass);
      char * encoded=(char *)malloc(2*len+3);
      len=encode_base64(encoded, toenc, len);
      proxyauth=(char *)malloc(len+32);
      sprintf(proxyauth, "Proxy-Authorization: basic %s", encoded);
      free(toenc);
      free(encoded);
      headers=e_list_append(headers, proxyauth);
    }
  } else {
    hdata->host=strdup(host);
    hdata->port=port;
    hdata->url=strdup(url);
  }

  for(n=extra_headers; n!=NULL; n=n->next)
  { headers=e_list_append(headers, n->data); }
  
  hdata->stage=0;
  hdata->code=-1;
  hdata->cb=cb;
  hdata->data=data;
  hdata->method=method;
  
  for(n=headers; n!=NULL; n=n->next)
  { len+=strlen((char *)n->data)+2; }
  hdata->send_headers=(char *)malloc(len+1);
  for(n=headers; n!=NULL; n=n->next)
  { strcat(hdata->send_headers, (char *)n->data); strcat(hdata->send_headers, "\r\n"); }

  e_list_free(headers);
  if(proxyauth!=NULL) { free(proxyauth); }
  return eb_connect_socket(host, port, eb_http_connect_cb, hdata);
}

void eb_http_done(int sock)
{
  close(sock); // Just in case we wanted to reuse 
}

/*****************************************************************************/

void eb_http_tunnel_cb(int s, int code, EList * headers, void * vdata)
{
  eb_http_connect_data * data=(eb_http_connect_data *)vdata;
  if(code==200)
  {
    data->cb(s, data->data);
  } else {
    eb_show_error("Proxy tunnel failed - check your proxy username and password. Perhaps you need to set the port to 443 in your account preferences. Does this service support HTTP proxy tunnelling?", "Proxy error");
    data->cb(-1, data->data);
    close(s);
  }
  free(data);
}

int eb_http_connect(char * hostname, int port, void (*cb)(int s, void * data), void * cbdata)
{
  eb_http_connect_data * data=(eb_http_connect_data *)malloc(sizeof(eb_http_connect_data));
  data->cb=cb;
  data->data=cbdata;
  return eb_http_request(hostname, port, NULL, EB_HTTP_CONNECT, NULL, eb_http_tunnel_cb, data);
}
