/***************************************************************************
                plugin.c - The mechanics of plugin loading
                             -------------------
                     (C) 2002 by the Everybuddy team
                            www.everybuddy.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#if HAVE_CONFIG_H
# include <config.h>
# if HAVE_DIRENT_H
#  include <sys/types.h>
#  include <dirent.h>
#  define NAMLEN(dirent) strlen((dirent)->d_name)
# else
#  define dirent direct
#  define NAMLEN(dirent) (dirent)->d_namlen
#  if HAVE_SYS_NDIR_H
#   include <sys/ndir.h>
#  endif
#  if HAVE_SYS_DIR_H
#   include <sys/dir.h>
#  endif
#  if HAVE_NDIR_H
#   include <ndir.h>
#  endif
# endif
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "elist.h"

#include "globals.h"
#include "plugin.h"
#include "gui_comms.h"
#include "services.h"
#include "accounts.h"

#include <ltdl.h>

typedef lt_dlhandle dlhandle;

eb_service dummy_service;
static eb_service_callbacks dummy_callbacks;

EList * plugins=NULL;
EList * services=NULL;

int load_plugin(char * path, char * fnam);

void plugins_init()
{
  lt_dlinit();

  dummy_service.name=strdup("NONE");
  dummy_service.capabilities=0;
  memset(&dummy_callbacks, 0, sizeof(dummy_callbacks));
  dummy_service.sc=&dummy_callbacks;
}

void load_plugins(char * dir)
{
  DIR * d;
  struct dirent * thisent;

  load_static_mods();

  if((d=opendir(dir))==NULL)
  {
    perror("Could not open plugin directory");
    return;
  }

  while((thisent=readdir(d))!=NULL)
  {
    int len=strlen(thisent->d_name);

    if(len<4) { continue; }

    if(thisent->d_name[len-3]!='.' || thisent->d_name[len-2]!='l' ||thisent->d_name[len-1]!='a')
    { continue; }

    load_plugin(dir, thisent->d_name);
  }

  closedir(d);
}

eb_plugin * plugin_by_name(char * fnam)
{
  EList * n;

  for(n=plugins; n!=NULL; n=n->next)
  {
    eb_plugin * plugin=(eb_plugin *)n->data;
    if(!strcmp(plugin->filename, fnam)) { return plugin; }
  }

  return NULL;
}

static eb_plugin * create_plugin(char * fnam)
{
  eb_plugin * plugin = (eb_plugin *)malloc(sizeof(eb_plugin));
  plugin->filename=(char *)strdup(fnam);
  plugin->info=NULL;
  plugin->status=PLUGIN_NOT_LOADED;
  plugin->errormsg=NULL;
  plugins=e_list_append(plugins, plugin);

  return plugin;
}

static void add_error_entry(char * fnam, char * errormsg)
{
  eb_plugin * plugin;
  int isnew=0;

  if((plugin=plugin_by_name(fnam))==NULL)
  {
    plugin=create_plugin(fnam);
    isnew=1;
  }

  if (plugin->info)
    free(plugin->info);
  if (plugin->errormsg)
    free(plugin->errormsg);

  plugin->info=NULL;
  plugin->status=PLUGIN_CANNOT_LOAD;
  plugin->errormsg=(char *)strdup(errormsg);

  if(isnew)
  {
    new_plugin_notify(plugin);
  } else {
    plugin_update_notify(plugin);
  }
}

int load_plugin(char * path, char * fnam)
{
  char * full_path;
  dlhandle dat;
  eb_plugin_info * info;
  eb_plugin * plugin;
  int isnew=0;

  full_path=(char *)malloc(strlen(path)+strlen(fnam)+2);
  sprintf(full_path, "%s/%s", path, fnam);

  plugin=plugin_by_name(full_path);
  if(plugin!=NULL && plugin->status==PLUGIN_LOADED)
  {
    eb_debug(DBG_CORE, "Not loading already-loaded module %s\n", full_path);
    free(full_path);
    return 0;
  }

  dat=lt_dlopen(full_path);


  if(dat==NULL)
  {
    char * errstring=(char *)lt_dlerror(); // weird warnings unless I cast *shrug*
    eb_debug(DBG_CORE, "Could not open module %s (%s)\n", full_path, errstring);
    add_error_entry(full_path, errstring);
    free(full_path);
    return -1;
  }

  info=(eb_plugin_info *)lt_dlsym(dat, "plugin_info");

  if(info==NULL)
  {
    lt_dlclose(dat);
    eb_debug(DBG_CORE, "Could not resolve plugin_info from %s, skipping...\n", full_path);
    add_error_entry(full_path, "Not a valid plugin - symbol plugin_info not found");
    free(full_path);
    return -1;
  }

  // Now, we dispatch based on plugin type. Are we all sitting comfortably?

  if(plugin==NULL)
  {
    plugin=create_plugin(full_path);
    isnew=1;
  }

  if (plugin->info)
    free(plugin->info);
  if (plugin->errormsg)
    free(plugin->errormsg);

  plugin->status=PLUGIN_LOADED;
  plugin->errormsg=NULL;
  plugin->info=info;

  if(isnew)
  {
    new_plugin_notify(plugin);
  } else {
    plugin_update_notify(plugin);
  }

  free(full_path);

  info->init();

  return 0;
}

int eb_load_service(eb_service * service)
{
  EList * n;

  for(n=services; n!=NULL; n=n->next)
  {
    eb_service * tserv=(eb_service *)n->data;

    if(!strcmp(service->name, tserv->name))
    {
      eb_debug(DBG_CORE, "Refusing to load service %s, as a plugin for that service is already loaded.\n", service->description);
      return -1;
    }
  }

  eb_debug(DBG_CORE, "Loading a service \"%s\"\n", service->name);

  services=e_list_append(services, service);;

  // Any accounts who wish to use this service, please raise your hands now...
  map_orphaned_accounts();

  return 0;
}
