/***************************************************************************
               gui_comms.c - Communication with user interfaces
                             -------------------
                     (C) 2002 by the Everybuddy team
                            www.everybuddy.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifndef __MINGW32__
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#else
#include <winsock2.h>
#endif
#include <errno.h>
#include <unistd.h>

#include "elist.h"

#include "globals.h"
#include "accounts.h"
#include "services.h"
#include "gui_comms.h"
#include "plugin.h"
#include "plugin_api.h"
#include "prefs.h"
#include "util.h"
#include "html.h"
#include "message_parse.h"
#include "stream.h"

// Defined as extern in globals.h:
char * away_msg=NULL;
char * away_short=NULL;

typedef enum {
  EB_ERROR_DIALOG,
  EB_LIST_DIALOG,
  EB_YESNO_DIALOG,
  EB_TEXT_DIALOG
} EB_DIALOG_TYPE;

typedef struct _eb_dialog
{
  EB_DIALOG_TYPE type;
  void * data;
  int id;
} eb_dialog;

typedef struct _eb_error_dialog
{
  char * title;
  char * message;
} eb_error_dialog;

typedef struct _eb_list_dialog
{
  char * title;
  char * message;
  EList * options;

  void * tag;
  void (*callback)(char * response, void * tag);
} eb_list_dialog;

typedef struct _eb_yesno_dialog
{
  char * title;
  char * message;
  int default_result;

  void * tag;
  void (*callback)(int response, void * tag);
} eb_yesno_dialog;

typedef struct _eb_text_dialog
{
  char * title;
  char * message;
  char * initial_contents;
  int is_html;

  void * tag;
  void (*callback)(char * response, void * tag);
} eb_text_dialog;

void new_plugin_notify(eb_plugin * e) {}
void plugin_update_notify(eb_plugin * e) {}

unsigned int next_dialog_id=1;
int sock;
int eb_quit_on_last=0;
unsigned char eb_authcookie[8]; // Mmmm...cookie...
EList * GUIs;
EList * dialogs; // outstanding

// GUI communications - this should be fun...

void eb_lose_gui(eb_gui * gui);
void eb_gui_welcome(eb_gui * gui);
void eb_gui_incoming_command(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_incoming(void *data, int source, eb_input_condition condition);
void eb_gui_connect(void *data, int source, eb_input_condition condition);
int eb_gui_socket(int * port);

void eb_gui_get_data(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_put_data(eb_gui * gui, char ** params, int * lengths, int num_params);
eb_registry_key * eb_gui_data_key(char ** params, int len);

void eb_gui_list_services(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_list_actions(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_perform_action(eb_gui * gui, char ** params, int * lengths, int num_params);

void eb_gui_local_account_update(eb_gui * gui, eb_local_account * acc);

void eb_gui_ignore(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_unignore(eb_gui * gui, char ** params, int * lengths, int num_params);

void eb_gui_dialog_send(eb_gui *, eb_dialog * d);

void eb_gui_send_data_message(eb_gui * gui, char ** params, int * lengths, int num_params);

void eb_gui_local_account_login(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_local_account_logout(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_set_state(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_add_group(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_add_contact(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_add_account(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_rename_contact(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_move_contact(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_move_account(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_del_group(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_del_contact(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_del_account(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_add_local_account(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_del_local_account(eb_gui * gui, char ** params, int * lengths, int num_params);

void eb_gui_list_pref_page(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_set_pref_value(eb_gui * gui, char ** params, int * lengths, int num_params);

void eb_gui_buddy_login(eb_gui * gui, eb_account * acc);
void eb_gui_buddy_update(eb_gui * gui, eb_account * acc);
void eb_gui_buddy_logout(eb_gui * gui, eb_account * acc);

void eb_gui_set_away(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_unset_away(eb_gui * gui, char ** params, int * lengths, int num_params);

void eb_gui_hold_messages(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_get_held_messages(eb_gui * gui, char ** params, int * lengths, int num_params);

void eb_gui_request_group_join(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_group_close(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_group_hide(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_group_invite(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_group_send(eb_gui * gui, char ** params, int * lengths, int num_params);

void eb_gui_get_stream(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_stream_data(eb_gui * gui, char ** params, int * lengths, int num_params);
void eb_gui_stream_abort(eb_gui * gui, char ** params, int * lengths, int num_params);

int eb_gui_socket(int * port)
{
  int s, opt=1;
  struct sockaddr_in addr;

  if((s=socket(AF_INET, SOCK_STREAM, 0))<0
   || setsockopt(s,SOL_SOCKET,SO_REUSEADDR, (char *)&opt, sizeof(opt)) < 0)
  {
    return -1;
  }

  memset(&addr, 0, sizeof(addr));
  addr.sin_family=AF_INET;
  addr.sin_port=htons(*port);

  if(bind(s, (struct sockaddr *)(&addr), sizeof(addr))<0 || listen(s, 5)<0)
  {
    close(s);
    return -1;
  }
  
  *port=ntohs(addr.sin_port);

  return s;
}

void eb_gui_connected(void *data, int source, eb_input_condition condition)
{
  int sock;
  eb_gui * gui;

  sock=accept(source, NULL, NULL);
  if(sock<=-1)
  {
    perror("accept() failed");
    return;
  }

  gui=(eb_gui *)malloc(sizeof(eb_gui));
  GUIs=e_list_append(GUIs, gui);
  gui->num_params=0;
  gui->params=NULL;
  gui->cookie_progress=0;
  gui->read_fd=sock;
  gui->write_fd=sock;
  gui->html=1;
  gui->msg_capable=1;
  gui->streams=NULL;

  gui->tag=eb_input_add(sock, EB_INPUT_READ | EB_INPUT_EXCEPTION, eb_gui_incoming, gui);
}

void eb_gui_incoming(void *data, int source, eb_input_condition condition)
{
  eb_gui * gui=(eb_gui *)data;
  int a;

  if(gui->read_fd!=source) { printf("What the flying...?\n"); }

  if(gui->cookie_progress<8)
  {
    unsigned char c;
    if(eread(source, &c, 1)!=1)
    {
      close(gui->read_fd);
      if(gui->write_fd!=gui->read_fd) { close(gui->write_fd); }
      eb_lose_gui(gui);
    }
    gui->cookie[gui->cookie_progress++]=c;

    if(gui->cookie_progress==8)
    {
      char * accept_cmd[]={"cookie_accepted"};
      int a;
      for(a=0; a<8; a++)
      {
        if(gui->cookie[a]!=eb_authcookie[a])
        {
          char * reject_cmd[]={"cookie_rejected"};
          eb_gui_send(gui, reject_cmd, 1);
          eb_lose_gui(gui); close(source);
          return;
        }
      }

      // OK, if we get this far, we're accepted. Tell them so.
      eb_gui_send(gui, accept_cmd, 1);
      eb_gui_welcome(gui);
    }

    return;
  }

  if(gui->num_params==0) // ooh, look Mummy, new command!
  {
    unsigned char c;
    if(eread(source, &c, 1)<1)
    {
      eb_lose_gui(gui);
      close(source);
      return;
    }
    gui->num_params=c;
    gui->params_arrived=0;
    gui->params=(char **)malloc(c*sizeof(char *));
    gui->lengths=(int *)malloc(c*sizeof(int));
  } else if (gui->params_arrived < gui->num_params) {
    unsigned char c[4];
    int size;
    char * s;

    if(eread(source, c, 2)<2)
    {
      perror("Lost connection with a GUI mid-command");
      eb_lose_gui(gui);
      return;
    }

    if(c[0]==255 && c[1]==255)
    {
      if(eread(source, c, 4)<4)
      {
        perror("Lost connection with a GUI mid-command");
        eb_lose_gui(gui);
        return;
      }
      size=c[0]<<24 | c[1]<<16 | c[2]<<8 | c[3];
    } else {
      size=c[0]*256+c[1];
    }

    s=(char *)malloc(size+1*sizeof(char));

    if(eread(source, s, size)<size)
    {
      perror("Lost connection with a GUI mid-read");
      s[size]='\0';
      eb_lose_gui(gui);
      return;
    }
    s[size]='\0';

    gui->lengths[gui->params_arrived]=size;
    gui->params[gui->params_arrived++]=s;

    if(gui->params_arrived==gui->num_params)
    {
      char ** params = gui->params;
      int * lengths = gui->lengths;
      int num_params = gui->num_params;
      gui->params = NULL;
      gui->lengths = NULL;
      gui->num_params=0;
      gui->params_arrived=0;
      eb_gui_incoming_command(gui, params, lengths, num_params);
      for(a=0; a<num_params; a++)
      {
        if(params[a]!=NULL)
        { free(params[a]); params[a]=NULL; }
      }
      free(lengths);
      free(params);
    }
  }
}

void eb_gui_welcome(eb_gui * gui)
{
  EList * n;
  char * hold_cmd[]={"message_hold", "0"};

  // Send them all outstanding dialog boxes
  for(n=dialogs; n!=NULL; n=n->next)
  {
    eb_dialog * d=(eb_dialog *)n->data;
    eb_gui_dialog_send(gui, d);
  }

  // If we're set as Away, say so...
  if(away_msg!=NULL)
  {
    char * away_command[]={"set_away", away_short, away_msg};
    eb_gui_send(gui, away_command, 3);
  }

  // ditto if we're holding messages
  if(eb_holding_messages) { hold_cmd[1]="1"; }

  eb_gui_send(gui, hold_cmd, 2);

  if(eb_held_messages!=NULL)
  {
    char * held_cmd[]={"holding_message"};
    eb_gui_send(gui, held_cmd, 1);
  }
}

void eb_gui_incoming_command(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  void (*action_func)(eb_gui *, char ** params, int * lengths, int num_params);
  int min_params=0;

  action_func=NULL;

  if(num_params==0)
  {
    eb_debug(DBG_CORE, "GUI SENT EMPTY COMMAND!\n");
    return;
  }

  if(!strcmp(params[0], "get_data"))
  {
    action_func=eb_gui_get_data;
    min_params=3;
  } else if (!strcmp(params[0], "put_data")) {
    action_func=eb_gui_put_data;
    min_params=4;
  } else if (!strcmp(params[0], "list_contacts")) {
    action_func=eb_gui_list_contacts;
    min_params=1;
  } else if (!strcmp(params[0], "list_local_accounts")) {
    action_func=eb_gui_list_accounts;
    min_params=1;
  } else if (!strcmp(params[0], "list_services")) {
    action_func=eb_gui_list_services;
    min_params=1;
  } else if (!strcmp(params[0], "message_send")) {
    action_func=eb_gui_send_message;
    min_params=6;
  } else if (!strcmp(params[0], "data_message_send")) {
    action_func=eb_gui_send_data_message;
    min_params=6;
  } else if (!strcmp(params[0], "list_actions")) {
    action_func=eb_gui_list_actions;
    min_params=1;
  } else if (!strcmp(params[0], "perform_action")) {
    action_func=eb_gui_perform_action;
    min_params=3; // sometimes more...
  } else if (!strcmp(params[0], "resolve_dialog")) {
    action_func=eb_gui_resolve_dialog;
    min_params=2; // also sometimes more
  } else if (!strcmp(params[0], "sign_on_all")) {
    eb_sign_on_all();
    return;
  } else if (!strcmp(params[0], "sign_off_all")) {
    eb_sign_off_all();
    return;
  } else if (!strcmp(params[0], "local_account_login")) {
    action_func=eb_gui_local_account_login;
    min_params=3; // also sometimes more
  } else if (!strcmp(params[0], "local_account_logout")) {
    action_func=eb_gui_local_account_logout;
    min_params=3; // also sometimes more
  } else if (!strcmp(params[0], "set_local_account_status")) {
    action_func=eb_gui_set_state;
    min_params=4;
  } else if (!strcmp(params[0], "join_group_chat")) {
    action_func=eb_gui_request_group_join;
    min_params=4;
  } else if (!strcmp(params[0], "close_group_chat")) {
    action_func=eb_gui_group_close;
    min_params=2;
  } else if (!strcmp(params[0], "hide_group_chat")) {
    action_func=eb_gui_group_hide;
    min_params=2;
  } else if (!strcmp(params[0], "group_chat_invite")) {
    action_func=eb_gui_group_invite;
    min_params=3;
  } else if (!strcmp(params[0], "group_chat_send")) {
    action_func=eb_gui_group_send;
    min_params=3;
  } else if (!strcmp(params[0], "add_group")) {
    action_func=eb_gui_add_group;
    min_params=2;
  } else if (!strcmp(params[0], "add_contact")) {
    action_func=eb_gui_add_contact;
    min_params=3;
  } else if (!strcmp(params[0], "add_account")) {
    action_func=eb_gui_add_account;
    min_params=6;
  } else if (!strcmp(params[0], "rename_contact")) {
    action_func=eb_gui_rename_contact;
    min_params=4;
  } else if (!strcmp(params[0], "move_contact")) {
    action_func=eb_gui_move_contact;
    min_params=4;
  } else if (!strcmp(params[0], "move_account")) {
    action_func=eb_gui_move_account;
    min_params=6;
  } else if (!strcmp(params[0], "del_group")) {
    action_func=eb_gui_del_group;
    min_params=2;
  } else if (!strcmp(params[0], "del_contact")) {
    action_func=eb_gui_del_contact;
    min_params=3;
  } else if (!strcmp(params[0], "del_account")) {
    action_func=eb_gui_del_account;
    min_params=4;
  } else if (!strcmp(params[0], "add_local_account")) {
    action_func=eb_gui_add_local_account;
    min_params=3;
  } else if (!strcmp(params[0], "del_local_account")) {
    action_func=eb_gui_del_local_account;
    min_params=3;
  } else if (!strcmp(params[0], "ignore_contact")) {
    action_func=eb_gui_ignore;
    min_params=4;
  } else if (!strcmp(params[0], "unignore_contact")) {
    action_func=eb_gui_unignore;
    min_params=3;
  } else if (!strcmp(params[0], "list_pref_page")) {
    action_func=eb_gui_list_pref_page;
    min_params=2;
  } else if (!strcmp(params[0], "set_pref_value")) {
    action_func=eb_gui_set_pref_value;
    min_params=4;
  } else if (!strcmp(params[0], "unset_away")) {
    action_func=eb_gui_unset_away;
    min_params=1;
  } else if (!strcmp(params[0], "set_away")) {
    action_func=eb_gui_set_away;
    min_params=3;
  } else if (!strcmp(params[0], "message_hold")) {
    action_func=eb_gui_hold_messages;
    min_params=2;
  } else if (!strcmp(params[0], "get_held_messages")) {
    action_func=eb_gui_get_held_messages;
    min_params=1;
  } else if (!strcmp(params[0], "get_stream")) {
    action_func=eb_gui_get_stream;
    min_params=3;
  } else if (!strcmp(params[0], "sdata")) {
    action_func=eb_gui_stream_data;
    min_params=3;
  } else if (!strcmp(params[0], "up_stream_abort")
      || !strcmp(params[0], "down_stream_abort")
      || !strcmp(params[0], "stream_ends")) {
    action_func=eb_gui_stream_abort;
    min_params=2;
  } else if (!strcmp(params[0], "html_strip")) {
    if(num_params>=2)
    { gui->html=(params[1][0]=='0'); }
    return;
  } else if (!strcmp(params[0], "msg_capable")) {
    if(num_params>=2)
    { gui->msg_capable=(params[1][0]=='1'); }
    return;
  } else if (!strcmp(params[0], "list_group_chats")) {
    EList * n;
    for(n=group_chats; n!=NULL; n=n->next)
    {
      eb_group_chat * chat=(eb_group_chat *)n->data;

      if(!chat->announced) { continue; }
      eb_gui_announce_group_chat(gui, chat);
    }
    return;
  } else if (!strcmp(params[0], "save_config")) {
    eb_save_config();
    return;
  }

  if(action_func==NULL)
  {
    eb_do_client_error(gui, "Unknown command");
  } else {
    if(num_params<min_params)
    {
      eb_do_client_error(gui, "Insufficient arguments");
    } else {
      action_func(gui, params, lengths, num_params);
    }
  }
}

void eb_lose_gui(eb_gui * gui)
{
  EList * l=GUIs;
  int a;

  while(l!=NULL)
  {
    eb_gui * tgui=(eb_gui *)l->data;
    if(tgui==gui)
    {
      eb_input_remove(gui->tag);

      if(gui->params!=NULL)
      {
        for(a=0; a<gui->params_arrived; a++)
        {
          if(gui->params[a]!=NULL)
          { free(gui->params[a]); }
        }
        free(gui->params);
      }

      free(gui);

      GUIs=e_list_remove_link(GUIs, l);
      e_list_free_1(l);

      if(GUIs==NULL && eb_quit_on_last)
      { printf("Obeying -q option, quitting on last disconnect...\n"); exit(0); }
      
      for(l=GUIs; l; l=l->next)
      {
        tgui=(eb_gui *)l->data;
        if(tgui->msg_capable) { break; }
      }
      
      if(l==NULL)
      {
        if(away_msg==NULL)
        {
          eb_set_away("Nobody home", "This is an automated message. The user you are addressing is probably away from their computer. Your message will be displayed when they return.");
        }

        eb_hold_messages(1);
        // no need to broadcast - no-one to do it to!
      }

      break;
    }
    l=l->next;
  }
}

void eb_gui_send(eb_gui * gui, char ** command, int num_params)
{
  static int plen[256];
  int a;

  for (a=0; a<num_params; a++)
  {
    plen[a]=strlen(command[a]);
  }

  eb_gui_send_data(gui, command, plen, num_params);
}

void eb_gui_send_data(eb_gui * gui, char ** command, int * lengths, int num_params)
{
  unsigned char c=num_params;
  int a;

  if(write(gui->write_fd, &c, 1)<1)
  { return; } // dud socket, let the cleanup routines take care of it

  for(a=0; a<num_params; a++)
  {
    c=lengths[a]/256;
    write(gui->write_fd, &c, 1);
    c=lengths[a]%256;
    write(gui->write_fd, &c, 1);

    write(gui->write_fd, command[a], lengths[a]);
  }
}

void eb_gui_broadcast(char ** command, int num_params)
{
  EList * n;

  for(n=GUIs; n!=NULL; n=n->next)
  {
    eb_gui * gui=(eb_gui *)n->data;

    if(gui->cookie_progress<8) { continue; }

    eb_gui_send(gui, command, num_params);
  }
}

int eb_gui_comms_init(int port)
{
  FILE * dat;
  char * cookie_file;
#ifdef __MINGW32__
  WSADATA wsaData;
  WSAStartup(MAKEWORD(2,0),&wsaData);
#endif

  if((sock=eb_gui_socket(&port))<0)
  {
    perror("Could not bind a GUI socket");
    exit(1);
  }

  cookie_file=(char *)malloc((strlen(eb_config_dir)+20)*sizeof(char));

  strcpy(cookie_file, eb_config_dir);
  strcat(cookie_file, "authcookie");

  if((dat=fopen(cookie_file, "w"))==NULL)
  {
    perror("Could not save local cookie");
  } else {
    int a;
    fprintf(dat, "%c%c", port/256, port%256);
    for(a=0; a<8; a++)
    {
      fprintf(dat, "%c", eb_authcookie[a]);
    }
    fclose(dat);
    
  }

  eb_input_add(sock, EB_INPUT_READ | EB_INPUT_WRITE | EB_INPUT_EXCEPTION, eb_gui_connected, NULL);
  // we don't keep the tag, because this socket will stay open until EB closes

  return 0;
}

// Responses to requests

eb_registry_key * eb_gui_data_key(char ** params, int len)
{
  if(!strcmp(params[0], "general"))
  { return eb_get_key(registry, "config/ui_data"); }

  if(!strcmp(params[0], "contact"))
  {
    eb_contact * contact;

    if(len<3) { return NULL; }
    contact=eb_get_contact(params[1], params[2]);
    if(contact==NULL) { return NULL; }

    return eb_get_key(contact->config_key, "ui_data");
  }

  return NULL;
}

void eb_gui_get_data(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_registry_key * key=eb_gui_data_key(params+2, num_params-2);
  char * notify_cmd[6];
  int a;

  if(key==NULL)
  { eb_do_client_error(gui, "Invalid data path specification"); return; }

  notify_cmd[0]="return_data";
  notify_cmd[1]=params[1];
  notify_cmd[2]=eb_get_value(key, params[1]);
  for(a=2; a<num_params; a++)
  {
    notify_cmd[a+1]=params[a];
  }

  eb_gui_send(gui, notify_cmd, num_params+1);
}

void eb_gui_put_data(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_registry_key * key=eb_gui_data_key(params+3, num_params-3);
  if(key==NULL)
  { eb_do_client_error(gui, "Invalid data path specification"); return; }

  eb_put_value(key, params[1], params[2]);
}

// little util functions
void eb_gui_list_service_actions(char * service, char * target, EList * actions, eb_gui * gui)
{
  EList * n;
  char ** notify_command;
  char buf[16];
  int a=4;
  int len=e_list_length(actions);

  notify_command=(char **)malloc((len+4)*sizeof(char *));

  notify_command[0]="list_service_actions";
  notify_command[1]=service;
  notify_command[2]=target;
  notify_command[3]=buf;

  sprintf(buf, "%d", len);

  for(n=actions; n!=NULL; n=n->next)
  {
    eb_action * action=(eb_action *)n->data;
    notify_command[a]=action->name;
    a++;
  }

  eb_gui_send(gui, notify_command, a);

  free(notify_command);
}

void eb_gui_list_services(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  char buf[16];
  char * begin_serv[]={"list_service", "", "", buf};
  char ** states_cmd;
  char * end_serv[]={"list_service_done"};
  char * end_cmd[]={"list_services_done"};
  EList * n;

  for(n=services; n!=NULL; n=n->next)
  {
    eb_service * service=(eb_service *)n->data;
    int numstates;
    EList * n2;

    begin_serv[1]=service->name;
    begin_serv[2]=service->color;
    sprintf(buf, "%d", service->capabilities);
    eb_gui_send(gui, begin_serv, 4);

    eb_gui_list_service_actions(service->name, "buddy", service->buddy_actions, gui);
    eb_gui_list_service_actions(service->name, "groupchat", service->groupchat_actions, gui);
    eb_gui_list_service_actions(service->name, "group_user", service->group_user_actions, gui);

    numstates=e_list_length(service->states);
    if(numstates==0)
    {
      states_cmd=(char **)malloc(4*sizeof(char *));
      states_cmd[0]="list_service_states";
      states_cmd[1]=service->name;
      states_cmd[2]="1";
      states_cmd[3]="No service plugin loaded";
      numstates=1;
    } else {
      int a=3;
      char buf[16];

      states_cmd=(char **)malloc((numstates+3)*sizeof(char *));
      states_cmd[0]="list_service_states";
      states_cmd[1]=service->name;
      sprintf(buf, "%d", numstates);
      states_cmd[2]=buf;

      for(n2=service->states; n2!=NULL; n2=n2->next)
      {
        char * state=(char *)n2->data;

        states_cmd[a++]=state;
      }
    }

    eb_gui_send(gui, states_cmd, numstates+3);
    free(states_cmd);

    eb_gui_send(gui, end_serv, 1);
  }

  eb_gui_send(gui, end_cmd, 1);
}

void eb_gui_list_accounts(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  char * done_command[]={"list_local_accounts_done"};
  char * notify_command[]={"list_local_account", "", ""};
  EList * l;

  for(l=local_accounts; l!=NULL; l=l->next)
  {
    eb_local_account * la=(eb_local_account *)l->data;

    notify_command[1]=la->handle;
    notify_command[2]=la->service_name;

    eb_gui_send(gui, notify_command, 3);
    eb_gui_local_account_update(gui, la);
  }

  eb_gui_send(gui, done_command, 1);
}

void eb_gui_list_contacts(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  EList * g;
  EList * c;
  EList * a;

  char * done_command[]={"list_contacts_done"};
  char * group_command[]={"list_group", ""};
  char * contact_command[]={"list_contact", "", "", "0", ""};
  char * account_command[]={"list_account", "", "", "", "", "", ""};

  for(g=groups; g!=NULL; g=g->next)
  {
    eb_group * group=(eb_group *)g->data;

    group_command[1]=group->name;
    contact_command[1]=group->name;
    account_command[1]=group->name;
    eb_gui_send(gui, group_command, 2);

    for(c=group->contacts; c!=NULL; c=c->next)
    {
      eb_contact * cont=(eb_contact *)c->data;

      contact_command[2]=cont->name;
      if(cont->ignore!=NULL)
      {
        contact_command[3]="1";
        contact_command[4]=cont->ignore;
      } else {
        contact_command[3]="";
        contact_command[4]="";
      }
      account_command[2]=cont->name;
      eb_gui_send(gui, contact_command, 5);

      for(a=cont->accounts; a!=NULL; a=a->next)
      {
        eb_account * acc=(eb_account *)a->data;
        account_command[3]=acc->buddy_of->handle;
        account_command[4]=acc->buddy_of->service_name;
        account_command[5]=acc->handle;
        account_command[6]=(acc->blocked)?("1"):("0");
        eb_gui_send(gui, account_command, 7);

        if(acc->status!=EB_ACCOUNT_OFFLINE)
        { eb_gui_buddy_login(gui, acc); eb_gui_buddy_update(gui, acc); }
      }
    }
  }

  eb_gui_send(gui, done_command, 1);
}

void eb_gui_send_message(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  EList * l;
  eb_group * group=NULL;
  eb_contact * cont=NULL;
  eb_account * acc=NULL;

  group=eb_get_group(params[1]);
  cont=eb_get_contact(params[1], params[2]);

  if(cont==NULL)
  {
    eb_do_client_error(gui, "No such contact");
    return;
  }

  acc=eb_get_message_account(cont, params[3], params[4], params[5], 0);

  if(acc==NULL)
  { eb_do_client_error(gui, "Contact not online"); return; }  

  if(away_msg!=NULL)
  { cont->next_away_msg=time(NULL)+60; }

  if(acc->buddy_of->service->sc->send_im)
  {
    char * msg=strdup(params[6]);
    msg=acc->buddy_of->service->sc->send_im(acc, msg);
    if(msg==NULL) { return; }

    eb_sent_message(acc, msg);
    
    msg=eb_filter_string(EB_POSTNOTIFY_OUT, msg, acc);
    if(msg!=NULL)   
    { free(msg); }
  }
}

typedef struct _eb_data_message_waiting {
  eb_account * buddy;
  int length;
  char * filename;
  char * contenttype;
  char * disposition;
  eb_stream * stream;
} eb_data_message_waiting;


static void eb_data_message_uploaded(int err, void * data)
{
  eb_data_message_waiting * ebdmw=(eb_data_message_waiting *)data;
  
  ebdmw->buddy->locked--;
  
  if(!err)
  {
    ebdmw->buddy->buddy_of->service->sc->send_data_message(ebdmw->buddy, ebdmw->filename, ebdmw->contenttype, ebdmw->disposition, ebdmw->length);
  } else {
    printf("Data message abort\n");
    unlink(ebdmw->filename);
  }
  
  free(ebdmw->filename);
  free(ebdmw->contenttype);
  free(ebdmw->disposition);
  free(ebdmw);
}

void eb_gui_send_data_message(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_contact * c=eb_get_contact(params[1], params[2]);
  eb_account * a;
  eb_stream * stream;
  eb_data_message_waiting * ebdmw;
  
  if(c==NULL)
  { eb_do_client_error(gui, "No such contact"); return ; }
  
  a=eb_get_message_account(c, params[3], params[4], params[5], SERVICE_CAN_SEND_DATA);
  
  if(a==NULL)
  { eb_do_client_error(gui, "Contact not online"); return; }
  
  if(!a->buddy_of->service->sc->send_data_message)
  {
    eb_do_client_error(gui, "This contact cannot accept data messages");
    return;
  }

  ebdmw=(eb_data_message_waiting *)malloc(sizeof(eb_data_message_waiting));
  ebdmw->filename=eb_get_filename("tmp");
  
  ebdmw->stream=eb_stream_file(gui, ebdmw->filename, EB_STREAM_IN, eb_data_message_uploaded, ebdmw, atoi(params[9]));
  if(ebdmw->stream==NULL)
  { free(ebdmw); perror("Stream open failed"); eb_do_client_error(gui, "Stream open failed"); return; }
  
  ebdmw->buddy=a;
  ebdmw->contenttype=strdup(params[6]);
  ebdmw->disposition=strdup(params[7]);
  ebdmw->length=atoi(params[8]);
  
  a->locked++;
}

void eb_gui_list_generic_actions(char * target, EList * actions, eb_gui * gui)
{
  EList * n;
  char ** notify_command;
  char buf[16];
  int a=3;
  int len=e_list_length(actions);

  notify_command=(char **)malloc((len+3)*sizeof(char *));

  notify_command[0]="list_actions";
  notify_command[1]=target;
  notify_command[2]=buf;

  sprintf(buf, "%d", len);

  for(n=actions; n!=NULL; n=n->next)
  {
    eb_action * action=(eb_action *)n->data;
    notify_command[a]=action->name;
    a++;
  }

  eb_gui_send(gui, notify_command, a);

  free(notify_command);
}


void eb_gui_list_actions(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  char * done_cmd[]={"list_actions_done"};

  eb_gui_list_generic_actions("group", eb_group_actions, gui);
  eb_gui_list_generic_actions("contact", eb_contact_actions, gui);
  eb_gui_list_generic_actions("buddy_generic", eb_buddy_actions, gui);
  eb_gui_list_generic_actions("general", eb_general_actions, gui);

  eb_gui_send(gui, done_cmd, 1);
}

void eb_gui_perform_action(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  void * target=NULL;
  EList * actions=NULL;
  EList * n;

  if(!strcmp(params[1], "buddy"))
  {
    eb_account * acc=eb_get_account(params[3], params[4], params[5]);
    if(acc==NULL) { return; }
    actions=acc->buddy_of->service->buddy_actions;
  } else if(!strcmp(params[1], "groupchat")) {
    eb_group_chat * chat=eb_get_group_chat(atoi(params[3]));
    if(chat==NULL) { eb_do_client_error(gui, "No such chat"); return; }
    target=chat;
    actions=chat->account->service->groupchat_actions;
  } else if(!strcmp(params[1], "group_users")) {
    eb_group_chat * chat=eb_get_group_chat(atoi(params[3]));
    if(chat==NULL) { eb_do_client_error(gui, "No such chat"); return; }

    actions=chat->account->service->group_user_actions;

    for(n=chat->users; n!=NULL; n=n->next)
    {
      eb_group_chat_user * user=(eb_group_chat_user *)n->data;
      if(!strcmp(user->handle, params[4]))
      { target=user; break; }
    }

    if(target==NULL)
    { eb_do_client_error(gui, "No such user in chatroom"); return; }
  } else if(!strcmp(params[1], "group")) {
    target=eb_get_group(params[3]);
    if(target==NULL)
    { eb_do_client_error(gui, "No such group"); return; }
    actions=eb_group_actions;
  } else if(!strcmp(params[1], "contact")) {
    target=eb_get_contact(params[3], params[4]);
    if(target==NULL)
    { eb_do_client_error(gui, "No such contact"); return; }
    actions=eb_contact_actions;
  } else if(!strcmp(params[1], "buddy_generic")) {
    eb_account * acc=eb_get_account(params[3], params[4], params[5]);
    if(acc==NULL) { eb_do_client_error(gui, "No such account"); return; }
    actions=eb_buddy_actions;
  } else if(!strcmp(params[1], "general")) {
    target=NULL;
    actions=eb_general_actions;
  } else {
    eb_do_client_error(gui, "Invalid action type");
    return;
  }

  for(n=actions; n!=NULL; n=n->next)
  {
    eb_action * action=(eb_action *)n->data;
    if(!strcmp(action->name, params[2]))
    { action->callback(target); return; }
  }

  eb_do_client_error(gui, "No such action");
}


void eb_buddy_logout(eb_account * acc)
{
  eb_gui_buddy_logout(NULL, acc);
}

void eb_gui_buddy_logout(eb_gui * gui, eb_account * acc)
{
  char * notify_command[]={"buddy_logout", "", "", ""};

  acc->status=EB_ACCOUNT_OFFLINE;

  notify_command[1]=acc->buddy_of->handle;
  notify_command[2]=acc->buddy_of->service_name;
  notify_command[3]=acc->handle;

  if(gui==NULL)
  {
    eb_gui_broadcast(notify_command, 4);
  } else {
    eb_gui_send(gui, notify_command, 4);
  }
}

void eb_buddy_update_status(eb_account * acc)
{
  eb_gui_buddy_update(NULL, acc);
}

void eb_gui_buddy_update(eb_gui * gui, eb_account * acc)
{
  char * notify_command[]={"buddy_status", "", "", "", "1", ""};
  char offline_fragment[]="0";
  char away_fragment[]="2";

  notify_command[1]=acc->buddy_of->handle;
  notify_command[2]=acc->buddy_of->service_name;
  notify_command[3]=acc->handle;
  if(acc->status==EB_ACCOUNT_OFFLINE) { notify_command[4]=offline_fragment; }
  if(acc->status==EB_ACCOUNT_AWAY) { notify_command[4]=away_fragment; }
  notify_command[5]=acc->status_string;

  if(gui==NULL)
  {
    eb_gui_broadcast(notify_command, 6);
  } else {
    eb_gui_send(gui, notify_command, 6);
  }
}

void eb_buddy_login(eb_account * acc)
{
  eb_gui_buddy_login(NULL, acc);
}

void eb_gui_buddy_login(eb_gui * gui, eb_account * acc)
{
  char * notify_command[]={"buddy_login", "", "", ""};

  if(acc->status==EB_ACCOUNT_OFFLINE)
  { acc->status=EB_ACCOUNT_ONLINE; }

  notify_command[1]=acc->buddy_of->handle;
  notify_command[2]=acc->buddy_of->service_name;
  notify_command[3]=acc->handle;

  if(gui==NULL)
  {
    eb_gui_broadcast(notify_command, 4);
  } else {
    eb_gui_send(gui, notify_command, 4);
  }
}

void eb_local_account_update(eb_local_account * acc)
{
  eb_gui_local_account_update(NULL, acc);
}

void eb_gui_local_account_update(eb_gui * gui, eb_local_account * acc)
{
  char * notify_command[]={"local_account_update", "", "", "0", "0", ""};

  notify_command[1]=acc->handle;
  notify_command[2]=acc->service_name;
  if(acc->connected)
  { notify_command[3]="1"; }
  if(acc->ready)
  { notify_command[4]="1"; }
  notify_command[5]=acc->status_string;

  if(gui==NULL)
  {
    eb_gui_broadcast(notify_command, 6);
  } else {
    eb_gui_send(gui, notify_command, 6);
  }
}

// Dialog stuff

void eb_show_error(char * message, char * title)
{
  eb_dialog * d=(eb_dialog *)malloc(sizeof(eb_dialog));
  eb_error_dialog * ed=(eb_error_dialog *)malloc(sizeof(eb_error_dialog));

  dialogs=e_list_append(dialogs, d);

  d->id=next_dialog_id++;
  d->data=ed;
  d->type=EB_ERROR_DIALOG;

  ed->message=strdup(message);
  ed->title=strdup(title);

  eb_gui_dialog_send(NULL, d);
}

void eb_gui_dialog_send(eb_gui * gui, eb_dialog * d)
{
  char ** notify_command=NULL;
  char error_dialog[]="error_dialog";
  char list_dialog[]="list_dialog";
  char yesno_dialog[]="yesno_dialog";
  char text_dialog[]="text_dialog";
  char buf[32];
  int num_commands=0;

  sprintf(buf, "%d", d->id);

  if(d->type==EB_ERROR_DIALOG)
  {
    eb_error_dialog * ed=(eb_error_dialog *)d->data;

    notify_command=(char **)malloc(4*sizeof(char *));
    notify_command[0]=error_dialog;
    notify_command[1]=buf;
    notify_command[2]=ed->title;
    notify_command[3]=ed->message;
    num_commands=4;
  }
  else if(d->type==EB_LIST_DIALOG)
  {
    eb_list_dialog * ld=(eb_list_dialog *)d->data;
    EList * n;
    int a, len;
    char buf2[16];

    len=e_list_length(ld->options);
    sprintf(buf2, "%d", len);

    notify_command=(char **)malloc((5+len)*sizeof(char *));
    notify_command[0]=list_dialog;
    notify_command[1]=buf;
    notify_command[2]=ld->title;
    notify_command[3]=ld->message;
    notify_command[4]=buf2;
    n=ld->options;

    for(a=0; a<len; a++)
    {
      notify_command[5+a]=(char *)n->data;
      n=n->next;
    }

    num_commands=5+len;
  }
  else if(d->type==EB_YESNO_DIALOG)
  {
    eb_yesno_dialog * ynd=(eb_yesno_dialog *)d->data;

    notify_command=(char **)malloc(5*sizeof(char *));
    notify_command[0]=yesno_dialog;
    notify_command[1]=buf;
    notify_command[2]=ynd->title;
    notify_command[3]=ynd->message;
    notify_command[4]=(ynd->default_result)?("1"):("0");

    num_commands=5;
  }
  else if(d->type==EB_TEXT_DIALOG)
  {
    eb_text_dialog * td=(eb_text_dialog *)d->data;

    notify_command=(char **)malloc(6*sizeof(char *));
    notify_command[0]=text_dialog;
    notify_command[1]=buf;
    notify_command[2]=td->title;
    notify_command[3]=td->message;
    notify_command[4]=td->initial_contents;
    notify_command[5]=(td->is_html)?("1"):("0");

    num_commands=6;
  }

  if(num_commands==0)
  {
    eb_show_error("Attempted to broadcast dialog of unknown type!", "Unknown dialog type");
    return;
  }

  if(notify_command==NULL)
  { eb_debug(DBG_CORE, "Attempted to send an invalid dialog\n"); return; }

  if(gui==NULL)
  {
    eb_gui_broadcast(notify_command, num_commands);
  } else {
    eb_gui_send(gui, notify_command, num_commands);
  }

  free(notify_command);
}

void eb_show_list_dialog(char * message, char * title, char **list, void (*callback)(char * response, void * tag), void * tag)
{
  eb_dialog * d=(eb_dialog *)malloc(sizeof(eb_dialog));
  eb_list_dialog * dialog=(eb_list_dialog *)malloc(sizeof(eb_list_dialog));
  int a;

  dialogs=e_list_append(dialogs, d);

  d->id=next_dialog_id++;
  d->data=dialog;
  d->type=EB_LIST_DIALOG;

  dialog->message=strdup(message);
  dialog->title=strdup(title);
  dialog->callback=callback;
  dialog->tag=tag;
  dialog->options=NULL;

  for(a=0; list[a]!=NULL; a++)
  {
    dialog->options=e_list_append(dialog->options, strdup(list[a]));
  }

  eb_gui_dialog_send(NULL, d);
}

void eb_show_text_dialog(char * message, char * title, char * value,
		int html, void (*callback)(char * response, void * tag),
		void * tag)
{
  eb_dialog * d=(eb_dialog *)malloc(sizeof(eb_dialog));
  eb_text_dialog * td=(eb_text_dialog *)malloc(sizeof(eb_text_dialog));

  dialogs=e_list_append(dialogs, d);

  d->id=next_dialog_id++;
  d->data=td;
  d->type=EB_TEXT_DIALOG;

  td->message=strdup(message);
  td->title=strdup(title);
  td->callback=callback;
  td->tag=tag;
  if (value == NULL)
    td->initial_contents=strdup("");
  else
    td->initial_contents=strdup(value);
  td->is_html=html;

  eb_gui_dialog_send(NULL, d);
}

void eb_show_yesno_dialog( char * message, char * title, int default_result,
		void (*callback)(int response, void * tag),
		void * tag)
{
  eb_dialog * d=(eb_dialog *)malloc(sizeof(eb_dialog));
  eb_yesno_dialog * ynd=(eb_yesno_dialog *)malloc(sizeof(eb_yesno_dialog));

  dialogs=e_list_append(dialogs, d);

  d->id=next_dialog_id++;
  d->data=ynd;
  d->type=EB_YESNO_DIALOG;

  ynd->message=strdup(message);
  ynd->title=strdup(title);
  ynd->callback=callback;
  ynd->tag=tag;
  ynd->default_result=default_result;

  eb_gui_dialog_send(NULL, d);
}

void eb_gui_resolve_dialog(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  EList * l;
  eb_dialog * d=NULL;
  char buf[32];
  char * resolve_command[]={"dialog_resolved", buf};
  int code=atoi(params[1]);

  for(l=dialogs; l!=NULL; l=l->next)
  {
    d=(eb_dialog *)l->data;
    if(d->id==code) { break; }
    d=NULL;
  }
  if(d==NULL)
  {
    eb_do_client_error(gui, "No such dialog");
    return;
  }

  switch(d->type)
  {
    case(EB_ERROR_DIALOG) : {
      eb_error_dialog * ed=(eb_error_dialog *)d->data;
      free(ed->title);
      free(ed->message);
      free(ed);
      break;
    }

    case(EB_LIST_DIALOG) : {
      eb_list_dialog * ld=(eb_list_dialog *)d->data;
      EList * n;
      int found=0;

      for(n=ld->options; n!=NULL; n=n->next)
      {
        if(!strcmp((char *)n->data, params[2]))
        { found=1; break; }
      }

      if(!found)
      {
        eb_do_client_error(gui, "Not a valid option");
        return;
      }

      ld->callback(params[2], ld->tag);

      for(n=ld->options; n!=NULL; n=n->next)
      { free(n->data); }

      free(ld->title);
      free(ld->message);
      e_list_free(ld->options);
      free(ld);
      break;
    }

    case(EB_YESNO_DIALOG) : {
      eb_yesno_dialog * ynd=(eb_yesno_dialog *)d->data;

      int result = 0;
      if(!strcmp("1", params[2]))
	result = 1;
      else if(!strcmp("0", params[2]))
	result = 0;
      else
      {
        eb_do_client_error(gui, "Not a valid option, must be \"1\" or \"0\"");
        return;
      }

      ynd->callback(result, ynd->tag);

      free(ynd->title);
      free(ynd->message);
      free(ynd);
      break;
    }

    case(EB_TEXT_DIALOG) : {
      eb_text_dialog * td=(eb_text_dialog *)d->data;

      td->callback(params[2], td->tag);

      free(td->title);
      free(td->message);
      free(td->initial_contents);
      free(td);
      break;
    }

    default : {} // to stop complaints that I haven't handled all the enums
  }

  if(d->type==EB_LIST_DIALOG)
  {
  }

  // If we got this far, the dialog was successfully resolved. Spread the word!

  dialogs=e_list_remove_link(dialogs, l);
  e_list_free_1(l);
  free(d);

  sprintf(buf, "%d", code);
  eb_gui_broadcast(resolve_command, 2);
}

void eb_gui_request_group_join(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_local_account * acc=eb_get_local_account(params[1], params[2]);

  if(acc==NULL) { eb_do_client_error(gui, "No such local account"); return; }

  if(!(acc->service->capabilities&SERVICE_CAN_GROUPCHAT))
  { eb_do_client_error(gui, "This service does not support group chat"); return; }

  if(!acc->ready)
  { eb_do_client_error(gui, "This local account is not ready to join a group chat"); return; }

  if(acc->service->sc->join_group_chat)
  { acc->service->sc->join_group_chat(acc, params[3]); }
}

void eb_gui_group_close(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_group_chat * chat=eb_get_group_chat(atoi(params[1]));

  if(chat==NULL) { eb_do_client_error(gui, "No such chat room"); return; }

  if(chat->account->service->sc->leave_group_chat)
  {
    chat->account->service->sc->leave_group_chat(chat);
  } else {
    eb_destroy_group_chat(chat);
  }
}

void eb_gui_group_hide(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_group_chat * chat=eb_get_group_chat(atoi(params[1]));

  if(chat==NULL) { eb_do_client_error(gui, "No such chat room"); return; }

  eb_hide_group_chat(chat);
}

void eb_gui_group_send(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_group_chat * chat=eb_get_group_chat(atoi(params[1]));

  if(chat==NULL) { eb_do_client_error(gui, "No such chat room"); return; }

  if(chat->account->service->sc->group_chat_send)
  {
    char * msg=strdup(params[2]);

    msg=chat->account->service->sc->group_chat_send(chat, msg);

    if(msg!=NULL)
    {
      char * notify_command[]={"group_chat_send", params[1], msg};

      eb_gui_broadcast(notify_command, 3);
      free(msg);
    }
  }
}

void eb_gui_group_invite(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_group_chat * chat=eb_get_group_chat(atoi(params[1]));

  if(chat==NULL) { eb_do_client_error(gui, "No such chat room"); return; }

  if(chat->account->service->sc->group_chat_invite)
  { chat->account->service->sc->group_chat_invite(chat, params[2]); }
}

void eb_gui_local_account_login(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_local_account * acc=eb_get_local_account(params[1], params[2]);

  if(acc==NULL) { eb_do_client_error(gui, "No such account"); return; }

  if(acc->service->sc->login)
  { acc->service->sc->login(acc); }
}

void eb_gui_local_account_logout(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_local_account * acc=eb_get_local_account(params[1], params[2]);

  if(acc==NULL) { eb_do_client_error(gui, "No such account"); return; }

  if(acc->service->sc->logout)
  { acc->service->sc->logout(acc); }
}

void eb_gui_set_state(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_local_account * acc=eb_get_local_account(params[1], params[2]);

  if(acc==NULL) { eb_do_client_error(gui, "No such account"); return; }

  if(acc->service->sc->set_current_state)
  { acc->service->sc->set_current_state(acc, params[3]); }
}

void eb_gui_add_group(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_group * group=eb_get_group(params[1]);

  if(group!=NULL) { eb_do_client_error(gui, "Group already exists"); return; }

  eb_add_group(params[1]);
}

void eb_gui_del_group(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_group * group=eb_get_group(params[1]);

  if(group==NULL) { eb_do_client_error(gui, "No such group"); return; }

  if(group->contacts!=NULL)
  { eb_do_client_error(gui, "Group is not empty"); return; }

  if(group->locked)
  { eb_show_error("This group cannot be removed because it is in use. Close any dialog boxes relating to this group and try again.", "Cannot remove group"); return; }

  eb_del_group(group);
}

void eb_gui_add_contact(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_group * group=eb_get_group(params[1]);
  eb_contact * contact=eb_get_contact(params[1], params[2]);

  if(group==NULL) { eb_do_client_error(gui, "No such group"); return; }
  if(contact!=NULL)
  { eb_do_client_error(gui, "A contact with this name already exists"); return; }

  eb_add_contact(group, params[2]);

  return;
}

void eb_gui_rename_contact(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_contact * contact=eb_get_contact(params[1], params[2]);
  if(contact==NULL) { eb_do_client_error(gui, "No such contact"); return; }

  eb_rename_contact(contact, params[3]);
}

void eb_gui_move_contact(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_contact * contact=eb_get_contact(params[1], params[2]);
  eb_group * group=eb_get_group(params[3]);
  if(contact==NULL) { eb_do_client_error(gui, "No such contact"); return; }
  if(group==NULL) { eb_do_client_error(gui, "No such target group"); return; }

  eb_move_contact(contact, group);
}

void eb_gui_del_contact(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_contact * contact=eb_get_contact(params[1], params[2]);
  if(contact==NULL) { eb_do_client_error(gui, "No such contact"); return; }
  if(contact->accounts!=NULL) { eb_do_client_error(gui, "Contact still has accounts"); }
  if(contact->locked)
  { eb_show_error("This contact cannot be removed because it is in use. Close any dialog boxes relating to this group and try again.", "Cannot remove contact"); return; }

  eb_del_contact(contact);
}

void eb_gui_add_account(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_account * adding=eb_get_account(params[3], params[4], params[5]);
  eb_contact * contact=eb_get_contact(params[1], params[2]);
  eb_local_account * buddy_of=eb_get_local_account(params[3], params[4]);

  if(contact==NULL) { eb_do_client_error(gui, "No such contact"); return; }
  if(buddy_of==NULL) { eb_do_client_error(gui, "No such local account"); return; }
  if(adding!=NULL)
  {
    char * buf=(char *)malloc(80+strlen(adding->contact->name)+strlen(adding->contact->group->name));

    sprintf(buf, "The account you are trying to add already exists, in this location:\nGroup: %s\nContact: %s",
      adding->contact->group->name, adding->contact->name);
    eb_show_error(buf, "Cannot add buddy");
    free(buf);
    return;
  }

  if(!buddy_of->ready)
  {
    eb_show_error("You cannot add a buddy to an account that is offline - sign on and try again.", "Cannot add account");
    return;
  }

  adding=(eb_account *)malloc(sizeof(eb_account));

  adding->handle=strdup(params[5]);
  adding->status_string=strdup("(Offline)");
  adding->status=EB_ACCOUNT_OFFLINE;
  adding->contact=contact;

  if(buddy_of->service->sc->add_user)
  {
    buddy_of->service->sc->add_user(buddy_of, adding);
  } else {
    eb_add_account(contact, buddy_of, adding);
  }
}

void eb_gui_move_account(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_account * account=eb_get_account(params[1], params[2], params[3]);
  eb_contact * contact=eb_get_contact(params[4], params[5]);

  if(account==NULL) { eb_do_client_error(gui, "No such buddy account"); return; }
  if(contact==NULL) { eb_do_client_error(gui, "No such destination contact"); return; }

  eb_move_account(account, contact);
}

void eb_gui_del_account(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_account * account=eb_get_account(params[1], params[2], params[3]);

  if(account==NULL) { eb_do_client_error(gui, "No such account"); return; }
  if(account->locked)
  {
    eb_show_error("This buddy cannot be deleted because it is in use. This may be because there are messages from it being held.", "Cannot delete buddy account");
    return;
  }

  if(account->buddy_of->service->sc->del_user)
  {
    account->buddy_of->service->sc->del_user(account);
  } else {
    eb_del_account(account);
  }
}

void eb_gui_add_local_account(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_local_account * acc=(eb_local_account *)malloc(sizeof(eb_local_account));
  eb_service * service=eb_get_service(params[2]);
  char * tmp_name;
  char * add_command[]={"add_local_account", params[1], params[2]};

  if(service==NULL || service==&dummy_service)
  { eb_do_client_error(gui, "No plugin loaded for this service"); return; }

  if(eb_get_local_account(params[1], params[2])!=NULL)
  { eb_do_client_error(gui, "This account already exists"); return; }

  acc->handle=strdup(params[1]);
  acc->service_name=strdup(params[2]);
  acc->status_string=strdup("Offline");
  acc->connected=acc->ready=0;
  acc->buddies=NULL;
  acc->group_chats=NULL;
  acc->service=&dummy_service;
  acc->locked=0;

  tmp_name=malloc(strlen(acc->handle)+strlen(acc->service_name)+5);
  strcpy(tmp_name, acc->handle);
  strcat(tmp_name, "_");
  strcat(tmp_name, acc->service_name);
  acc->config_key=eb_get_key(eb_get_key(registry, "local_accounts"), tmp_name);
  free(tmp_name);

  eb_put_value(acc->config_key, "handle", acc->handle);
  eb_put_value(acc->config_key, "service", acc->service_name);

  local_accounts=e_list_append(local_accounts, acc);

  eb_gui_broadcast(add_command, 3);

  map_orphaned_accounts();
  map_orphaned_buddies();
}

void eb_gui_del_local_account(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_local_account * acc=eb_get_local_account(params[1], params[2]);
  char * notify_command[]={"del_local_account", params[1], params[2]};
  EList * n;

  if(acc==NULL)
  { eb_do_client_error(gui, "No such local account"); return; }

  if(acc->connected && acc->service->sc->logout)
  { acc->service->sc->logout(acc); }

  if(acc->locked)
  {
    eb_show_error("This local account cannot be removed because it is in use.\nClose any dialog boxes or group chats relating to this account and try again.", "Cannot remove account");
    return;
  }

  for(n=acc->buddies; n!=NULL; n=n->next)
  {
    eb_account * buddy=(eb_account *)n->data;
    if(buddy->locked)
    {
      char * buf=(char *)malloc(1024+strlen(buddy->handle));
      sprintf(buf, "This local account cannot be removed because the buddy account \"%s\" is in use.\nClose any dialog boxes relating to this buddy and try again.", buddy->handle);
      eb_show_error(buf, "Cannot remove account");
      free(buf);
      return;
    }
  }

  // Here, we do a manual delete from memory of all these buddies, but, please note,
  // do NOT update the registry. This means that they auto-reappear if the user
  // re-adds this local account (if there is no matching local account, a buddy
  // account is ignored by the contact-list loader)

  for(n=acc->buddies; n!=NULL; n=n->next)
  {
    eb_account * buddy=(eb_account *)n->data;
    char * buddy_command[]={"delete_account", acc->handle, acc->service_name, buddy->handle};

    if(acc->service->sc->release_buddy_account)
    { acc->service->sc->release_buddy_account(buddy); } // links are still in place, so only that is required

    buddy->contact->accounts=e_list_remove(buddy->contact->accounts, buddy);
/*
    if(buddy->contact->accounts==NULL)
    {
      eb_group * group=buddy->contact->group;
      group->contacts=e_list_remove(group->contacts, buddy->contact);

    }
*/

    eb_gui_broadcast(buddy_command, 4);

    free(buddy->handle);
    free(buddy->status_string);
    free(buddy);
  }

  e_list_free(acc->buddies);

  if(acc->service->sc->release_local_account)
  { acc->service->sc->release_local_account(acc); }

  eb_del_key(registry, acc->config_key);
  local_accounts=e_list_remove(local_accounts, acc);

  free(acc->handle);
  free(acc->service_name);
  free(acc->status_string);
  free(acc);

  eb_gui_broadcast(notify_command, 3);
}

void eb_gui_ignore(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  char * ignore_command[]={"ignore_contact", params[1], params[2], params[3]};
  EList * n;
  eb_contact * contact=eb_get_contact(params[1], params[2]);

  if(contact==NULL) { eb_do_client_error(gui, "No such contact"); return; }

  if(contact->ignore!=NULL) { free(contact->ignore); }

  contact->ignore=strdup(params[3]);
  eb_put_value(contact->config_key, "ignore", "1");
  eb_put_value(contact->config_key, "ignore_message", contact->ignore);

  for(n=contact->accounts; n!=NULL; n=n->next)
  {
    eb_account * acc=(eb_account *)n->data;

    if(!acc->blocked && acc->buddy_of->ready && acc->buddy_of->service->sc->block_user)
    { acc->buddy_of->service->sc->block_user(acc); }
  }

  eb_gui_broadcast(ignore_command, 4);
}

void eb_gui_unignore(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  char * unignore_command[]={"unignore_contact", params[1], params[2]};
  EList * n;
  eb_contact * contact=eb_get_contact(params[1], params[2]);

  if(contact==NULL) { eb_do_client_error(gui, "No such contact"); return; }

  if(contact->ignore==NULL) { eb_do_client_error(gui, "Contact is not ignored"); return; }

  free(contact->ignore);
  contact->ignore=NULL;
  eb_del_value(contact->config_key, "ignore");
  eb_del_value(contact->config_key, "ignore_message");
  
  for(n=contact->accounts; n!=NULL; n=n->next)
  {
    eb_account * acc=(eb_account *)n->data;

    if(acc->blocked && acc->buddy_of->ready && acc->buddy_of->service->sc->unblock_user)
    { acc->buddy_of->service->sc->unblock_user(acc); }
  }

  eb_gui_broadcast(unignore_command, 3);
}

void eb_do_client_error(eb_gui * gui, char * error)
{
  char * notify_command[]={"client_error", error};
  eb_gui_send(gui, notify_command, 2);
}

// Prefs stuff

void eb_gui_list_pref_page(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  char * list_pref_page[]={"list_pref_page", params[1], ""};
  char * begin_subpages[]={"list_subpages", params[1], ""};
  char * list_subpage[]={"list_subpage", params[1], "", ""};
  char * end_subpages[]={"list_subpages_done"};
  char * begin_components[]={"list_components", params[1]};
  char * list_component[]={"list_component", params[1], "", "", "", ""}; // expand this
  char * end_components[]={"list_components_done"};
  char * end_pref_page[]={"list_pref_page_done"};
  eb_pref_page * page=eb_get_pref_page(params[1]);
  EList * n;

  if(page==NULL) { eb_do_client_error(gui, "No such page"); return; }

  list_pref_page[2]=page->title;
  eb_gui_send(gui, list_pref_page, 3);
  eb_gui_send(gui, begin_subpages, 2);

  for(n=page->subpages; n!=NULL; n=n->next)
  {
    eb_pref_page * spage=(eb_pref_page *)n->data;
    list_subpage[2]=spage->name;
    list_subpage[3]=spage->title;
    eb_gui_send(gui, list_subpage, 4);
  }
  eb_gui_send(gui, end_subpages, 1);

  eb_gui_send(gui, begin_components, 2);

  for(n=page->components; n!=NULL; n=n->next)
  {
    eb_pref * pref=(eb_pref *)n->data;
    list_component[2]=eb_pref_type_names[pref->type];
    list_component[3]=pref->name;
    list_component[4]=pref->title;
    list_component[5]=eb_get_value(pref->target, pref->valuename);
    eb_gui_send(gui, list_component, 6);

    if(pref->type==EB_PREF_OPTION)
    {
      eb_option_data * eod=(eb_option_data *)pref->data;
      int a, listlen=e_list_length(eod->options);
      EList * n2;
      char ** option_data=(char **)malloc((listlen+4)*sizeof(char *));
      char buf[16];

      option_data[0]="option_data";
      option_data[1]=page->name;
      option_data[2]=pref->name;
      sprintf(buf, "%d", listlen);
      option_data[3]=buf;
      a=4;

      for(n2=eod->options; n2!=NULL; n2=n2->next)
      {
        option_data[a++]=(char *)n2->data;
      }

      eb_gui_send(gui, option_data, listlen+4);

      free(option_data);
    }
  }
  eb_gui_send(gui, end_components, 1);

  eb_gui_send(gui, end_pref_page, 1);
}

void eb_gui_set_pref_value(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_pref_page * page=eb_get_pref_page(params[1]);
  eb_pref * pref;

  if(page==NULL) { eb_do_client_error(gui, "No such page"); return; }

  pref=eb_get_component(page, params[2]);
  if(pref==NULL) { eb_do_client_error(gui, "No such component"); return; }

  eb_put_value(pref->target, pref->valuename, params[3]);
  if(pref->callback)
  {
    pref->callback(pref, pref->callback_tag);
  }
}

// Message stuff

void eb_gui_got_message(eb_account * remote, eb_html_item * message)
{
  char * notify_command[]={"message_receive", "", "", "", "", "", ""};
  char * html=NULL;
  char * plain=NULL;
  EList * n;

  notify_command[1]=remote->contact->group->name;
  notify_command[2]=remote->contact->name;
  notify_command[3]=remote->buddy_of->handle;
  notify_command[4]=remote->buddy_of->service_name;
  notify_command[5]=remote->handle;

  // manually walk through the UIs, so I can send them the HTML or plaintext version, as I wish

  for(n=GUIs; n!=NULL; n=n->next)
  {
    eb_gui * gui=(eb_gui *)n->data;

    if(gui->html)
    {
      if(html==NULL) { html=eb_html_render(message); }
      notify_command[6]=html;
    } else {
      if(plain==NULL) { plain=eb_html_render_plain(message); }
      notify_command[6]=plain;
    }    

    eb_gui_send(gui, notify_command, 7);
  }

  if(html!=NULL) { free(html); }
  if(plain!=NULL) { free(plain); }
}

void eb_gui_notify_3rdperson(eb_contact * contact, eb_html_item * message)
{
  char * notify_command[]={"notify_3rdperson", "", "", ""};
  char * html=NULL;
  char * plain=NULL;
  EList * n;

  notify_command[1]=contact->group->name;
  notify_command[2]=contact->name;

  for(n=GUIs; n!=NULL; n=n->next)
  {
    eb_gui * gui=(eb_gui *)n->data;

    if(gui->html)
    {
      if(html==NULL) { html=eb_html_render(message); }
      notify_command[3]=html;
    } else {
      if(plain==NULL) { plain=eb_html_render_plain(message); }
      notify_command[3]=plain;
    }    

    eb_gui_send(gui, notify_command, 4);
  }

  if(html!=NULL) { free(html); }
  if(plain!=NULL) { free(plain); }
}

void eb_gui_sent_message(eb_account * remote, eb_html_item * message)
{
  char * notify_command[]={"message_send", "", "", "", "", "", ""};
  char * html=NULL;
  char * plain=NULL;
  EList * n;

  notify_command[1]=remote->contact->group->name;
  notify_command[2]=remote->contact->name;
  notify_command[3]=remote->buddy_of->handle;
  notify_command[4]=remote->buddy_of->service_name;
  notify_command[5]=remote->handle;
  for(n=GUIs; n!=NULL; n=n->next)
  {
    eb_gui * gui=(eb_gui *)n->data;

    if(gui->html)
    {
      if(html==NULL) { html=eb_html_render(message); }
      notify_command[6]=html;
    } else {
      if(plain==NULL) { plain=eb_html_render_plain(message); }
      notify_command[6]=plain;
    }    

    eb_gui_send(gui, notify_command, 7);
  }
  if(html!=NULL) { free(html); }
  if(plain!=NULL) { free(plain); }
}

void eb_gui_data_message(eb_account * remote, char * filename, char * contenttype, char * disposition, char * suggested_filename, int length, int direction, int held)
{
  EList * n;
  int id=eb_stream_next_id();
  char buf[16], lbuf[16];
  sprintf(buf, "%d", id);
  sprintf(lbuf, "%d", length);
  
  for(n=GUIs; n!=NULL; n=n->next)
  {
    eb_gui * gui=(eb_gui *)n->data;
    char * notify_cmd[]={NULL, remote->contact->group->name, remote->contact->name, remote->buddy_of->handle, remote->buddy_of->service_name, remote->handle, contenttype, disposition, lbuf, buf};
    
    if(direction==EB_MSG_SEND || direction==EB_MSG_INLINE_SEND)
    { notify_cmd[0]=held?"held_sent_data_message":"send_data_message"; }
    else
    { notify_cmd[0]=held?"held_data_message":"receive_data_message"; }
    
    eb_gui_send(gui, notify_cmd, 10);
    eb_stream_file(gui, filename, EB_STREAM_OUT, NULL, NULL, id);
  }
}

void eb_gui_set_away(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_set_away(params[1], params[2]);
}

void eb_gui_unset_away(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_unset_away();
}

void eb_gui_hold_messages(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_hold_messages(params[1][0]=='1');
}

void eb_gui_get_held_messages(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  char * notify_cmd[8];
  char buf[32];
  char * end_cmd[]={"held_messages_done", buf};
  int total=0;
  EList * n;

  for(n=eb_held_messages; n!=NULL; n=n->next)
  {
    EList * n2;
    int ncmd;
    eb_held_message * msg=(eb_held_message *)n->data;
    char * plain=NULL;
    char * html=NULL;

    
    if(msg->direction==EB_MSG_INLINE_SEND || msg->direction==EB_MSG_INLINE_RECV)
    {
      printf("Data message (type %s)\n", msg->contenttype);
      eb_gui_data_message(msg->buddy, msg->filename, msg->contenttype, "inline", "", 0, msg->direction, 0);
      msg->buddy->locked--;
      free(msg->contenttype);
      free(msg->filename);
      free(msg);
      total++;
      continue;
    }
    
    if(msg->direction==EB_MSG_SEND)
    {
      notify_cmd[0]="held_sent_message";
    } else {
      notify_cmd[0]="held_message";
    }

    sprintf(buf, "%lu", time(NULL)-msg->when);
    notify_cmd[1]=buf;

    if(msg->direction==EB_MSG_3RDPERSON)
    {
      notify_cmd[0]="held_3rdperson";
      notify_cmd[2]=msg->contact->group->name;
      notify_cmd[3]=msg->contact->name;
      ncmd=4;
      msg->contact->locked--;
    } else {
      notify_cmd[2]=msg->buddy->contact->group->name;
      notify_cmd[3]=msg->buddy->contact->name;
      notify_cmd[4]=msg->buddy->buddy_of->handle;
      notify_cmd[5]=msg->buddy->buddy_of->service_name;
      notify_cmd[6]=msg->buddy->handle;
      ncmd=7;
      msg->buddy->locked--;
    }

    for(n2=GUIs; n2!=NULL; n2=n2->next)
    {
      eb_gui * g=(eb_gui *)n2->data;

      if(g->html)
      {
        if(html==NULL) { html=eb_html_render(msg->message); }
        notify_cmd[ncmd]=html;
      } else {
        if(plain==NULL) { plain=eb_html_render_plain(msg->message); }
        notify_cmd[ncmd]=plain;
      }

      eb_gui_send(g, notify_cmd, ncmd+1);
    }

    if(html!=NULL) { free(html); }
    if(plain!=NULL) { free(plain); }
    eb_html_destroy(msg->message);
    free(msg);

    total++;
  }

  e_list_free(eb_held_messages);
  eb_held_messages=NULL;

  sprintf(buf, "%d", total);
  eb_gui_broadcast(end_cmd, 2);
}

static void streamerr_cb(int error, void * data)
{
  if(error!=0) { eb_show_error("Could not transfer the requested chat log. Is the ID correct?", "Log error"); }
}

void eb_gui_get_stream(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  // This will be a generalised call to request a stream from any of several sources.
  // (chat logs, emoticons, completed file transfers...)
  // ...but it's just chat logs for now.
  int id=eb_stream_next_id();
  char buf[16];
  char * notify_cmd[4]={"get_stream", params[0], params[1], buf};
  
  sprintf(buf, "%d", id);
  
  if(!strcmp(params[1], "log"))
  {
    char * fnam;
    if(strstr(params[2], "/")) { eb_do_client_error(gui, "Invalid path"); return; }
    fnam=(char *)malloc(strlen(eb_config_dir)+strlen(params[2])+16);
    sprintf(fnam, "%slogs/%s", eb_config_dir, params[2]);
    eb_gui_send(gui, notify_cmd, 4);
    eb_stream_file(gui, fnam, EB_STREAM_OUT, streamerr_cb, gui, id);
    free(fnam);
    return;
  }
  
  eb_do_client_error(gui, "Stream class not found");
}

void eb_gui_stream_data(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_stream * stream=eb_stream_get(gui, atoi(params[1]), EB_STREAM_IN);
  if(stream==NULL)
  { eb_do_client_error(gui, "No such stream"); return; }
  eb_stream_incoming(stream, params[2], lengths[2]);
}

void eb_gui_stream_abort(eb_gui * gui, char ** params, int * lengths, int num_params)
{
  eb_stream * stream=eb_stream_get(gui, atoi(params[1]),
    (!strcmp(params[0], "down_stream_abort")?(EB_STREAM_OUT):(EB_STREAM_IN)));
  if(stream==NULL)
  { eb_do_client_error(gui, "No such stream"); return; }
  eb_stream_cleanup(stream, (!strcmp(params[0], "stream_ends"))?(EB_STREAM_SUCCESS):(EB_STREAM_ABORT));
}

void eb_gui_ping(eb_gui * gui, char * params, int num_params)
{ 
  
}
