/*
 * Copyright (C) 2000,2001,2004 Peter Samuelson <peter@p12n.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2,
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <sys/types.h>

/*
 * Encode MIME BASE-64 (see RFC 1341)
 * Output buffer is assumed to be large enough
 * (it needs to be 4/3 of in_len, plus a small bit of overhead for newlines)
 * Returns length of output buffer
 */
size_t encode_base64(char *out, const char *in, size_t in_len)
{
	int i;
	static char base64[] =
		"ABCDEFGHIJKLMNOPQRSTUVWXYZ"
		"abcdefghijklmnopqrstuvwxyz0123456789+/";
	size_t out_len = 0;

	for (i=0; i<in_len; i+=3) {
		unsigned char ch[4];
		char *fmt;

		ch[0] = in[i];
		ch[1] = i+1 < in_len ? in[i+1] : 0;
		ch[2] = i+2 < in_len ? in[i+2] : 0;

		ch[3] = base64[ 0x3f & ch[2] ];
		ch[2] = base64[ 0x3f & ((ch[1] << 2) | (ch[2] >> 6)) ];
		ch[1] = base64[ 0x3f & ((ch[0] << 4) | (ch[1] >> 4)) ];
		ch[0] = base64[ ch[0] >> 2 ];

		if (i+2 >= in_len)
			ch[3] = '=';
		if (i+1 == in_len)
			ch[2] = '=';

		out_len += sprintf(out+out_len, "%c%c%c%c", ch[0],ch[1],ch[2],ch[3]);
	}
	return out_len;
}


/*
 * Decode MIME BASE-64 (see RFC 1341), "in situ"
 * This is possible because the output is always shorter than the input
 * Returns the final length of the buffer, 0 for invalid char(s)
 */
size_t decode_base64(char *buf, size_t len)
{
	/* how much each ASCII character is worth (0-63) */
	/* ...plus one, because zero is so handy for 'invalid' */
	static int base64[256] = {
		 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,63, 0, 0, 0,64, /* + / */
		53,54,55,56,57,58,59,60,61,62, 0, 0, 0,-1, 0, 0, /* 0-9 = */
		 0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,15, /* A-O */
		16,17,18,19,20,21,22,23,24,25,26, 0, 0, 0, 0, 0, /* P-Z */
		 0,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41, /* a-o */
		42,43,44,45,46,47,48,49,50,51,52, 0, 0, 0, 0, 0, /* p-z */
	};
	int i, j;
	size_t total_len = 0;
	unsigned char *ub = buf;
	
	for (i=0; i < len/4; i++) {
		total_len += 3;
		for (j=0; j<4; j++) {
			int tmp = base64[ub[4*i+j]];
			if (tmp == -1)
				total_len--;
			else if (!tmp) {
				/* warn ("unexpected character in input"); */
				return 0;
			} else
				ub[4*i+j] = tmp - 1;
		}

		ub[3*i]   = ub[4*i]   << 2 | ub[4*i+1] >> 4;
		ub[3*i+1] = ub[4*i+1] << 4 | ub[4*i+2] >> 2;
		ub[3*i+2] = ub[4*i+2] << 6 | ub[4*i+3];
	}
	return total_len;
}
