/***************************************************************************
                   yahoo.c - Yahoo! Messenger glue code
                             -------------------
                     (C) 2002 by the Everybuddy team
                            www.everybuddy.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <stdarg.h>
#include <string.h>

#include <errno.h>

#include "contactlist.h"
#include "accounts.h"
#include "dialog.h"
#include "elist.h"
#include "util.h"
#include "globals.h"
#include "groupchat.h"
#include "message_parse.h"
#include "plugin.h"
#include "plugin_api.h"
#include "prefs.h"
#include "registry.h"
#include "services.h"
#include "stream.h"

#include "yahoo_list.h"
#include "yahoo2.h"
#include "yahoo2_callbacks.h"
#include "yahoo_util.h"
#include "yahoo_debug.h"

typedef struct _eb_ylad
{
  int id;
  int upcoming;
  int invisible;
  eb_pref_page * prefpage;
} eb_ylad;

#define GET_LAD(la) ((eb_ylad *)la->protocol_data)

static eb_service_callbacks yahoo_callbacks;

int yahoo_plugin_init(void);
int yahoo_plugin_finish(void);

eb_plugin_info yahoo_LTX_plugin_info =
{
  "Yahoo! messenger",
  "Philip Tellis' libyahoo2",
  "(0.7.4)", "2004-01-21",
  yahoo_plugin_init,
  yahoo_plugin_finish
};

eb_service yahoo_service =
{
  "Yahoo",
  "Philip Tellis' libyahoo2",
  &yahoo_callbacks,
  SERVICE_CAN_OFFLINE,
  "#de9400",
  NULL,
  NULL,
  NULL,
  NULL
};

static eb_local_account * get_local(int id)
{
  EList * n;
  for (n = local_accounts; n != NULL; n = n->next)
  {
    eb_local_account * la = (eb_local_account *)n->data;
    if (la->service == &yahoo_service && GET_LAD(la)->id == id)
    {
      return la;
    }
  }
  return NULL;
}

static void set_current_state(eb_local_account * la, char * state);

static void login(eb_local_account * la)
{
  eb_ylad * lad;
  int istate;

  la->connected = 1;
  la->ready = 0;
  eb_local_account_update(la);

  lad = GET_LAD(la);
  if (lad->id != -1)
  {
    yahoo_close(lad->id);
    lad->id = -1;
  }
  lad->id = yahoo_init(la->handle, eb_get_value(la->config_key, "password"));

  lad->invisible = (eb_get_value(la->config_key, "yahoo/login_invisible")[0] == '1');
  istate = (lad->invisible) ? (YAHOO_STATUS_INVISIBLE) : (YAHOO_STATUS_AVAILABLE);
  yahoo_login(lad->id, istate);

  set_current_state(la, (lad->invisible) ? "Invisible" : "Online");
}

static void logout(eb_local_account * la)
{
  EList * en;
  eb_ylad * lad = GET_LAD(la);
  if (!la->connected)
  {
    return; /* already done */
  }
  for (en = la->buddies; en != NULL; en = en->next)
  {
    eb_account * acc = (eb_account *)en->data;
    if (acc->status != EB_ACCOUNT_OFFLINE)
    {
      free(acc->status_string);
      acc->status_string = strdup("Offline");
      acc->status = EB_ACCOUNT_OFFLINE;
      eb_buddy_update_status(acc);
      eb_buddy_logout(acc);
    }
  }

  free(la->status_string);
  la->status_string = strdup("Offline");
  la->ready = la->connected = 0;
  eb_local_account_update(la);
  if (lad->id != -1)
  {
    yahoo_logoff(lad->id);
    lad->id = -1;
  }
}

static void set_current_state(eb_local_account * la, char * state)
{
  eb_ylad * lad = GET_LAD(la);
  int st_id;
  char * st_str = NULL;
  
  if (!strcmp(state, "Offline"))
  {
    logout(la);
    return;
  }
  else
  {
    /* Check if we are connected yet */
    if (!la->connected)
    {
      /* connect to the server */
      login(la);
      return; /* so we don't do this processing twice */
    }

    if (!strcmp(state, "Online"))
    {
      st_id = YAHOO_STATUS_AVAILABLE;
    }
    else if (!strcmp(state, "BRB"))
    {
      st_id = YAHOO_STATUS_BRB;
    }
    else if (!strcmp(state, "Busy"))
    {
      st_id = YAHOO_STATUS_BUSY;
    }
    else if (!strcmp(state, "Not at Home"))
    {
      st_id = YAHOO_STATUS_NOTATHOME;
    }
    else if (!strcmp(state, "Not at Desk"))
    {
      st_id = YAHOO_STATUS_NOTATDESK;
    }
    else if (!strcmp(state, "Not in Office"))
    {
      st_id = YAHOO_STATUS_NOTINOFFICE;
    }
    else if (!strcmp(state, "Phone"))
    {
      st_id = YAHOO_STATUS_ONPHONE;
    }
    else if (!strcmp(state, "Vacation"))
    {
      st_id = YAHOO_STATUS_ONVACATION;
    }
    else if (!strcmp(state, "Lunch"))
    {
      st_id = YAHOO_STATUS_OUTTOLUNCH;
    }
    else if (!strcmp(state, "Stepped Out"))
    {
      st_id = YAHOO_STATUS_STEPPEDOUT;
    }
    else if (!strcmp(state, "Invisible"))
    {
      st_id = YAHOO_STATUS_INVISIBLE;
      lad->invisible = 1;
    }
    else if (!strcmp(state, "Idle"))
    {
      st_id = YAHOO_STATUS_IDLE;
    }
    else
    {
      st_id = YAHOO_STATUS_CUSTOM;
      st_str = state;
    }
	
    if (la->ready)
    {
      /* Only set the away message if the connection is ready */
      yahoo_set_away(lad->id, st_id, st_str, away_msg != NULL);
      free(la->status_string);
      la->status_string = strdup(state);
      eb_local_account_update(la);
    }
  }
}

static void set_away(eb_local_account * la, char * shortmsg, char * longmsg)
{
  char * st = shortmsg;
  char * lwr = strdup(longmsg);
  int a;
  for (a = 0; lwr[a] != '\0'; a++)
  {
    lwr[a] = tolower(lwr[a]);
  }
  
  if (strstr(lwr, "busy"))
  {
    st = "Busy";
  }
  else if (strstr(lwr, "brb") || strstr(lwr, "be right back"))
  {
    st = "BRB";
  }
  else if (strstr(lwr, "phone"))
  {
    st = "Phone";
  }
  else if (strstr(lwr, "eat") || strstr(lwr, "lunch") || strstr(lwr, "tea")
    || strstr(lwr, "supper") || strstr(lwr, "dinner"))
  {
    st = "Lunch";
  }
  
  if (!GET_LAD(la)->invisible)
  {
    set_current_state(la, st);
  }
  
  free(lwr);
}

static void unset_away(eb_local_account * la)
{
  if (!GET_LAD(la)->invisible)
  {
    yahoo_set_away(GET_LAD(la)->id, YAHOO_STATUS_AVAILABLE, NULL, 0);
    free(la->status_string);
    la->status_string = strdup("Online");
  }
}

static void setup_local_account(eb_local_account * la)
{
  char * buf1;
  char * buf2;
  eb_ylad * lad = (eb_ylad *)malloc(sizeof(eb_ylad));
  
  la->protocol_data = lad;
  
  lad->id = -1;
  lad->upcoming = 0;
  lad->invisible = 0;

  buf1 = (char *)malloc(strlen(la->handle) + strlen(la->service_name) + 2);
  sprintf(buf1, "%s_%s", la->handle, la->service_name);
  buf2 = (char *)malloc(strlen(la->handle) + strlen(la->service_name) + 4);
  sprintf(buf2, "%s (%s)", la->handle, la->service_name);
  lad->prefpage = eb_make_pref_page(account_prefs, buf1, buf2);
  free(buf1);
  free(buf2);
  
  eb_add_component(lad->prefpage, EB_PREF_PASSWORD, "password", "Password:", la->config_key, "password", NULL, NULL);
  eb_add_component(lad->prefpage, EB_PREF_TOGGLE, "invisible", "Sign on as invisible", la->config_key, "yahoo/login_invisible", NULL, NULL);
}

static void release_local_account(eb_local_account * la)
{
  eb_ylad * lad = GET_LAD(la);
  eb_destroy_pref_page(lad->prefpage);
  free(lad);
}

static void add_user(eb_local_account * buddy_of, eb_account * acc)
{
  yahoo_add_buddy(GET_LAD(buddy_of)->id, acc->handle, acc->contact->group->name, NULL);
  eb_add_account(acc->contact, buddy_of, acc);
}

static void del_user(eb_account * acc)
{
  const YList * n;
  for (n = yahoo_get_buddylist(GET_LAD(acc->buddy_of)->id); n != NULL; n = n->next)
  {
    struct yahoo_buddy * yb = (struct yahoo_buddy *)n->data;
    if (yb!=NULL && !strcmp(yb->id, acc->handle))
    {
      yahoo_remove_buddy(GET_LAD(acc->buddy_of)->id, yb->id, yb->group);
    }
  }
  
  eb_destroy_inlined_data_state((eb_inlined_data_state *)acc->protocol_data);
  
  eb_del_account(acc);
}

static void change_group(eb_account * acc, char * new_group)
{
  const YList * n;
  int done = 0;
  for (n = yahoo_get_buddylist(GET_LAD(acc->buddy_of)->id); n != NULL; n = n->next)
  {
    struct yahoo_buddy * yb = (struct yahoo_buddy *)n->data;
    if (!strcmp(yb->id, acc->handle))
    {
      if (!done)
      {
        done = 1;
        yahoo_change_buddy_group(GET_LAD(acc->buddy_of)->id, yb->id, yb->group, new_group);
      }
      else
      {
        yahoo_remove_buddy(GET_LAD(acc->buddy_of)->id, yb->id, yb->group);
      }
    }
  }
}

char * send_im(eb_account * to, char * msg)
{
  char * sbuf;
  char * rbuf;
  
  eb_ylad * lad=GET_LAD(to->buddy_of);
  
  rbuf = eb_filter_string(EB_PREFILTER_OUT, msg, to);
  if (rbuf == NULL)
  {
    return NULL;
  }
  
  sbuf = eb_filter_string(EB_POSTFILTER_OUT, strdup(rbuf), to);
  if (sbuf == NULL)
  {
    free(rbuf);
    return NULL;
  }
  
  yahoo_send_im(lad->id, NULL, to->handle, sbuf, 1);
  
  free(sbuf);
  
  return rbuf;
}

void send_data_message(eb_account * to, char * filename, char * contenttype, char * disposition, int reported_length)
{
  eb_ylad * lad=GET_LAD(to->buddy_of);

  printf("Sending with content-type %s\n", contenttype);

  if(!strcmp(disposition, "inline") && reported_length<5500)
  {
    char * buf[5];
    int a, num_msg;
    
    for(a=0; a<5; a++) { buf[a]=(char *)malloc(2048); }
    
    eb_package_inlined_data(filename, contenttype, reported_length, 800,  buf, &num_msg);
    
    for(a=0; a<num_msg; a++)
    {
      yahoo_send_im(lad->id, NULL, to->handle, buf[a], 1);
      printf("Sending: %s\n", buf[a]);
    }
  
    for(a=0; a<5; a++) { free(buf[a]); }
    
    eb_sent_inline_data_message(to, filename, contenttype);
    
    return;
  }
  
  eb_notify_3rdperson(to->contact, "<font color=red><i>Yahoo cannot yet send large or non-inlined data messages");
}

int yahoo_plugin_init(void)
{
  EList * states = NULL;
  printf("Hi-ho, hi-ho, Yahoo here we go!\n");
  fflush(stdout);
  /*yahoo_set_log_level(YAHOO_LOG_DEBUG);*/

  memset(&yahoo_callbacks, 0, sizeof(yahoo_callbacks));
  
  yahoo_callbacks.login = login;
  yahoo_callbacks.logout = logout;
  yahoo_callbacks.set_current_state = set_current_state;
  yahoo_callbacks.set_away = set_away;
  yahoo_callbacks.unset_away = unset_away;
  yahoo_callbacks.setup_local_account = setup_local_account;
  yahoo_callbacks.release_local_account = release_local_account;
  yahoo_callbacks.add_user = add_user;
  yahoo_callbacks.del_user = del_user;
  yahoo_callbacks.change_group = change_group;
  yahoo_callbacks.send_im = send_im;
  yahoo_callbacks.send_data_message = send_data_message;
  
  states = e_list_append(states, strdup("Online"));
  states = e_list_append(states, strdup("BRB"));
  states = e_list_append(states, strdup("Busy"));
  states = e_list_append(states, strdup("Not at Home"));
  states = e_list_append(states, strdup("Not at Desk"));
  states = e_list_append(states, strdup("Not in Office"));
  states = e_list_append(states, strdup("Phone"));
  states = e_list_append(states, strdup("Vacation"));
  states = e_list_append(states, strdup("Lunch"));
  states = e_list_append(states, strdup("Stepped Out"));
  states = e_list_append(states, strdup("Invisible"));
  states = e_list_append(states, strdup("Idle"));
  states = e_list_append(states, strdup("Offline"));
  yahoo_service.states = states;
  
  eb_load_service(&yahoo_service);

  return 0;
}

int yahoo_plugin_finish(void)
{
  return 0;
}

 /**************************************************************************
                        YAHOO LIBRARY CALLBACKS
  **************************************************************************/

/*
 * Name: ext_yahoo_login_response
 * 	Called when the login process is complete
 * Params:
 * 	id   - the id that identifies the server connection
 * 	succ - enum yahoo_login_status
 * 	url  - url to reactivate account if locked
 */
void ext_yahoo_login_response(int id, int succ, char *url)
{
  eb_local_account * la = get_local(id);
  if (la == NULL)
  {
    printf("Martian Yahoo connection\n");
    fflush(stdout);
    return;
  }

  switch (succ)
  {
    case YAHOO_LOGIN_SOCK:
    {
      if (la->ready)
      {
	printf("Bad Yahoo connection but already valid, must be stale\n");
	fflush(stdout);
	return;
      }
      logout(la);
      break;
    }
    case YAHOO_LOGIN_OK:
    {
      if (la->ready)
      {
	printf("Redundant Yahoo connection\n");
	fflush(stdout);
	return;
      }
      la->ready = 1;
      free(la->status_string);
      la->status_string = strdup(GET_LAD(la)->invisible ? "Invisible" : "Online");
      break;
    }
    case YAHOO_LOGIN_DUPL:
    {
      printf("Yahoo forced offline by duplicate login\n");
      fflush(stdout);
      logout(la);
      break;
    }
    case YAHOO_SOCK_ERR:
    default:
    {
      char buf[512];
      perror("Weird error");
      sprintf(buf, "Disconnected from the Yahoo network - error code %d", succ);
      eb_show_error(buf, "Yahoo error");
      logout(la);
      printf("Login error %d for %s (Yahoo id %d)\n", succ, la->handle, id);
      fflush(stdout);
    }
  }
  eb_local_account_update(la);
}




/*
 * Name: ext_yahoo_got_buddies
 * 	Called when the contact list is got from the server
 * Params:
 * 	id   - the id that identifies the server connection
 * 	buds - the buddy list
 */
void ext_yahoo_got_buddies(int id, YList * buds)
{
  eb_local_account * la = get_local(id);
  YList * n;
  EList * en;
  int p = 0;
  
  if (la == NULL)
  {
    printf("Got buddies on a martian account!\n");
    fflush(stdout);
    return;
  }
  
  for (en = la->buddies; en != NULL; en = en->next)
  {
    eb_account * acc = (eb_account *)en->data;
    if(acc->protocol_data!=NULL)
    { eb_destroy_inlined_data_state((eb_inlined_data_state *)acc->protocol_data); }
    acc->protocol_data = NULL;
  }

  for (n = buds; n != NULL; n = n->next)
  {
    struct yahoo_buddy * yb = (struct yahoo_buddy *)n->data;
    eb_account * acc = eb_get_account(la->handle, "Yahoo", yb->id);
    if (acc == NULL)
    {
      acc = eb_add_account_plus(yb->group, (yb->real_name != NULL) ? (yb->real_name) : (yb->id), la, yb->id);
    }
    acc->protocol_data = &p; // Just something not-NULL, we reset it later
  }

  for (en = la->buddies; en != NULL; en = en->next)
  {
    eb_account * acc = (eb_account *)en->data;
    
    if (acc->protocol_data == NULL) /* wasn't on the list */
    {
      p = 1;
      free(acc->status_string);
      acc->status_string = strdup("[NOT ON LIST]");
      acc->status=EB_ACCOUNT_AWAY;
      eb_buddy_update(acc);
    }
    
    acc->protocol_data=NULL;
  }
  
  if (p)
  {
    eb_show_error("Yahoo! Buddy list reverse mismatch: You have one or more buddies on your buddy list but not on the Yahoo server.\n\n", "Warning");
  }
}




/*
 * Name: ext_yahoo_got_ignore
 * 	Called when the ignore list is got from the server
 * Params:
 * 	id   - the id that identifies the server connection
 * 	igns - the ignore list
 */
void ext_yahoo_got_ignore(int id, YList * igns)
{
  YList* n;
  printf("Yahoo ignore list downloaded OK. Shame we don't do anything with it\nyet, really.\n");
  for (n = igns; n != NULL; n = n->next)
  {
    struct yahoo_buddy * yb = (struct yahoo_buddy *)n->data;
    printf("	%s\n", yb->id);
  }
  fflush(stdout);
}





/*
 * Name: ext_yahoo_got_identities
 * 	Called when the contact list is got from the server
 * Params:
 * 	id   - the id that identifies the server connection
 * 	ids  - the identity list
 */
void ext_yahoo_got_identities(int id, YList * ids)
{
  YList* n;
  printf("Identities downloaded\n");
  for (n = ids; n != NULL; n = n->next)
  {
    char * id = (char *) n->data;
    printf("	%s\n", id);
  }
  fflush(stdout);
}





/*
 * Name: ext_yahoo_got_cookies
 * 	Called when the cookie list is got from the server
 * Params:
 * 	id   - the id that identifies the server connection
 */
void ext_yahoo_got_cookies(int id)
{
  printf("Mmmm, cookies\n");
  fflush(stdout);
}




/*
 * Name: ext_yahoo_status_changed
 * 	Called when remote user's status changes.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the handle of the remote user
 * 	stat - status code (enum yahoo_status)
 * 	msg  - the message if stat == YAHOO_STATUS_CUSTOM
 * 	away - whether the contact is away or not (YAHOO_STATUS_CUSTOM)
 * 	       for YAHOO_STATUS_IDLE, this is the number of seconds he is idle
 */
void ext_yahoo_status_changed(int id, char *who, int stat, char *msg, int away)
{
  eb_local_account * la = get_local(id);
  eb_account * acc;
  int do_login = 0;
  int do_logout = 0;
  if (la == NULL)
  {
    printf("Martian Yahoo connection\n");
    fflush(stdout);
    return;
  }
  acc = eb_get_account(la->handle, "Yahoo", who);
  if (acc == NULL)
  {
    acc = eb_add_account_plus("Unknown", who, la, who);
  }
  
  if (acc->status == EB_ACCOUNT_OFFLINE)
  {
    do_login = 1;
  }
  free(acc->status_string);
  acc->status = EB_ACCOUNT_AWAY;
  switch (stat)
  {
    case YAHOO_STATUS_AVAILABLE:
    {
      acc->status = EB_ACCOUNT_ONLINE;
      acc->status_string = strdup("");
      break;
    }
    case YAHOO_STATUS_BRB:
    {
      acc->status_string = strdup("BRB");
      break;
    }
    case YAHOO_STATUS_BUSY:
    {
      acc->status_string = strdup("Busy");
      break;
    }
    case YAHOO_STATUS_NOTATHOME:
    {
      acc->status_string = strdup("Not at Home");
      break;
    }
    case YAHOO_STATUS_NOTATDESK:
    {
      acc->status_string = strdup("Not at Desk");
      break;
    }
    case YAHOO_STATUS_NOTINOFFICE:
    {
      acc->status_string = strdup("Not in Office");
      break;
    }
    case YAHOO_STATUS_ONPHONE:
    {
      acc->status_string = strdup("Phone");
      break;
    }
    case YAHOO_STATUS_ONVACATION:
    {
      acc->status_string = strdup("On Vacation");
      break;
    }
    case YAHOO_STATUS_OUTTOLUNCH:
    {
      acc->status_string = strdup("Lunch");
      break;
    }
    case YAHOO_STATUS_STEPPEDOUT:
    {
      acc->status_string = strdup("Stepped Out");
      break;
    }
    case YAHOO_STATUS_IDLE:
    {
      acc->status_string = strdup("Idle");
      break;
    }
    case YAHOO_STATUS_OFFLINE:
    {
      do_login = 0;
      acc->status = EB_ACCOUNT_OFFLINE;
      acc->status_string = strdup("Offline");
      do_logout = 1;
      break;
    }
    default:
    {
      acc->status = away ? EB_ACCOUNT_AWAY : EB_ACCOUNT_ONLINE;
      acc->status_string = msg ? (strdup(msg)) : (strdup(""));
    }
  }

  if (do_login)
  {
    eb_buddy_login(acc);
  }
  eb_buddy_update_status(acc);
  if (do_logout)
  {
    eb_buddy_logout(acc);
  }
}




/*
 * Name: ext_yahoo_got_im
 * 	Called when remote user sends you a message.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the handle of the remote user
 * 	msg  - the message - NULL if stat == 2
 * 	tm   - timestamp of message if offline
 * 	stat - message status - 0
 * 				1
 * 				2 == error sending message
 * 				5
 * 	utf8 - whether the message is encoded as utf8 or not
 */
void ext_yahoo_got_im(int id, char *who, char *msg, long tm, int stat, int utf8)
{
  eb_local_account * la = get_local(id);
  eb_account * acc;
  if (la == NULL)
  {
    printf("Martian Yahoo connection\n");
    fflush(stdout);
    return;
  }
  acc = eb_get_account(la->handle, "Yahoo", who);
  if (acc == NULL)
  {
    acc = eb_add_account_plus("Unknown", who, la, who);
    acc->status = EB_ACCOUNT_AWAY;
    free(acc->status_string);
    acc->status_string = strdup("???");
  }
  
  if (stat == 2)
  {
    eb_notify_3rdperson(acc->contact, "<font color=red><i><b>Error</b>: Failed to send your last message</i></font>");
  }
  else
  {
    char * m;
    
    if(!strncmp(msg, "EB+]#[", 6)) // inlined transfer
    {
      printf("Yahoo handling inlined: %s\n", msg);
      if(eb_handle_inlined_data(acc, msg, strlen(msg),
      (eb_inlined_data_state **)&(acc->protocol_data))==0)
      { return; }
    }
    
    
    m=(utf8) ? (strdup(msg)) : (y_str_to_utf8(msg));
    eb_incoming_message(acc, m);
    free(m);
  }
}




/*
 * Name: ext_yahoo_got_conf_invite
 * 	Called when remote user sends you a conference invitation.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the user inviting you
 * 	room - the room to join
 * 	msg  - the message
 *      members - the initial members of the conference (null terminated list)
 */
void ext_yahoo_got_conf_invite(int id, char *who, char *room, char *msg, YList *members)
{
}




/*
 * Name: ext_yahoo_conf_userdecline
 * 	Called when someone declines to join the conference.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the user who has declined
 * 	room - the room
 * 	msg  - the declining message
 */
void ext_yahoo_conf_userdecline(int id, char *who, char *room, char *msg)
{
}




/*
 * Name: ext_yahoo_conf_userjoin
 * 	Called when someone joins the conference.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the user who has joined
 * 	room - the room joined
 */
void ext_yahoo_conf_userjoin(int id, char *who, char *room)
{
}




/*
 * Name: ext_yahoo_conf_userleave
 * 	Called when someone leaves the conference.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the user who has left
 * 	room - the room left
 */
void ext_yahoo_conf_userleave(int id, char *who, char *room)
{
}


/*
 * Name: ext_yahoo_chat_cat_xml
 * 	Called when joining the chatroom.
 * Params:
 * 	id      - the id that identifies the server connection
 * 	room    - the room joined, used in all other chat calls, freed by library after call
 * 	topic   - the topic of the room, freed by library after call
 *      members - the initial members of the chatroom (null terminated YList of yahoo_chat_member's) Must be freed by the client
 */
void ext_yahoo_chat_cat_xml(int id, char *xml)
{
}

/*
 * Name: ext_yahoo_chat_join
 * 	Called when joining the chatroom.
 * Params:
 * 	id      - the id that identifies the server connection
 * 	room    - the room joined, used in all other chat calls, freed by library after call
 * 	topic   - the topic of the room, freed by library after call
 *  members - the initial members of the chatroom (null terminated YList of yahoo_chat_member's) Must be freed by the client
 * 	fd      - the socket where the connection is coming from (for tracking)
 */
void ext_yahoo_chat_join(int id, char *room, char *topic, YList *members, int fd)
{
}

/*
 * Name: ext_yahoo_chat_userjoin
 * 	Called when someone joins the chatroom.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	room - the room joined
 * 	who  - the user who has joined, Must be freed by the client
 */
void ext_yahoo_chat_userjoin(int id, char *room, struct yahoo_chat_member *who)
{
}




/*
 * Name: ext_yahoo_chat_userleave
 * 	Called when someone leaves the chatroom.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	room - the room left
 * 	who  - the user who has left (Just the User ID)
 */
void ext_yahoo_chat_userleave(int id, char *room, char *who)
{
}




/*
 * Name: ext_yahoo_chat_message
 * 	Called when someone messages in the chatroom.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	room - the room
 * 	who  - the user who messaged (Just the user id)
 * 	msg  - the message
 * 	msgtype  - 1 = Normal message
 * 		   2 = /me type message
 * 	utf8 - whether the message is utf8 encoded or not
 */
void ext_yahoo_chat_message(int id, char *who, char *room, char *msg, int msgtype, int utf8)
{}

/*
 *
 * Name: ext_yahoo_chat_yahoologout
 *	called when yahoo disconnects your chat session
 *	Note this is called whenver a disconnect happens, client or server
 *	requested. Care should be taken to make sure you know the origin 
 *	of the disconnect request before doing anything here (auto-join's etc)
 * Params:
 *	id   - the id that identifies this connection
 * Returns:
 *	nothing.
 */
void ext_yahoo_chat_yahoologout(int id)
{}

/*
 *
 * Name: ext_yahoo_chat_yahooerror
 *	called when yahoo sends back an error to you
 *	Note this is called whenver chat message is sent into a room
 *	in error (fd not connected, room doesn't exists etc)
 *	Care should be taken to make sure you know the origin 
 *	of the error before doing anything about it.
 * Params:
 *	id   - the id that identifies this connection
 * Returns:
 *	nothing.
 */
void ext_yahoo_chat_yahooerror(int id)
{}

/*
 * Name: ext_yahoo_conf_message
 * 	Called when someone messages in the conference.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the user who messaged
 * 	room - the room
 * 	msg  - the message
 * 	utf8 - whether the message is utf8 encoded or not
 */
void ext_yahoo_conf_message(int id, char *who, char *room, char *msg, int utf8)
{}




/*
 * Name: ext_yahoo_got_file
 * 	Called when someone sends you a file
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the user who sent the file
 * 	url  - the file url
 * 	expires  - the expiry date of the file on the server (timestamp)
 * 	msg  - the message
 * 	fname- the file name if direct transfer
 * 	fsize- the file size if direct transfer
 */
void ext_yahoo_got_file(int id, char *who, char *url, long expires, char *msg, char *fname, unsigned long fesize)
{}




/*
 * Name: ext_yahoo_contact_added
 * 	Called when a contact is added to your list
 * Params:
 * 	id   - the id that identifies the server connection
 * 	myid - the identity he was added to
 * 	who  - who was added
 * 	msg  - any message sent
 */
void ext_yahoo_contact_added(int id, char *myid, char *who, char *msg)
{}




/*
 * Name: ext_yahoo_rejected
 * 	Called when a contact rejects your add
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - who rejected you
 * 	msg  - any message sent
 */
void ext_yahoo_rejected(int id, char *who, char *msg)
{
  char * buf = (char *)malloc(strlen(who) + 128);
  sprintf(buf, "The user \"%s\" has refused to allow you to see their status. They will therefore appear permanently offline.", who);
  eb_show_error(buf, "Buddy list rejection");
  free(buf);
}



/*
 * Name: ext_yahoo_typing_notify
 * 	Called when remote user starts or stops typing.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the handle of the remote user
 * 	stat - 1 if typing, 0 if stopped typing
 */
void ext_yahoo_typing_notify(int id, char *who, int stat)
{
  printf("%s has %sed typing\n", who, stat ? "start" : "stop");
  fflush(stdout);
}




/*
 * Name: ext_yahoo_game_notify
 * 	Called when remote user starts or stops a game.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the handle of the remote user
 * 	stat - 1 if game, 0 if stopped gaming
 */
void ext_yahoo_game_notify(int id, char *who, int stat)
{}




/*
 * Name: ext_yahoo_mail_notify
 * 	Called when you receive mail, or with number of messages
 * Params:
 * 	id   - the id that identifies the server connection
 * 	from - who the mail is from - NULL if only mail count
 * 	subj - the subject of the mail - NULL if only mail count
 * 	cnt  - mail count - 0 if new mail notification
 */
void ext_yahoo_mail_notify(int id, char *from, char *subj, int cnt)
{
  eb_local_account * la = get_local(id);
  if (la == NULL)
  {
    printf("Martian Yahoo connection\n");
    fflush(stdout);
    return;
  }

  if (eb_get_value(la->config_key, "mail_notify")[0] == '0')
  {
    return;
  }
  
  if (cnt == 0)
  {
    char * buf = (char *)malloc(strlen(from) + strlen(subj) + 128);
    sprintf(buf, "You have new mail in your inbox.\nFrom: %s\nSubject: %s", from, subj);
    eb_show_error(buf, "New mail");
    free(buf);
  }
  else
  {
    char buf[512];
    sprintf(buf, "Your inbox contains %d messages", cnt);
    eb_show_error(buf, "Yahoo mail");
  }
}




/*
 * Name: ext_yahoo_system_message
 * 	System message
 * Params:
 * 	id   - the id that identifies the server connection
 * 	msg  - the message
 */
void ext_yahoo_system_message(int id, char *msg)
{
  eb_show_error(msg, "Yahoo system message");
}





/*
 * Name: ext_yahoo_got_webcam_image
 * 	Called when you get a webcam update
 *	An update can either be receiving an image, a part of an image or
 *	just an update with a timestamp
 * Params:
 * 	id         - the id that identifies the server connection
 * 	who        - the user who's webcam we're viewing
 *	image      - image data
 *	image_size - length of the image in bytes
 *	real_size  - actual length of image data
 *	timestamp  - milliseconds since the webcam started
 *
 *	If the real_size is smaller then the image_size then only part of
 *	the image has been read. This function will keep being called till
 *	the total amount of bytes in image_size has been read. The image
 *	received is in JPEG-2000 Code Stream Syntax (ISO/IEC 15444-1).
 *	The size of the image will be either 160x120 or 320x240.
 *	Each webcam image contains a timestamp. This timestamp should be
 *	used to keep the image in sync since some images can take longer
 *	to transport then others. When image_size is 0 we can still receive
 *	a timestamp to stay in sync
 */
void ext_yahoo_got_webcam_image(int id, const char * who,
		const unsigned char *image, unsigned int image_size, unsigned int real_size,
		unsigned int timestamp)
{
  printf("Weeebcam!\n");
  fflush(stdout);
}




/*
 * Name: ext_yahoo_webcam_invite
 * 	Called when you get a webcam invitation
 * Params:
 * 	id   - the id that identifies the server connection
 * 	from - who the invitation is from
 */
void ext_yahoo_webcam_invite(int id, char *from)
{
  printf("%s would like to view your webcam\n", from);
  fflush(stdout);
}




/*
 * Name: ext_yahoo_webcam_invite_reply
 * 	Called when you get a response to a webcam invitation
 * Params:
 * 	id   - the id that identifies the server connection
 * 	from - who the invitation response is from
 *      accept - 0 (decline), 1 (accept)
 */
void ext_yahoo_webcam_invite_reply(int id, char *from, int accept)
{
  printf("%s has %sed your webcam request\n", from, accept ? "accept" : "reject");
  fflush(stdout);
}



/*
 * Name: ext_yahoo_webcam_closed
 * 	Called when the webcam connection closed
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the user who we where connected to
 *      reason - reason why the connection closed
 *	         1 = user stopped broadcasting
 *	         2 = user cancelled viewing permission
 *	         3 = user declines permission
 *	         4 = user does not have webcam online
 */
void ext_yahoo_webcam_closed(int id, char *who, int reason)
{
  printf("Webcam connection to %s closed (code %d)\n", who, reason);
  fflush(stdout);
}


/*
 * Name: ext_yahoo_got_search_result
 *      Called when the search result received from server
 * Params:
 *      id  	 - the id that identifies the server connection
 * 	found	 - total number of results returned in the current result set
 * 	start	 - offset from where the current result set starts
 * 	total	 - total number of results available (start + found <= total)
 * 	contacts - the list of results as a YList of yahoo_found_contact
 * 		   these will be freed after this function returns, so
 * 		   if you need to use the information, make a copy
 */
void ext_yahoo_got_search_result(int id, int found, int start, int total, YList *contacts)
{}




/*
 * Name: ext_yahoo_error
 * 	Called on error.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	err  - the error message
 * 	fatal- whether this error is fatal to the connection or not
 */
void ext_yahoo_error(int id, char *err, int fatal)
{
  char * buf;
  eb_local_account * la = get_local(id);
  if (la == NULL)
  {
    printf("Error on martian Yahoo acct\n");
    fflush(stdout);
    return;
  }

  buf = (char *)malloc(strlen(err) + strlen(la->handle) + 128);
  sprintf(buf, "An error has occurred on the local account \"%s\":\n%s%s",
    la->handle, err, fatal ? "This error is fatal, and the connection will now terminate." : "");
  eb_show_error(buf, "Yahoo error");
  free(buf);
  
  if (fatal)
  {
    la->connected = la->ready = 0;
    free(la->status_string);
    la->status_string = strdup("Offline");
    eb_local_account_update(la);
  }
}




/*
 * Name: ext_yahoo_webcam_viewer
 *	Called when a viewer disconnects/connects/requests to connect
 * Params:
 *	id  - the id that identifies the server connection
 *	who - the viewer
 *	connect - 0=disconnect 1=connect 2=request
 */
void ext_yahoo_webcam_viewer(int id, char *who, int connect)
{}




/*
 * Name: ext_yahoo_webcam_data_request
 *	Called when you get a request for webcam images
 * Params:
 *	id   - the id that identifies the server connection
 *	send - whether to send images or not
 */
void ext_yahoo_webcam_data_request(int id, int send)
{}




/*
 * Name: ext_yahoo_log
 * 	Called to log a message.
 * Params:
 * 	fmt  - the printf formatted message
 * Returns:
 * 	0
 */
int ext_yahoo_log(char *fmt, ...)
{
  int r;
  va_list ap;
  va_start(ap, fmt);
  r=vprintf(fmt, ap);
  va_end(ap);
  fflush(stdout);
  return r;
}






typedef struct _fd_data
{
  int id;
  int fd;
  int tag;
  void * moredata;
} fd_data;

EList * fds = NULL;

static void rd_cb(void * data, int src, int conditions)
{
  fd_data * d = (fd_data *)data;
  yahoo_read_ready(d->id, src, d->moredata);
}

static void wr_cb(void * data, int src, int conditions)
{
  fd_data * d = (fd_data *)data;
  yahoo_write_ready(d->id, src, d->moredata);
}

/*
 * Name: ext_yahoo_add_handler
 * 	Add a listener for the fd.  Must call yahoo_read_ready
 * 	when a YAHOO_INPUT_READ fd is ready and yahoo_write_ready
 * 	when a YAHOO_INPUT_WRITE fd is ready.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	fd   - the fd on which to listen
 * 	cond - the condition on which to call the callback
 * 	data - callback data to pass to yahoo_*_ready
 *
 * Returns: a tag to be used when removing the handler
 */
int ext_yahoo_add_handler(int id, int fd, yahoo_input_condition cond, void *data)
{
  fd_data * d = (fd_data *)malloc(sizeof(fd_data));
  d->id = id;
  d->fd = fd;
  d->moredata = data;  
  if (cond & YAHOO_INPUT_WRITE)
  {
    d->tag = eb_input_add(fd, EB_INPUT_WRITE, wr_cb, d);
  }
  else
  {
    d->tag = eb_input_add(fd, EB_INPUT_READ, rd_cb, d);
  }
  fds = e_list_append(fds, d);
  return d->tag;
}




/*
 * Name: ext_yahoo_remove_handler
 * 	Remove the listener for the fd.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	tag  - the handler tag to remove
 */
void ext_yahoo_remove_handler(int id, int tag)
{
  EList * n;
  for (n = fds; n != NULL; n = n->next)
  {
    fd_data * d = (fd_data *)n->data;
    if (d->id == id && d->tag == tag)
    {
      eb_input_remove(d->tag);
    
      fds = e_list_remove_link(fds, n);
      e_list_free_1(n);
      free(d);
      
      return;
    }
  }
}





/*
 * Name: ext_yahoo_connect
 * 	Connect to a host:port
 * Params:
 * 	host - the host to connect to
 * 	port - the port to connect on
 * Returns:
 * 	a unix file descriptor to the socket
 */
int ext_yahoo_connect(char *host, int port)
{
  return eb_connect_sync_socket(host, port);
}





typedef struct _conn_cb_data
{
  yahoo_connect_callback cb;
  void * data;
  int done;
} conn_cb_data;

static void conn_cb(int fd, void * data)
{
  conn_cb_data * d = (conn_cb_data *)data;
  d->cb(fd, (fd >= 0) ? (0) : (errno), d->data);
  d->done = 1;
}

/*
 * Name: ext_yahoo_connect_async
 * 	Connect to a host:port asynchronously.  This function should return
 * 	immediately returing a tag used to identify the connection handler,
 * 	or a pre-connect error (eg: host name lookup failure).
 * 	Once the connect completes (successfully or unsuccessfully), callback
 * 	should be called (see the signature for yahoo_connect_callback).
 * 	The callback may safely be called before this function returns, but
 * 	it should not be called twice.
 * Params:
 * 	id   - the id that identifies this connection
 * 	host - the host to connect to
 * 	port - the port to connect on
 * 	callback - function to call when connect completes
 * 	callback_data - data to pass to the callback function
 * Returns:
 * 	a unix file descriptor to the socket
 */
int ext_yahoo_connect_async(int id, char *host, int port, 
		yahoo_connect_callback callback, void *callback_data)
{
  int rval;
  conn_cb_data * d = (conn_cb_data *)malloc(sizeof(conn_cb_data));
  d->cb = callback;
  d->data = callback_data;
  d->done = 0;
  rval = eb_connect_socket(host, port, conn_cb, d);
  if (d->done && rval > 0)
    return 0; /* Successful, but callback has already been called */
  return rval;
}

