#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <expat.h>

#include "services.h"
#include "plugin.h"
#include "plugin_api.h"
#include "gui_comms.h"
#include "elist.h"
#include "dialog.h"
#include "prefs.h"
#include "util.h"
#include "message_parse.h"
#include "sha.h"

/* Bible:
   http://docs.jabber.org/general/html/protocol.html
	 MUC Reference:
	 http://www.jabber.org/jeps/jep-0045.html */

#define GET_LAD(x) ((eb_jabber_lad *)(x)->protocol_data)
#define GET_GCD(x) ((eb_jabber_gcd *)(x)->protocol_data)

#ifndef DUMP_JABBER_XML
#define DUMP_JABBER_XML 0
#endif

#if DUMP_JABBER_XML
#define JLOG(...) fprintf(stderr, __VA_ARGS__);
#else
#define JLOG(...)
#endif

typedef struct _eb_jabber_listnode
{
	char *path;
	char *value;
	struct _eb_jabber_listnode *next;
} eb_jabber_listnode;

typedef struct _eb_jabber_stanza
{
	eb_jabber_listnode *list;
	eb_jabber_listnode *tail;
} eb_jabber_stanza;

typedef struct _eb_jabber_lad
{
	int fd;
	int listener;
	
	char *username;
	char *server;

	int depth;
	char *path;
	char *chardata;
	eb_jabber_stanza stanza;

	eb_pref_page *prefs;
	XML_Parser parser;
} eb_jabber_lad;

typedef struct _eb_jabber_gcd
{
	char *name;
	char *topic;
	char *nick;
} eb_jabber_gcd;

int eb_jabber_init();
int eb_jabber_finish();

void eb_jabber_login(eb_local_account *ela);
void eb_jabber_logout(eb_local_account *ela);
char *eb_jabber_send_im(eb_account *to, char *message);
void eb_jabber_setup_local_account(eb_local_account *ela);
void eb_jabber_release_local_account(eb_local_account *ela);
void eb_jabber_set_current_state(eb_local_account *ela, char *state);
void eb_jabber_set_away(eb_local_account *ela, char *short_msg, char *long_msg);
void eb_jabber_unset_away(eb_local_account *ela);
void eb_jabber_join_group_chat(eb_local_account *ela, char *name);
void eb_jabber_leave_group_chat(eb_group_chat *chat);
char *eb_jabber_group_chat_send(eb_group_chat *chat, char *raw);
void eb_jabber_add_user(eb_local_account *ela, eb_account *buddy);
void eb_jabber_del_user(eb_account *buddy);
void eb_jabber_query_subscribe(int response, void *data);
void eb_jabber_query_unsubscribe(int response, void *data);

void eb_jabber_data_received(void *data, int source, int conditions);
void eb_jabber_open_stream(eb_local_account *ela);

void eb_jabber_parser_start(void *data, const char *name, const char **atts);
void eb_jabber_parser_end(void *data, const char *name);
void eb_jabber_parser_data(void *data, const char *s, int len);

char *eb_jabber_stanza_find(eb_jabber_stanza *stanza, const char *path);
char *eb_jabber_stanza_find_index(eb_jabber_stanza *stanza, const char *path, int index);
void eb_jabber_stanza_empty(eb_jabber_stanza *stanza);
void eb_jabber_stanza_append(eb_jabber_stanza *stanza, const char *path, const char *value);
void eb_jabber_stanza_dump(eb_jabber_stanza *stanza);
void eb_jabber_send_stanza(int fd, eb_jabber_stanza *s);
void eb_jabber_stream_opened(eb_local_account *ela, const char *id);
void eb_jabber_received_stanza(eb_local_account *ela, eb_jabber_stanza *stanza);
void eb_jabber_send_presence(eb_local_account *ela, const char *message);

eb_plugin_info jabber_LTX_plugin_info = {"Jabber", "Jabber Service", "(broken)", "2004-1-18", eb_jabber_init, eb_jabber_finish};
static eb_service_callbacks callbacks;
eb_service jabber_service_info = {"Jabber", "Jabber Client", &callbacks, SERVICE_CAN_OFFLINE|SERVICE_CAN_GROUPCHAT|SERVICE_CAN_NAME_CHAT,
	"#adbeef", NULL, NULL, NULL, NULL};
EList *jabber_states=NULL;

/* -------------------
    EB-Lite Callbacks
   ------------------- */

int eb_jabber_init()
{
	JLOG("eb_jabber_init()\n");
	memset(&callbacks, 0, sizeof(callbacks));
	callbacks.setup_local_account=eb_jabber_setup_local_account;
	callbacks.release_local_account=eb_jabber_release_local_account;
	// setup/release buddy accounts?

	callbacks.login=eb_jabber_login;
	callbacks.logout=eb_jabber_logout;
	callbacks.send_im=eb_jabber_send_im;

	callbacks.set_current_state=eb_jabber_set_current_state;
	callbacks.set_away=eb_jabber_set_away;
	callbacks.unset_away=eb_jabber_unset_away;
	callbacks.add_user=eb_jabber_add_user;
	callbacks.del_user=eb_jabber_del_user;
	// block_user, unblock_user?
	
	callbacks.group_chat_send=eb_jabber_group_chat_send;
	callbacks.join_group_chat=eb_jabber_join_group_chat;
	callbacks.leave_group_chat=eb_jabber_leave_group_chat;

	jabber_states=e_list_append(jabber_states, strdup("Online"));
	jabber_states=e_list_append(jabber_states, strdup("Away"));
	jabber_states=e_list_append(jabber_states, strdup("Chat"));
	jabber_states=e_list_append(jabber_states, strdup("DND"));
	jabber_states=e_list_append(jabber_states, strdup("XA"));
	jabber_states=e_list_append(jabber_states, strdup("Offline"));
	jabber_service_info.states=jabber_states;

	eb_load_service(&jabber_service_info);

	return 0;
}

int eb_jabber_finish()
{

	return 0;
}

void eb_jabber_login(eb_local_account *ela)
{
	eb_jabber_lad *lad=GET_LAD(ela);

	ela->connected=1;
	eb_local_account_update(ela);

	//lad->fd=eb_connect_sync_socket(eb_get_value(ela->config_key, "server"), atoi(eb_get_value(ela->config_key, "port")));
	lad->fd=eb_connect_sync_socket(lad->server, atoi(eb_get_value(ela->config_key, "port")));

	fprintf(stderr, "USERNAME: %s, SERVER: %s\n", lad->username, lad->server);

	if (lad->fd==-1)
	{
		eb_show_error("Jabber connection failed.", "Jabber Error");
		ela->connected=0;
		eb_local_account_update(ela);
		return;
	}

	lad->listener=eb_input_add(lad->fd, EB_INPUT_READ|EB_INPUT_EXCEPTION, eb_jabber_data_received, (void *)ela);

	XML_ParserReset(lad->parser, "UTF-8");
	XML_SetUserData(lad->parser, (void *)ela);
	XML_SetElementHandler(lad->parser, eb_jabber_parser_start, eb_jabber_parser_end);
	XML_SetCharacterDataHandler(lad->parser, eb_jabber_parser_data);
	lad->depth=0;
	eb_jabber_stanza_empty(&lad->stanza);

	eb_jabber_open_stream(ela);
}

void eb_jabber_logout(eb_local_account *ela)
{
	EList *n;

	if (ela->connected)
	{
		eb_jabber_lad *lad = GET_LAD(ela);

		if (lad->fd != -1)
		{
			ewrite(lad->fd, "</stream:stream>", 16);
		}
		eb_input_remove(lad->listener);
		close(lad->fd);
		lad->fd=-1;
	}

	ela->ready=0;
	ela->connected=0;
	free(ela->status_string);
	ela->status_string=strdup("Offline");

	for (n = ela->buddies; n!=NULL; n=n->next)
	{
		eb_account *acc=(eb_account *)n->data;

		if (acc->status!=EB_ACCOUNT_OFFLINE) eb_buddy_logout(acc);
	}

	for (n=ela->group_chats; n!=NULL; n=n->next)
	{
		eb_group_chat *chat=(eb_group_chat *)n->data;
		EList *n2;
		EList *queue=NULL;

		for (n2=chat->users; n2!=NULL; n2=n2->next)
		{
			queue=e_list_append(queue, n2->data);
		}

		for (n2=queue; n2!=NULL; n2=n2->next)
		{
			eb_group_chat_left(chat, ((eb_group_chat_user *)n2->data)->handle);
		}

		eb_group_chat_3rdperson(chat, "<font color=red><b><i>Connection to server lost");
	}

	eb_local_account_update(ela);
}

char *eb_jabber_send_im(eb_account *to, char *message)
{
	char *send;
	char *disp;
	eb_jabber_stanza stanza;

	if (!(send = eb_filter_string(EB_PREFILTER_OUT, message, to))) return 0;
	disp = strdup(send);
	if (!(send = eb_filter_string(EB_POSTFILTER_OUT, send, to)))
	{
		free(disp);
		return 0;
	}
	
	memset(&stanza, 0, sizeof(stanza));
	eb_jabber_stanza_append(&stanza, "/message/#", "message");
	eb_jabber_stanza_append(&stanza, "/message/@to", to->handle);
	eb_jabber_stanza_append(&stanza, "/message/@type", "chat");
	eb_jabber_stanza_append(&stanza, "/message/body/#", "body");
	eb_jabber_stanza_append(&stanza, "/message/body/$", send);
	eb_jabber_stanza_append(&stanza, "/message/body/!", "body");
	eb_jabber_stanza_append(&stanza, "/message/!", "message");

	eb_jabber_send_stanza(GET_LAD(to->buddy_of)->fd, &stanza);
	eb_jabber_stanza_empty(&stanza);

	free(send);
	
	return disp;
}

void eb_jabber_join_group_chat(eb_local_account *ela, char *name)
{
	eb_jabber_stanza stanza;

	/*	<presence from='hag66@shakespeare.lit/pda' to='darkcave@macbeth.shakespeare.lit/thirdwitch'>
				<x xmlns='http://jabber.org/protocol/muc'/>
			</presence> */
	
	memset(&stanza, 0, sizeof(stanza));
	eb_jabber_stanza_append(&stanza, "/presence/#", "presence");
	eb_jabber_stanza_append(&stanza, "/presence/@to", name);
	eb_jabber_stanza_append(&stanza, "/presence/@id", "groupchatjoin");
	eb_jabber_stanza_append(&stanza, "/presence/x/#", "x");
	eb_jabber_stanza_append(&stanza, "/presence/x/@xmlns", "http://jabber.org/protocol/muc");
	eb_jabber_stanza_append(&stanza, "/presence/x/!", "x");
	eb_jabber_stanza_append(&stanza, "/presence/!", "presence");

	eb_jabber_send_stanza(GET_LAD(ela)->fd, &stanza);
	eb_jabber_stanza_empty(&stanza);
}

char *eb_jabber_group_chat_send(eb_group_chat *chat, char *raw)
{
	if (chat->account->ready)
	{
		char *name = GET_GCD(chat)->name;
		int fd = GET_LAD(chat->account)->fd;
		eb_jabber_stanza stanza;
	
		/*	<message to='darkcave@macbeth.shakespeare.lit/thirdwitch' type='groupchat'>
					<body>Hello, world!</body>
				</message> */
		
		memset(&stanza, 0, sizeof(stanza));
		eb_jabber_stanza_append(&stanza, "/message/#", "message");
		eb_jabber_stanza_append(&stanza, "/message/@to", name);
		eb_jabber_stanza_append(&stanza, "/message/@type", "groupchat");
		eb_jabber_stanza_append(&stanza, "/message/body/#", "body");
		eb_jabber_stanza_append(&stanza, "/message/body/$", raw);
		eb_jabber_stanza_append(&stanza, "/message/body/!", "body");
		eb_jabber_stanza_append(&stanza, "/message/!", "message");
	
		eb_jabber_send_stanza(fd, &stanza);
		eb_jabber_stanza_empty(&stanza);
	}

	free(raw);

	return 0;
}

void eb_jabber_leave_group_chat(eb_group_chat *chat)
{
	if (chat->account->ready)
	{
		char *name = GET_GCD(chat)->name;
		int fd = GET_LAD(chat->account)->fd;
		eb_jabber_stanza stanza;
	
		/*	<presence to='darkcave@macbeth.shakespeare.lit/thirdwitch' type='unavailable'>
					<comment>reason</comment>
				</presence> */
		
		memset(&stanza, 0, sizeof(stanza));
		eb_jabber_stanza_append(&stanza, "/presence/#", "presence");
		eb_jabber_stanza_append(&stanza, "/presence/@to", name);
		eb_jabber_stanza_append(&stanza, "/presence/@type", "unavailable");
		/*eb_jabber_stanza_append(&stanza, "/presence/comment/#", "comment");
		eb_jabber_stanza_append(&stanza, "/presence/comment/$", "");
		eb_jabber_stanza_append(&stanza, "/presence/comment/!", "comment");*/
		eb_jabber_stanza_append(&stanza, "/presence/!", "presence");
	
		eb_jabber_send_stanza(fd, &stanza);
		eb_jabber_stanza_empty(&stanza);
	}

	eb_destroy_group_chat(chat);
}

void eb_jabber_setup_local_account(eb_local_account *ela)
{
	char *buf1;
	char *buf2;
	char *tmp;

	eb_jabber_lad *lad=malloc(sizeof(eb_jabber_lad));
	lad->fd=-1;
	lad->parser=XML_ParserCreate("UTF-8");
	XML_SetUserData(lad->parser, (void *)ela);
	XML_SetElementHandler(lad->parser, eb_jabber_parser_start, eb_jabber_parser_end);
	XML_SetCharacterDataHandler(lad->parser, eb_jabber_parser_data);
	lad->path = strdup("/");
	lad->depth = 0;
	lad->chardata = 0;
	lad->stanza.list = lad->stanza.tail = 0;
	ela->protocol_data=lad;

  /* Here's what I'm doing:
    username@host/resource
    ^       ^    ^--buf2
    |       \- buf1
    \- ela->handle

    All to get lad->username and lad->server */

	buf1 = index(ela->handle, '@');
	if (buf1)
	{
		lad->username = malloc(buf1 - ela->handle + 1);
		strncpy(lad->username, ela->handle, buf1 - ela->handle);
		lad->username[buf1 - ela->handle] = '\0';
	}
	else
	{
		lad->username = strdup("");
	}
	if (buf1++)
	{
		int len = strlen(ela->handle) - (buf1 - ela->handle) + 1;
		
		lad->server = malloc(len);
		strcpy(lad->server, buf1);
		lad->server[len - 1] = '\0';
	}
	else
	{
		lad->server = strdup("");
	}

	buf1=(char *)malloc(strlen(ela->handle)+strlen(ela->service_name)+2);
	buf2=(char *)malloc(strlen(ela->handle)+strlen(ela->service_name)+4);
	sprintf(buf1, "%s_%s", ela->handle, ela->service_name);
	sprintf(buf2, "%s (%s)", ela->handle, ela->service_name);
	lad->prefs=eb_make_pref_page(account_prefs, buf1, buf2);
	free(buf1);
	free(buf2);

	//eb_add_component(lad->prefs, EB_PREF_STRING, "username", "Username:", ela->config_key, "username", NULL, NULL);
	eb_add_component(lad->prefs, EB_PREF_PASSWORD, "password", "Password:", ela->config_key, "password", NULL, NULL);
	//eb_add_component(lad->prefs, EB_PREF_STRING, "server", "Server:", ela->config_key, "server", NULL, NULL);
	eb_add_component(lad->prefs, EB_PREF_STRING, "port", "Port:", ela->config_key, "port", NULL, NULL);
	eb_add_component(lad->prefs, EB_PREF_STRING, "resource", "Resource:", ela->config_key, "resource", NULL, NULL);
}

void eb_jabber_release_local_account(eb_local_account *ela)
{
	eb_jabber_lad *lad=GET_LAD(ela);
	free(lad->username);
	free(lad->server);
	XML_ParserFree(GET_LAD(ela)->parser);
	eb_destroy_pref_page(lad->prefs);
	eb_jabber_stanza_empty(&lad->stanza);
	free(lad);
}

void eb_jabber_set_current_state(eb_local_account *ela, char *state)
{
	if (!strcmp(state, "Offline"))
	{
		if (ela->connected) eb_jabber_logout(ela);
	}
	else
	{
		if (!ela->connected) eb_jabber_login(ela);
		if (ela->connected)
		{
			free(ela->status_string);
			ela->status_string=strdup(state);
			if (ela->ready) eb_jabber_send_presence(ela, 0);
			eb_local_account_update(ela);
		}
	}
}

void eb_jabber_set_away(eb_local_account *ela, char *short_msg, char *long_msg)
{
	eb_jabber_stanza s;
	memset(&s, 0, sizeof(s));
	
	eb_jabber_stanza_append(&s, "/presence/#", "presence");
	eb_jabber_stanza_append(&s, "/presence/show/#", "show");
	eb_jabber_stanza_append(&s, "/presence/show/$", "away");
	eb_jabber_stanza_append(&s, "/presence/show/!", "show");
	eb_jabber_stanza_append(&s, "/presence/status/#", "status");
	eb_jabber_stanza_append(&s, "/presence/status/$", long_msg);
	eb_jabber_stanza_append(&s, "/presence/status/!", "status");
	eb_jabber_stanza_append(&s, "/presence/!", "presence");

	eb_jabber_send_stanza(GET_LAD(ela)->fd, &s);

	eb_jabber_stanza_empty(&s);
}

void eb_jabber_unset_away(eb_local_account *ela)
{
	ewrite(GET_LAD(ela)->fd, "<presence/>", 11);
}


void eb_jabber_add_user(eb_local_account *ela, eb_account *buddy)
{
	eb_jabber_stanza s;

	if (!ela->ready)
	{
		eb_show_error("Can't add an account to an offline account.", "Jabber Error");
		free(buddy->handle);
		free(buddy->contact);
		free(buddy->status_string);
		free(buddy);
		return;
	}

	memset(&s, 0, sizeof(s));

	eb_jabber_stanza_append(&s, "/presence/#", "presence");
	eb_jabber_stanza_append(&s, "/presence/@to", buddy->handle);
	eb_jabber_stanza_append(&s, "/presence/@type", "subscribe");
	eb_jabber_stanza_append(&s, "/presence/!", "presence");

	eb_jabber_send_stanza(GET_LAD(ela)->fd, &s);
	eb_jabber_stanza_empty(&s);

	eb_add_account(buddy->contact, ela, buddy);
}

void eb_jabber_del_user(eb_account *buddy)
{
	eb_jabber_stanza s;

	if (!buddy->buddy_of->ready)
	{
		eb_show_error("Can't delete an account from an offline account.", "Jabber Error");
		return;
	}
	
	memset(&s, 0, sizeof(s));

	eb_jabber_stanza_append(&s, "/presence/#", "presence");
	eb_jabber_stanza_append(&s, "/presence/@to", buddy->handle);
	eb_jabber_stanza_append(&s, "/presence/@type", "unsubscribe");
	eb_jabber_stanza_append(&s, "/presence/!", "presence");

	eb_jabber_send_stanza(GET_LAD(buddy->buddy_of)->fd, &s);
	eb_jabber_stanza_empty(&s);
	
	eb_del_account(buddy);
}

void eb_jabber_data_received(void *data, int source, int conditions)
{
	eb_local_account *ela = (eb_local_account *)data;
	eb_jabber_lad *lad=GET_LAD(ela);;
	if (conditions & EB_INPUT_READ)
	{
		char buffer[4096];
		int len = read(lad->fd, buffer, sizeof(buffer));
		if (len > 0)
		{
			buffer[len] = '\0';
			JLOG(">> %s\n", buffer);
			if (XML_Parse(lad->parser, buffer, len, 0) == XML_STATUS_ERROR)
				JLOG("Jabber: error parsing XML\n");
		}
		else if (len == 0)
		{
			JLOG("Jabber: read of size 0\n");
			eb_jabber_logout(ela);
		}
		else
		{
			JLOG("Jabber: invalid read\n");
			eb_jabber_logout(ela);
		}
	}
	else if (conditions & EB_INPUT_EXCEPTION)
	{
		JLOG("Jabber: error on file descriptor %d\n", source);
	}
}

/* ----------------------
    XML Parser Callbacks
   ---------------------- */

void eb_jabber_parser_enddata(eb_jabber_lad *lad)
{
	unsigned int pathlen = strlen(lad->path);
	char *path = (char *)malloc(pathlen + 2);

	strcpy(path, lad->path);
	path[pathlen] = '$';
	path[pathlen+1] = '\0';

	eb_jabber_stanza_append(&lad->stanza, path, lad->chardata);

	free(lad->chardata);
	lad->chardata = 0;
}

void eb_jabber_parser_start(void *data, const char *name, const char **atts)
{
	eb_jabber_lad *lad=GET_LAD((eb_local_account *)data);
	unsigned int pathlen = strlen(lad->path);
	unsigned int namelen = strlen(name);
	const char **att;

	if (lad->chardata) eb_jabber_parser_enddata(lad);
	
	if (lad->depth == 0 && !strcmp(name, "stream:stream"))
	{
		const char *id = 0;
		const char **att;
		for (att = atts; *att; att++)
		{
			if (!strcmp(att[0], "id"))
			{
				id = att[1];
			}
		}
		eb_jabber_stream_opened((eb_local_account *)data, id);
	}
	else
	{
		char *path = (char *)malloc(pathlen + namelen + 3);

		lad->path = realloc(lad->path, pathlen + namelen + 2);
		strcpy(lad->path + pathlen, name);
		lad->path[pathlen + namelen] = '/';
		lad->path[pathlen + namelen + 1]= '\0';
		pathlen += namelen + 1;

		strcpy(path, lad->path);
		path[pathlen] = '#';
		path[pathlen+1] = '\0';
	
		eb_jabber_stanza_append(&lad->stanza, path, name);
	
		free(path);
	
		for (att = atts; *att; att += 2)
		{
			size_t attlen = strlen(att[0]);
	
			path = (char *)malloc(pathlen + 2 + attlen);
			strcpy(path, lad->path);
			path[pathlen] = '@';
			strcpy(path+pathlen+1, att[0]);
			path[pathlen+attlen+1] = '\0';
	
			eb_jabber_stanza_append(&lad->stanza, path, att[1]);
	
			free(path);
		}
	}

	lad->depth++;
}

void eb_jabber_parser_end(void *data, const char *name)
{
	eb_jabber_lad *lad=GET_LAD((eb_local_account *)data);
	char *pos;
	char *tmp;

	if (lad->chardata) eb_jabber_parser_enddata(lad);
	lad->depth--;

	for (pos = lad->path + strlen(lad->path) - 2; pos > lad->path; pos--)
	{
		if (*pos == '/')
			break;
		else
			*pos = '\0';
	}

	tmp = (char *)malloc(strlen(lad->path)+2);
	strcpy(tmp, lad->path);
	strcat(tmp, "!");

	eb_jabber_stanza_append(&lad->stanza, tmp, name);

	if (lad->depth == 1)
	{
		eb_jabber_received_stanza(data, &lad->stanza);
		eb_jabber_stanza_empty(&lad->stanza);
	}
}

void eb_jabber_parser_data(void *data, const char *s, int len)
{
	eb_jabber_lad *lad=GET_LAD((eb_local_account *)data);
	size_t curlen = (lad->chardata)?strlen(lad->chardata):0;
	lad->chardata = (char *)realloc(lad->chardata, len + curlen + 1);
	memcpy(lad->chardata + curlen, s, len);
	lad->chardata[curlen+len] = '\0';
}

/* ---------
    Utility
   --------- */

void eb_jabber_write_parameter(int fd, const char *name, const char *value)
{
	if (name)
	{
		ewrite(fd, " ", 1);
		ewrite(fd, (char *)name, strlen(name));
		ewrite(fd, "='", 2);
		ewrite(fd, (char *)value, strlen(value));
		ewrite(fd, "'", 1);
	}
}

void eb_jabber_write_simple_tag(int fd, const char *name, const char *value)
{
	if (name)
	{
		ewrite(fd, "<", 1);
		ewrite(fd, (char *)name, strlen(name));
		ewrite(fd, ">'", 2);
		ewrite(fd, (char *)value, strlen(value));
		ewrite(fd, "</", 2);
		ewrite(fd, (char *)name, strlen(name));
		ewrite(fd, ">", 1);
	}
}

int safe_strcmp(const char *a, const char *b)
{
	if (a && b) return strcmp(a, b);
	else return a || b;
}

char *eb_jabber_xml_escape(const char *in)
{
	char *tmp = (char *)malloc(6*strlen(in)+1); // Worst case = '"' x strlen(in)
	char *out = tmp;

	while (*in)
	{
		switch (*in)
		{
		case '<':
			strcpy(out, "&lt;");
			out += 4;
			break;
		case '>':
			strcpy(out, "&gt;");
			out += 4;
			break;
		case '&':
			strcpy(out, "&amp;");
			out += 5;
			break;
		case '\'':
			strcpy(out, "&apos;");
			out += 6;
			break;
		case '"':
			strcpy(out, "&quot;");
			out += 6;
			break;
		default:
			*out++ = *in;
			break;
		}
		in++;
	}

	*out = '\0';

	return tmp;
}

eb_group_chat *eb_jabber_get_chat(eb_local_account *acc, char *name)
{
	EList *n;

	for (n=group_chats; n!=0; n=n->next)
	{
		eb_group_chat *chat=(eb_group_chat *)n->data;
		if (chat->account!=acc) continue;
		if (!strcasecmp(GET_GCD(chat)->name, name)) return chat;
	}

	return 0;
}

/* -----------------------
    Jabber event handling
   ----------------------- */

void eb_jabber_open_stream(eb_local_account *ela)
{
	eb_jabber_lad *lad=GET_LAD(ela);
	char *prefix="<?xml version='1.0' encoding='UTF-8' ?><stream:stream to='";
	char *suffix="' xmlns='jabber:client' xmlns:stream='http://etherx.jabber.org/streams'>";
	char *server=eb_jabber_xml_escape(lad->server);
	//char *server=eb_jabber_xml_escape(eb_get_value(ela->config_key, "server"));
	char *buff=(char *)malloc(strlen(prefix)+strlen(suffix)+strlen(server)+1);

	sprintf(buff, "%s%s%s", prefix, server, suffix); // Quote lad->server!

	ewrite(lad->fd, buff, strlen(buff));
	JLOG(">>> %s\n", buff);

	free(ela->status_string);
	ela->status_string=strdup("Online");

	free(buff);
	free(server);
}

/* The caller must free the return value */
char *eb_jabber_password_digest(const char *id, const char *password)
{
	int len = strlen(id) + strlen(password);
	char *plain = malloc(len + 1);
	SHA_CTX ctx;
	unsigned char hash[20];
	char hex[41];
	unsigned char *pos;
	char *outpos = hex;

	strcpy(plain, id);
	strcat(plain, password);

	shaInit(&ctx);
	shaUpdate(&ctx, plain, len);
	shaFinal(&ctx, hash);

	free(plain);

	for (pos = hash; pos < hash + 20; pos++)
	{
		const char *alphabet = "0123456789abcdef";
		*outpos++ = alphabet[(*pos & 0xF0) >> 4];
		*outpos++ = alphabet[*pos & 0x0F];
	}
	*outpos = '\0';

	return strdup(hex);
}

void eb_jabber_stream_opened(eb_local_account *ela, const char *id)
{
	eb_jabber_stanza s;
	char *digest = eb_jabber_password_digest(id, eb_get_value(ela->config_key, "password"));

	JLOG("Stream opened with id: %s\n", id);
	memset(&s, 0, sizeof(s));
	eb_jabber_stanza_append(&s, "/#", "iq");
	eb_jabber_stanza_append(&s, "/iq/@type", "set");
	eb_jabber_stanza_append(&s, "/iq/@id", "auth");
	eb_jabber_stanza_append(&s, "/iq/query/#", "query");
	eb_jabber_stanza_append(&s, "/iq/query/@xmlns", "jabber:iq:auth");
	eb_jabber_stanza_append(&s, "/iq/query/username/#", "username");
	//eb_jabber_stanza_append(&s, "/iq/query/username/$", eb_get_value(ela->config_key, "username"));
	eb_jabber_stanza_append(&s, "/iq/query/username/$", GET_LAD(ela)->username);
	eb_jabber_stanza_append(&s, "/iq/query/username/!", "username");
	/*eb_jabber_stanza_append(&s, "/iq/query/password/#", "password");
	eb_jabber_stanza_append(&s, "/iq/query/password/$", eb_get_value(ela->config_key, "password"));
	eb_jabber_stanza_append(&s, "/iq/query/password/!", "password");*/
	eb_jabber_stanza_append(&s, "/iq/query/digest/#", "digest");
	eb_jabber_stanza_append(&s, "/iq/query/digest/$", digest);
	eb_jabber_stanza_append(&s, "/iq/query/digest/!", "digest");
	eb_jabber_stanza_append(&s, "/iq/query/resource/#", "resource");
	eb_jabber_stanza_append(&s, "/iq/query/resource/$", eb_get_value(ela->config_key, "resource"));
	eb_jabber_stanza_append(&s, "/iq/query/resource/!", "resource");
	eb_jabber_stanza_append(&s, "/iq/query/!", "query");
	eb_jabber_stanza_append(&s, "/!", "iq");
	eb_jabber_send_stanza(GET_LAD(ela)->fd, &s);
	eb_jabber_stanza_empty(&s);

	free(digest);
}

void eb_jabber_received_stanza(eb_local_account *ela, eb_jabber_stanza *stanza)
{
	//eb_jabber_lad *lad=GET_LAD(ela);

	if (eb_jabber_stanza_find(stanza, "/iq/#"))
	{
		char *type = eb_jabber_stanza_find(stanza, "/iq/@type");
		if (type)
		{
			char *query_xmlns = eb_jabber_stanza_find(stanza, "/iq/query/@xmlns");

			if (!strcmp(type, "error"))
			{
				eb_show_error(eb_jabber_stanza_find(stanza, "/iq/error/$"), "Jabber Error");
				return;
			}
			if (query_xmlns && !strcmp(query_xmlns, "jabber:iq:roster"))
			{
				eb_jabber_listnode *node = stanza->list;
				char *id = eb_jabber_stanza_find(stanza, "/iq/@id");

				while (node)
				{
					char *jid = 0;
					char *name = 0;
					char *subscription = 0;
					char *group = 0;
					char *ask = 0;

					while (node && strcmp(node->path, "/iq/query/item/#")) node = node->next;

					while (node && strcmp(node->path, "/iq/query/item/!"))
					{
						if (!strcmp(node->path, "/iq/query/item/@jid")) jid = node->value;
						if (!strcmp(node->path, "/iq/query/item/@name")) name = node->value;
						if (!strcmp(node->path, "/iq/query/item/@subscription")) subscription = node->value;
						if (!strcmp(node->path, "/iq/query/item/group/$")) group = node->value;
						if (!strcmp(node->path, "/iq/query/item/@ask")) ask = node->value;

						node = node->next;
					}

					if (jid && subscription && (!strcmp(subscription, "to") || !strcmp(subscription, "both")))
					{
						eb_account *acc = eb_get_account(ela->handle, "Jabber", jid);

						if (!acc) acc=eb_add_account_plus(group?group:"Unknown", name?name:jid, ela, jid);
					}

					if (jid && subscription && (!strcmp(subscription, "remove") || !strcmp(subscription, "none")) && (!ask || strcmp(ask, "subscribe")))
					{
						eb_account *acc = eb_get_account(ela->handle, "Jabber", jid);
						if (acc) eb_del_account(acc);
					}

					if (node) node = node->next;
				}
			}

			if (!strcmp(type, "result") && !safe_strcmp(eb_jabber_stanza_find(stanza, "/iq/@id"), "auth"))
			{
				ela->ready=1;
				eb_local_account_update(ela);
				eb_jabber_send_presence(ela, 0);

				ewrite(GET_LAD(ela)->fd, "<iq type='get' id='load'><query xmlns='jabber:iq:roster'/></iq>", 63);
				JLOG("<< <iq type='get' id='load'><query xmlns='jabber:iq:roster'/></iq>\n");
			}
		}
	}
	else if (eb_jabber_stanza_find(stanza, "/message/#"))
	{
		char *from = strdup(eb_jabber_stanza_find(stanza, "/message/@from"));

		if (!safe_strcmp(eb_jabber_stanza_find(stanza, "/message/@type"), "groupchat"))
		{
			char *room = from;
			char *nick=index(from, '/');
			if (nick)
			{
				*nick='\0';
				nick++;
			}

			eb_group_chat *chat=eb_jabber_get_chat(ela, room);
			char *message=eb_jabber_stanza_find(stanza, "/message/body/$");

			if (message && chat && nick) eb_group_chat_message(chat, nick, message);
			else if (message && chat) eb_group_chat_3rdperson(chat, message);
		}
		else
		{
			char *slash = index(from, '/');
			if (slash) *slash = '\0';
			eb_account *acc = eb_get_account(ela->handle, "Jabber", from);

			if (!acc)
			{
				acc=eb_add_account_plus("Unknown", from, ela, from);
				eb_buddy_login(acc);
			}
			else if (acc->status==EB_ACCOUNT_OFFLINE)
			{
				free(acc->status_string);
				acc->status_string=strdup("");
				eb_buddy_login(acc);
				eb_buddy_update_status(acc);
			}
	
			eb_incoming_message(acc, eb_jabber_stanza_find(stanza, "/message/body/$"));
		}
		
		free(from);
	}
	else if (eb_jabber_stanza_find(stanza, "/presence/#"))
	{
		char *from = strdup(eb_jabber_stanza_find(stanza, "/presence/@from"));
		char *slash = index(from, '/');
		char *id = eb_jabber_stanza_find(stanza, "/presence/@id");
		char *type = eb_jabber_stanza_find(stanza, "/presence/@type");
		int online = !type || !safe_strcmp(type, "available") || !safe_strcmp(type, "chat") || !safe_strcmp(type, "xa") || !safe_strcmp(type, "dnd") || !safe_strcmp(type, "away");

		if (slash) *slash = '\0';

		if (type && !strcmp(type, "error"))
		{
			char *error = eb_jabber_stanza_find(stanza, "/presence/error/$");
			eb_show_error(error?error:"Unknown error", "Jabber Error");
		}
		else if (type && !strcmp(type, "subscribe"))
		{
			char *text = "%s wants to add you to his or her roster.  Do you accept this request?";
			char *message = malloc(strlen(from) + strlen(text) + 1);
			void **data = (void **)malloc(sizeof(void *) * 2);
			data[0] = (void *)ela;
			data[1] = (void *)strdup(from);
			sprintf(message, text, from);
			JLOG("Subscription request!\n");
			//eb_show_yesno_dialog(message, "Authorization Request", 1, eb_jabber_query_subscribe, ((void *)data));
			eb_jabber_query_subscribe(1, data);
			free(message);
		}
		else if (type && !strcmp(type, "unsubscribe"))
		{
			char *text = "%s wants to remove you from his or her roster.  Do you accept this request?";
			char *message = malloc(strlen(from) + strlen(text) + 1);
			void **data = (void **)malloc(sizeof(void *) * 2);
			data[0] = (void *)ela;
			data[1] = (void *)strdup(from);
			sprintf(message, text, from);
			JLOG("Unsubscription request!\n");
			//eb_show_yesno_dialog(message, "Authorization Request", 1, eb_jabber_query_unsubscribe, ((void *)data));
			eb_jabber_query_unsubscribe(1, data);
			free(message);
		}
		else if (id && !strcmp(id, "groupchatjoin") && online)
		{
			eb_group_chat *chat;
			chat = eb_jabber_get_chat(ela, from);

			if (!chat)
			{
				chat=eb_create_group_chat(ela);
				chat->protocol_data=(eb_jabber_gcd *)malloc(sizeof(eb_jabber_gcd));
				chat->title=strdup(from);
				GET_GCD(chat)->name=strdup(from);
				GET_GCD(chat)->topic=strdup("");
				GET_GCD(chat)->nick=strdup(slash?slash+1:"");

				if (chat) eb_group_chat_joined(chat, slash+1);
			}
			else
			{
				eb_group_chat_joined(chat, slash+1);
			}
		}
		else
		{
			eb_group_chat *chat = eb_jabber_get_chat(ela, from);

			if (chat)
			{
				int me=(slash && !strcmp(slash+1, GET_GCD(chat)->nick));
				if (online)
					eb_group_chat_joined(chat, slash+1);
				else
				{
					eb_group_chat_left(chat, slash+1);
					if (me) eb_destroy_group_chat(chat);
				}
			}
			else
			{
				eb_account *acc = eb_get_account(ela->handle, ela->service_name, from);

				if (acc && type && !strcmp(type, "unsubscribed"))
				{
					eb_jabber_del_user(acc);
					return;
				}
				
				if (!acc) acc=eb_add_account_plus("Unknown", from, ela, from);
	
				if (online) // Available
				{
					char *show = eb_jabber_stanza_find(stanza, "/presence/show/$");
					eb_buddy_login(acc);
					free(acc->status_string);
					acc->status_string=strdup(show?show:"Online");
					eb_buddy_update_status(acc);
				}
				else
				{
					eb_buddy_logout(acc);
				}
			}
		}
	}
}

void eb_jabber_send_presence(eb_local_account *ela, const char *message)
{
	eb_jabber_stanza s;
	char *newState=0;
	memset(&s, 0, sizeof(s));

	if (!strcmp(ela->status_string, "Away")) newState="away";
	if (!strcmp(ela->status_string, "XA")) newState="xa";
	if (!strcmp(ela->status_string, "DND")) newState="dnd";
	if (!strcmp(ela->status_string, "Chat")) newState="chat";

	if (!newState && !message)
	{
		JLOG("<< <presence/>\n");
		ewrite(GET_LAD(ela)->fd, "<presence/>", 11);
	}
	else
	{
		eb_jabber_stanza_append(&s, "/presence/#", "presence");
		if (newState)
		{
			eb_jabber_stanza_append(&s, "/presence/show/#", "show");
			eb_jabber_stanza_append(&s, "/presence/show/$", newState);
			eb_jabber_stanza_append(&s, "/presence/show/!", "show");
		}
		if (message)
		{
			eb_jabber_stanza_append(&s, "/presence/status/#", "status");
			eb_jabber_stanza_append(&s, "/presence/status/$", message);
			eb_jabber_stanza_append(&s, "/presence/status/!", "status");
		}
		eb_jabber_stanza_append(&s, "/presence/!", "presence");
	
		eb_jabber_send_stanza(GET_LAD(ela)->fd, &s);
	
		eb_jabber_stanza_empty(&s);
	}
}

void eb_jabber_query_unsubscribe(int response, void *indata)
{
	void **data = (void **)indata;
	eb_local_account *ela = (eb_local_account *)data[0];
	char *jid=(char *)data[1];
	eb_jabber_stanza s;

	JLOG("eb_jabber_query_unsubscribe!\n");

	free(data);

	memset(&s, 0, sizeof(s));

	eb_jabber_stanza_append(&s, "/presence/#", "presence");
	eb_jabber_stanza_append(&s, "/presence/@to", jid);
	eb_jabber_stanza_append(&s, "/presence/@type", "unsubscribed");
	eb_jabber_stanza_append(&s, "/presence/!", "presence");
	eb_jabber_send_stanza(GET_LAD(ela)->fd, &s);
	eb_jabber_stanza_empty(&s);

	free(jid);
}

void eb_jabber_query_subscribe(int response, void *indata)
{
	void **data = (void **)indata;
	eb_local_account *ela = (eb_local_account *)data[0];
	char *jid=(char *)data[1];
	eb_jabber_stanza s;

	JLOG("eb_jabber_query_subscribe!\n");

	free(data);

	memset(&s, 0, sizeof(s));

	eb_jabber_stanza_append(&s, "/presence/#", "presence");
	eb_jabber_stanza_append(&s, "/presence/@to", jid);
	eb_jabber_stanza_append(&s, "/presence/@type", "subscribed");
	eb_jabber_stanza_append(&s, "/presence/!", "presence");
	eb_jabber_send_stanza(GET_LAD(ela)->fd, &s);
	eb_jabber_stanza_empty(&s);

	free(jid);
}

/* ---------------------
    Stanza manipulation
   --------------------- */

void eb_jabber_stanza_append(eb_jabber_stanza *stanza, const char *path, const char *value)
{
	eb_jabber_listnode *new = (eb_jabber_listnode *)malloc(sizeof(eb_jabber_listnode));
	new->path = strdup(path);
	new->value = strdup(value);
	new->next = 0;

	if (stanza->tail)
	{
		stanza->tail->next = new;
		stanza->tail = new;
	}
	else
	{
		stanza->list = stanza->tail = new;
	}
}

void eb_jabber_stanza_empty(eb_jabber_stanza *stanza)
{
	eb_jabber_listnode *node = stanza->list;
	eb_jabber_listnode *next;

	while (node) {
		next = node->next;
		free(node->path);
		free(node->value);
		free(node);
		node = next;
	}

	stanza->list = stanza->tail = 0;
}

char *eb_jabber_stanza_find(eb_jabber_stanza *stanza, const char *path)
{
	eb_jabber_listnode *node = stanza->list;

	while (node)
	{
		if (!strcmp(node->path, path))
			return node->value;
		node = node->next;
	}

	return 0;
}

char *eb_jabber_stanza_find_index(eb_jabber_stanza *stanza, const char *path, int index)
{
	eb_jabber_listnode *node = stanza->list;
	int item = 0;

	while (node)
	{
		if (!strcmp(node->path, path))
		{
			if (item++ == index) return node->value;
		}
		node = node->next;
	}

	return 0;
}

void eb_jabber_stanza_dump(eb_jabber_stanza *stanza)
{
#if DUMP_JABBER_XML
	eb_jabber_listnode *node = stanza->list;

	JLOG("PATH\tVALUE\n");
	JLOG("----\t-----\n");

	while (node)
	{
		JLOG("%s\t%s\n", node->path, node->value);
		node = node->next;
	}
	
	JLOG("\n");
#endif
}

void eb_jabber_send_stanza(int fd, eb_jabber_stanza *s)
{
	eb_jabber_listnode *node = s->list;
	int tagOpen = 0;
	char *i;

	JLOG("<< ");

	while (node)
	{
		if ((i = index(node->path, '#')))
		{
			if (tagOpen)
			{
				ewrite(fd, ">", 1);
				JLOG(">");
				tagOpen = 0;
			}
			ewrite(fd, "<", 1);
			JLOG("<");
			tagOpen = 1;
			ewrite(fd, node->value, strlen(node->value));
			JLOG("%s", node->value);
		}
		else if ((i = index(node->path, '@')))
		{
			char *tmp = eb_jabber_xml_escape(node->value);
			if (tmp) eb_jabber_write_parameter(fd, i+1, tmp);
			JLOG(" %s='%s'", i+1, tmp);
		}
		else if ((i = index(node->path, '$')))
		{
			char *tmp = eb_jabber_xml_escape(node->value);

			if (tagOpen)
			{
				ewrite(fd, ">", 1);
				JLOG(">");
				tagOpen = 0;
			}

			if (tmp)
			{
				ewrite(fd, tmp, strlen(tmp));
				JLOG("%s", tmp);
				free(tmp);
			}
		}
		else if ((i = index(node->path, '!')))
		{
			if (tagOpen)
			{
				ewrite(fd, ">", 1);
				JLOG(">");
				tagOpen = 0;
			}
			ewrite(fd, "</", 2);
			ewrite(fd, node->value, strlen(node->value));
			ewrite(fd, ">", 1);
			JLOG("</%s>", node->value);
		}
		node = node->next;
	}

	JLOG("\n");
}
